/*

    mmap functions for Windows

    this functionality only for Apache 1.3.x

    bero@bekkoame.ne.jp

*/

#include <windows.h>
#include <io.h>
#include <errno.h>
#include <stdlib.h> /* for malloc */
#include <stdio.h>
#include <mman.h>

#define HIDWORD(off)    (0)
#define LODWORD(off)    (off)
#define set_errno(no)   errno = no


/*
    address -> memory mapping object 
*/

struct mmap_list_t {
    HANDLE hMap;
    caddr_t addr;
    struct mmap_list_t *next;
};

static struct mmap_list_t *top = NULL;
static HANDLE mmap_list_mutex = NULL;

static int mmap_add(HANDLE hMap,caddr_t addr)
{
    struct mmap_list_t *cur;

    /* first */
    if (mmap_list_mutex==NULL) {
        /* this mutex is automaticaly closed by OS when process end */
        mmap_list_mutex = CreateMutex(NULL,FALSE,NULL);
        if (mmap_list_mutex == NULL) return -1;
    }

    cur = malloc(sizeof(struct mmap_list_t));

    WaitForSingleObject(mmap_list_mutex,INFINITE);

    cur->hMap = hMap;
    cur->addr = addr;
    cur->next = top;
    top = cur;

    ReleaseMutex(mmap_list_mutex);

    return 0;
}

static HANDLE mmap_search_delete(caddr_t addr)
{
    struct mmap_list_t *cur,*prev;
    HANDLE ret;

    WaitForSingleObject(mmap_list_mutex,INFINITE);

    for(prev=NULL,cur = top;cur && cur->addr!=addr;prev=cur,cur=cur->next) ;

    if (cur==NULL) {
        ret = NULL;
    } else {
        ret = cur->hMap;
        if (prev) {
            prev->next = cur->next;
        } else {
            top = cur->next;
        }
        free(cur);
    }

    ReleaseMutex(mmap_list_mutex);

    return ret;
}


/*
    main
*/

caddr_t mmap (caddr_t addr, size_t len, int prot, int flags, int fd, long off)
{
    HANDLE hFile;
    HANDLE hMap;
    DWORD access,protect;
    size_t maxsize = len; /* 0 means filesize */
    caddr_t ret;

    static DWORD granularity=0,pagesize=0;

    /* get pagesize */
    if (!granularity) {
        SYSTEM_INFO si;
        GetSystemInfo (&si);
        granularity = si.dwAllocationGranularity;
        pagesize = si.dwPageSize;
    }

    /* check param */
    if (off % pagesize
    || (!(flags & MAP_SHARED) && !(flags & MAP_PRIVATE))
    || ((flags & MAP_SHARED) && (flags & MAP_PRIVATE))
    || ((flags & MAP_FIXED) && ((DWORD)addr % granularity))
    || len==0
    || prot&PROT_EXEC)
    {
        set_errno (EINVAL);
        return MAP_FAILED;
    }

    /* convert mmap param to windows param */
/*
    if (prot&(PROT_READ|PROT_WRITE)==PROT_READ) {
        access = FILE_MAP_READ;
        protect = PAGE_READONLY;
    } else if (prot&(PROT_READ|PROT_WRITE)==(PROT_READ|PROT_WRITE)) {
        if (flags&MAP_PRIVATE) {
            access = FILE_MAP_COPY;
            protect = PAGE_WRITECOPY;
        } else {
            access = FILE_MAP_WRITE;
            protect = PAGE_READWRITE;
        }
    }
*/

    if (flags & MAP_ANONYMOUS) fd = -1;

    if (fd==-1) {
        hFile = INVALID_HANDLE_VALUE;
    } else {
        hFile = (HANDLE)_get_osfhandle(fd);
    }
    
    access = FILE_MAP_READ;
    protect = PAGE_READONLY;

    hMap = CreateFileMapping(hFile,NULL,protect,HIDWORD(maxsize),
			     LODWORD(maxsize),NULL /* no name*/);

    if (hMap!=NULL) {
        ret = (caddr_t)MapViewOfFileEx(hMap,access,HIDWORD(off),
				       LODWORD(off),len,addr);
        if (ret!=NULL) {
            if (!(flags&MAP_FIXED) || ret==addr) {
                if (mmap_add(hMap,ret)==0)
                    return ret; /* success */
            }
            UnmapViewOfFile(ret);
        }
        CloseHandle(hMap);
    }
    return MAP_FAILED;

}

int munmap (caddr_t addr, size_t len)
{
    HANDLE hMap;

    if (UnmapViewOfFile(addr)!=0) {
        hMap = mmap_search_delete(addr);
        if (hMap!=NULL) {;
            if (CloseHandle(hMap)!=0) {
                return 0;   /* success */
            }
        }
    }
    return -1;
}

#if 0
int mprotect (caddr_t addr, size_t len, int prot)
{
    DWORD old_prot,new_prot;

    switch (prot) {
    case PROT_NONE:
        new_prot = PAGE_NOACCESS;
        break;
    case PROT_READ | PROT_WRITE | PROT_EXEC:
        new_prot = PAGE_EXECUTE_READWRITE;
        break;
    case PROT_READ | PROT_WRITE:
        new_prot = PAGE_READWRITE;
        break;
    case PROT_READ | PROT_EXEC:
        new_prot = PAGE_EXECUTE_READ;
        break;
    case PROT_READ:
        new_prot = PAGE_READONLY;
        break;
    case PROT_EXEC:
        new_prot = PAGE_EXECUTE;
        break;
    default:
        set_errno (EINVAL);
        return -1;
    }

    if (VirtualProtect(addr,len,new_prot,&old_prot)==0) {
        set_errno(EACCES);
        return -1;
    }
    return 0;
}


int msync (caddr_t addr, size_t len, int flags)
{
    if (FlushViewOfFile(addr,len)==0) {
        set_errno(EINVAL);
        return -1;
    }
    return 0;
}
#endif
