# -*- coding: utf-8 -*-
import os
import xbmc
import xbmcaddon
import xbmcvfs

ADDON = xbmcaddon.Addon(id='script.audiobooks')
ADDON_ID = ADDON.getAddonInfo('id')
ICON = ADDON.getAddonInfo('icon')


# Common logging module
def log(txt, loglevel=xbmc.LOGDEBUG):
    if (ADDON.getSetting("logEnabled") == "true") or (loglevel != xbmc.LOGDEBUG):
        if isinstance(txt, str):
            txt = txt.decode("utf-8")
        message = u'%s: %s' % (ADDON_ID, txt)
        xbmc.log(msg=message.encode("utf-8"), level=loglevel)


def os_path_join(dir, file):
    # Check if it ends in a slash
    if dir.endswith("/") or dir.endswith("\\"):
        # Remove the slash character
        dir = dir[:-1]

    # Convert each argument - if an error, then it will use the default value
    # that was passed in
    try:
        dir = dir.decode("utf-8")
    except:
        pass
    try:
        file = file.decode("utf-8")
    except:
        pass
    return os.path.join(dir, file)


# Splits a path the same way as os.path.split but supports paths of a different
# OS than that being run on
def os_path_split(fullpath):
    # Check if it ends in a slash
    if fullpath.endswith("/") or fullpath.endswith("\\"):
        # Remove the slash character
        fullpath = fullpath[:-1]

    try:
        slash1 = fullpath.rindex("/")
    except:
        slash1 = -1

    try:
        slash2 = fullpath.rindex("\\")
    except:
        slash2 = -1

    # Parse based on the last type of slash in the string
    if slash1 > slash2:
        return fullpath.rsplit("/", 1)

    return fullpath.rsplit("\\", 1)


# Checks if a directory exists (Do not use for files)
def dir_exists(dirpath):
    # There is an issue with password protected smb shares, in that they seem to
    # always return false for a directory exists call, so if we have a smb with
    # a password and user name, then we return true
    if '@' in dirpath:
        return True

    directoryPath = dirpath
    # The xbmcvfs exists interface require that directories end in a slash
    # It used to be OK not to have the slash in Gotham, but it is now required
    if (not directoryPath.endswith("/")) and (not directoryPath.endswith("\\")):
        dirSep = "/"
        if "\\" in directoryPath:
            dirSep = "\\"
        directoryPath = "%s%s" % (directoryPath, dirSep)
    return xbmcvfs.exists(directoryPath)


##############################
# Stores Various Settings
##############################
class Settings():
    FFMPEG_NONE = 0
    FFMPEG_LIB = 1
    FFMPEG_EXEC = 2

    @staticmethod
    def getAudioBookFolder():
        return ADDON.getSetting("audioBooksFolder")

    @staticmethod
    def setAudioBookFolder(audioBooksFolder):
        ADDON.setSetting("audioBooksFolder", audioBooksFolder)

    @staticmethod
    def isFFmpegAutoDetect():
        return ADDON.getSetting("ffmpegDetectOnStartup") == 'true'

    @staticmethod
    def clearFFmpegAutoDetect():
        return ADDON.setSetting("ffmpegDetectOnStartup", 'false')

    @staticmethod
    def getFFmpegSetting():
        index = int(ADDON.getSetting("ffmpegSetting"))
        if index == 0:
            return Settings.FFMPEG_NONE
        elif index == 1:
            return Settings.FFMPEG_LIB
        elif index == 2:
            return Settings.FFMPEG_EXEC

        return Settings.FFMPEG_NONE

    @staticmethod
    def setFFmpegSetting(newValue):
        settingsValue = "0"
        if newValue == Settings.FFMPEG_LIB:
            settingsValue = "1"
        elif newValue == Settings.FFMPEG_EXEC:
            settingsValue = "2"

        ADDON.setSetting("ffmpegSetting", settingsValue)

    @staticmethod
    def getFFmpegExecLocation():
        if Settings.getFFmpegSetting() != Settings.FFMPEG_EXEC:
            return None

        location = ADDON.getSetting("ffmpegLocation")
        # Check to see if this location actually exists
        if location not in [None, ""]:
            if not xbmcvfs.exists(location):
                log("FFmpeg does not exist in location %s" % location)
                location = None
            # If the ffmpeg file is in a directory with spaces, need to escape it
            # by adding quotes
            if ' ' in location:
                location = '"%s"' % location
        return location

    @staticmethod
    def setFFmpegExecLocation(execPath):
        ADDON.setSetting("ffmpegLocation", execPath)

    @staticmethod
    def getFFmpegLibraryLocation():
        if Settings.getFFmpegSetting() != Settings.FFMPEG_LIB:
            return None

        return ADDON.getSetting("ffmpegLibraryLocation")

    @staticmethod
    def setFFmpegLibraryLocation(libPath):
        ADDON.setSetting("ffmpegLibraryLocation", libPath)

    @staticmethod
    def isMarkCompletedItems():
        return ADDON.getSetting("markCompletedItems") == 'true'

    @staticmethod
    def autoNumberChapters():
        return ADDON.getSetting("autoNumberChapters") == 'true'

    @staticmethod
    def isShowArtistInBookList():
        return ADDON.getSetting("showArtistInBookList") == 'true'

    @staticmethod
    def isShowPlayButtonIfOneChapter():
        return ADDON.getSetting("showPlayButtonIfOneChapter") == 'true'

    @staticmethod
    def getFallbackCoverImage():
        fallbackCover = ADDON.getSetting("fallbackCoverImage")
        if fallbackCover in [None, ""]:
            fallbackCover = ICON
        return fallbackCover

    @staticmethod
    def getCoverCacheLocation():
        coverCache = xbmc.translatePath('special://profile/addon_data/%s/covers' % ADDON_ID).decode("utf-8")

        # Make sure the directory to cache the covers exists
        if not dir_exists(xbmc.translatePath('special://profile/addon_data/%s' % ADDON_ID).decode("utf-8")):
            xbmcvfs.mkdir(xbmc.translatePath('special://profile/addon_data/%s' % ADDON_ID).decode("utf-8"))
        if not dir_exists(coverCache):
            xbmcvfs.mkdir(coverCache)
        return coverCache

    @staticmethod
    def getTempLocation():
        tmpdestination = xbmc.translatePath('special://profile/addon_data/%s/temp' % ADDON_ID).decode("utf-8")

        # Make sure the directory to cache the covers exists
        if not dir_exists(xbmc.translatePath('special://profile/addon_data/%s' % ADDON_ID).decode("utf-8")):
            xbmcvfs.mkdir(xbmc.translatePath('special://profile/addon_data/%s' % ADDON_ID).decode("utf-8"))
        if not dir_exists(tmpdestination):
            xbmcvfs.mkdir(tmpdestination)
        return tmpdestination

    @staticmethod
    def isPlainAudioFile(filename):
        audioFileType = False
        aFile = filename.lower()
        if aFile.endswith('.mp3') or aFile.endswith('.flac') or aFile.endswith('.wav') or aFile.endswith('.m4a') or aFile.endswith('.wma'):
            audioFileType = True
        return audioFileType
