from stream.utils import get_api_language

BASE_URL = "http://www.thetvdb.com"
HEADERS = {
    "Referer": BASE_URL,
}
API_URL = "%s/api" % BASE_URL
API_KEY = "1D62F2F90030C444"
LANG = get_api_language()

def dom2dict(node):
    ret = {}
    for child in node:
        if len(child):
            ret.setdefault(child.tag.lower(), []).append(dom2dict(child))
        else:
            ret[child.tag.lower()] = child.text
    return ret

def split_keys(meta, *keys):
    for key in keys:
        if meta.get(key):
            meta[key] = filter(None, meta[key].split("|"))
    return meta


def image_url(fragment):
    return "%s/banners/%s" % (BASE_URL, fragment)


def banner(show_id):
    from urlparse import urljoin
    return urljoin(BASE_URL, "banners/graphical/%s" % show_id)


def season_url(show_id, season):
    return "%s/banners/seasons/%s-%d-2.jpg" % (BASE_URL, show_id, season)


def update_image_urls(meta):
    if isinstance(meta, dict):
        for k, v in meta.items():
            if isinstance(v, list):
                map(update_image_urls, v)
            elif isinstance(v, dict):
                update_image_urls(v)
            elif k in ["banner", "fanart", "poster", "filename", "bannerpath", "vignettepath", "thumbnailpath"] and isinstance(v, basestring):
                meta[k] = image_url(v)
    return meta


def show_url(show_id):
    return "%s/%s/series/%s/%s.xml" % (API_URL, API_KEY, show_id, LANG)


def show_base_url(show_id):
    return "%s/%s/series/%s" % (API_URL, API_KEY, show_id)


def show_banners_url(show_id):
    return "%s/banners.xml" % (API_URL, API_KEY, show_id)


def get(show_id):
    from stream.caching import shelf
    with shelf("com.thetvdb.show.%s" % show_id) as show:
        if not show:
            import xml.etree.ElementTree as ET
            from stream.utils import url_get

            dom = ET.fromstring(url_get(show_url(show_id), headers=HEADERS, with_immunicity=False))
            if not len(dom):
                return
            meta = dom2dict(dom[0])
            meta = split_keys(meta, "actors", "genre", "writer")
            update_image_urls(meta)
            show.update(meta)
        return dict(show)


def search(name, complete=False):
    from stream.caching import shelf
    import hashlib
    search_hash = hashlib.sha1(name).hexdigest()
    with shelf("com.thetvdb.search.%s" % search_hash) as show:
        if not show:
            import re
            import xml.etree.ElementTree as ET
            from stream.utils import url_get
            dom = ET.fromstring(url_get("%s/api/GetSeries.php" % BASE_URL, params={
                "seriesname": name,
            }, headers=HEADERS, with_immunicity=False))
            if not len(dom):
                return
            meta = dom2dict(dom[0])
            if not complete:
                return update_image_urls(meta)
            show.update(get(meta["id"]))
        return show


def get_banners(show_id):
    import xml.etree.ElementTree as ET
    from stream.utils import url_get

    r = url_get("%s/banners.xml" % show_base_url(show_id), headers=HEADERS, with_immunicity=False)
    dom = ET.fromstring(r)
    if not len(dom):
        return
    return update_image_urls(dom2dict(dom))["banner"]


def get_all_meta(show_id):
    import xml.etree.ElementTree as ET
    from concurrent import futures
    from stream.utils import url_get, joining

    def _get_all_meta():
        r = url_get("%s/all/%s.xml" % (show_base_url(show_id), LANG), headers=HEADERS, with_immunicity=False)
        dom = ET.fromstring(r)
        if not len(dom):
            return
        return update_image_urls(dom2dict(dom))
    with futures.ThreadPoolExecutor(max_workers=2) as pool:
        meta = pool.submit(_get_all_meta)
        banners = pool.submit(get_banners, show_id)
    meta = meta.result()
    meta["series"][0]["episodes"] = meta["episode"]
    meta = meta["series"][0]
    meta["banners"] = banners.result() or []
    return meta


def get_list_item(meta):
    m = lambda x: meta.get(x) or ""
    m_list = lambda x: meta.get(x) and ", ".join(meta[x]) or ""
    return {
        "label": meta["seriesname"],
        "icon": m("poster"),
        "thumbnail": m("poster"),
        "info": {
            "count": meta["id"],
            "title": meta["seriesname"],
            "genre": m_list("genre"),
            "plot": m("overview"),
            "plot_outline": m("overview"),
            "tagline": m("overview"),
            "rating": m("rating"),
            "code": m("imdb_id"),
            "mpaa": m("contentrating"),
            "cast": m("actors") or [],
            "castandrole": m("actors") or [],
            "tvshowtitle": meta["seriesname"],
            "studio": m("network"),
            "status": m("status"),
            "premiered": m("firstaired"),
            "duration": m("runtime"),
            "picturepath": m("poster"),
            "year": meta.get("firstaired") and meta["firstaired"].split("-")[0] or "",
            "votes": "%s votes" % meta["ratingcount"],
        },
        "properties": {
            "fanart_image": m("fanart"),
        },
    }


def get_season_list_item(meta, season):
    m = lambda x: meta.get(x) or ""
    m_list = lambda x: meta.get(x) and ", ".join(meta[x]) or ""
    season_id = filter(lambda ep: int(ep["seasonnumber"]) == season, meta["episodes"])[0]["seasonid"]
    item = {
        "label": "Season %d" % season,
        "info": {
            "count": season_id,
            "tvshowtitle": meta["seriesname"],
            "season": season,
        },
        "properties": {
            "fanart_image": m("fanart"),
        },
    }
    season_banners = [banner for banner in meta["banners"] if banner["bannertype"] == "season" and int(banner["season"]) == season]
    if season_banners:
        item["icon"] = item["thumbnail"] = season_banners[0]["bannerpath"]
    return item


def build_episode_list_items(show_meta, season):
    episodes = [episode for episode in show_meta["episodes"] if int(episode["seasonnumber"]) == season]
    episodes = sorted(episodes, key=lambda ep: int(ep["episodenumber"]))
    for episode in episodes:
        m = lambda x: episode.get(x) or ""
        yield {
            "label": m("episodename"),
            "icon": m("filename"),
            "thumbnail": m("filename"),
            "info": {
                "count": m("id"),
                "season": season,
                "episode": m("episodenumber"),
                "title": m("episodename"),
                "originaltitle": m("episodename"),
                "plot": m("overview"),
                "plot_outline": m("overview"),
                "tagline": m("overview"),
                "rating": float(m("rating") or 0),
                "code": m("imdb_id"),
                "premiered": m("firstaired"),
                "cast": episode.get("gueststars") and filter(None, episode["gueststars"].split("|")) or [],
                "tvshowtitle": show_meta.get("seriesname") or "",
                "writer": episode.get("writer") and ", ".join(filter(None, episode["writer"].split("|"))) or "",
            },
            "properties": {
                "fanart_image": show_meta.get("fanart") or "",
            },
        }
