package org.ovirt.engine.core.tools;

import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This class is intended to handle the regular input and output generated by
 * the tools and destined usually to human users. Output will be sent to the
 * system console and also to the log, input will be taken from the system
 * console and sent to the log as well, so that the log keeps a record of all
 * what happened in the console.
 */
public class ToolConsole {
    // The log:
    private static final Logger log = LoggerFactory.getLogger(ToolConsole.class);

    // This is a singleton and this is the instance:
    private static final ToolConsole instance = new ToolConsole();

    public static ToolConsole getInstance() {
        return instance;
    }

    private ToolConsole() {
        // Nothing, just prevent creation of new instances.
    }

    /**
     * Write the result of converting an object to a string to the standard
     * output and to the log.
     *
     * @param what the object to convert to string and write
     */
    public void write(Object what) {
        System.out.print(what);
        System.out.flush();
        log.info("Written to standard output \"{}\".", what);
    }

    /**
     * Write and end of line to the standard system output and to the log.
     */
    public void writeLine() {
        write("\n");
    }

    /**
     * Write the result of converting an object to a string to the standard
     * output and to the log.
     *
     * @param what the object to convert to string and write
     */
    public void writeLine(Object what) {
        write(what + "\n");
    }

    /**
     * Formats a set of objects with the given format string (the same used
     * with <code>System.out.printf()</code> and write the result to the
     * standard output and to the log.
     *
     * @param format the format specification
     * @param args the objects to convert to strings according to the given
     *   format
     */
    public void writeFormat(String format, Object... args) {
        String text = String.format(format, args);
        write(text);
    }

    /**
     * Write the password to the standard output, but not to the log. Note that
     * the parameter is an array of characters in order to make it possible
     * to clean it after calling this method.
     *
     * @param password the characters of the password
     */
    public void writePassword(char[] password) {
        System.out.print(password);
        System.out.flush();
        log.info("Written password to standard output.");
    }

    /**
     * Write the result of converting an object to a string to the standard
     * error stream and to the log.
     *
     * @param what the object to convert to string and write
     */
    public void writeError(Object what) {
        System.err.print(what);
        System.err.flush();
        log.info("Written to standard error \"{}\".", what);
    }

    /**
     * Write and end of line to the standard error stream.
     */
    public void writeErrorLine() {
        writeError("\n");
    }

    /**
     * Write the result of converting an object to a string to the standard
     * error stream followed by an end of line.
     *
     * @param what the object to convert to string and write
     */
    public void writeErrorLine(Object what) {
        writeError(what + "\n");
    }

    /**
     * Formats a set of objects with the given format string (the same used
     * with <code>System.out.printf()</code> and write the result to the
     * standard error stream.
     *
     * @param format the format specification
     * @param args the objects to convert to strings according to the given
     *   format
     */
    public void writeErrorFormat(String format, Object... args) {
        String text = String.format(format, args);
        writeError(text);
    }

    /**
     * Read a line from the standard input.
     */
    public String readLine() {
        StringBuilder buffer = new StringBuilder();
        for (;;) {
            int character;
            try {
                character = System.in.read();
            }
            catch (IOException exception) {
                log.error(
                    "Error while reading line from standard input. Will " +
                    "consider it the end of the line and continue.",
                    exception
                );
                break;
            }
            if (character == -1 || character == '\n') {
                break;
            }
            buffer.append((char) character);
        }
        String line = buffer.toString();
        log.info("Read from stdin \"{}\".", line);
        return line;
    }

    /**
     * Write a prompt to the console and then read a line without echoing it.
     */
    public String readPassword(String prompt) throws IOException {
        char[] password = System.console().readPassword(prompt);
        if (password == null) {
            throw new IOException("EOF");
        }
        log.info("Written to console \"{}\".", prompt);
        log.info("Read password from console.");
        return new String(password);
    }
}
