/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.AgentBuilder;
import org.ovirt.engine.sdk4.types.Agent;

/**
 * A service to manage fence agents for a specific host.
 */
public interface FenceAgentsService extends Service {
    /**
     * Add a new fencing-agent to the host.
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest agent(Agent agent);
        AddRequest agent(AgentBuilder agent);
    }
    
    /**
     * Add a new fencing-agent to the host.
     */
    public interface AddResponse extends Response {
        Agent agent();
    }
    
    /**
     * Add a new fencing-agent to the host.
     */
    AddRequest add();
    
    /**
     * Returns the list of fencing agents configured for the host.
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/hosts/123/fenceagents
     * ----
     * 
     * And here is sample response:
     * 
     * [source,xml]
     * ----
     * <agents>
     *   <agent id="0">
     *     <type>apc</type>
     *     <order>1</order>
     *     <ip>192.168.1.101</ip>
     *     <user>user</user>
     *     <password>xxx</password>
     *     <port>9</port>
     *     <options>name1=value1, name2=value2</options>
     *   </agent>
     * </agents>
     * ----
     * 
     * The order of the returned list of fencing agents isn't guaranteed.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of agents to return. If not specified all the agents are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of agents to return. If not specified all the agents are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of agents to return. If not specified all the agents are returned.
         */
        ListRequest max(BigInteger max);
    }
    
    /**
     * Returns the list of fencing agents configured for the host.
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/hosts/123/fenceagents
     * ----
     * 
     * And here is sample response:
     * 
     * [source,xml]
     * ----
     * <agents>
     *   <agent id="0">
     *     <type>apc</type>
     *     <order>1</order>
     *     <ip>192.168.1.101</ip>
     *     <user>user</user>
     *     <password>xxx</password>
     *     <port>9</port>
     *     <options>name1=value1, name2=value2</options>
     *   </agent>
     * </agents>
     * ----
     * 
     * The order of the returned list of fencing agents isn't guaranteed.
     */
    public interface ListResponse extends Response {
        /**
         * List of fence agent details.
         */
        List<Agent> agents();
    }
    
    /**
     * Returns the list of fencing agents configured for the host.
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/hosts/123/fenceagents
     * ----
     * 
     * And here is sample response:
     * 
     * [source,xml]
     * ----
     * <agents>
     *   <agent id="0">
     *     <type>apc</type>
     *     <order>1</order>
     *     <ip>192.168.1.101</ip>
     *     <user>user</user>
     *     <password>xxx</password>
     *     <port>9</port>
     *     <options>name1=value1, name2=value2</options>
     *   </agent>
     * </agents>
     * ----
     * 
     * The order of the returned list of fencing agents isn't guaranteed.
     */
    ListRequest list();
    
    /**
     * Reference to service that manages a specific fence agent
     * for this host.
     */
    FenceAgentService agentService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

