/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.BookmarkBuilder;
import org.ovirt.engine.sdk4.types.Bookmark;

/**
 * A service to manage a bookmark.
 */
public interface BookmarkService extends Service {
    /**
     * Get a bookmark.
     * 
     * An example for getting a bookmark:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/bookmarks/123
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
     *   <name>example_vm</name>
     *   <value>vm: name=example*</value>
     * </bookmark>
     * ----
     */
    public interface GetRequest extends Request<GetRequest, GetResponse> {
    }
    
    /**
     * Get a bookmark.
     * 
     * An example for getting a bookmark:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/bookmarks/123
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
     *   <name>example_vm</name>
     *   <value>vm: name=example*</value>
     * </bookmark>
     * ----
     */
    public interface GetResponse extends Response {
        /**
         * The requested bookmark.
         */
        Bookmark bookmark();
    }
    
    /**
     * Get a bookmark.
     * 
     * An example for getting a bookmark:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/bookmarks/123
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
     *   <name>example_vm</name>
     *   <value>vm: name=example*</value>
     * </bookmark>
     * ----
     */
    GetRequest get();
    
    /**
     * Remove a bookmark.
     * 
     * An example for removing a bookmark:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/bookmarks/123
     * ----
     */
    public interface RemoveRequest extends Request<RemoveRequest, RemoveResponse> {
        /**
         * Indicates if the remove should be performed asynchronously.
         */
        RemoveRequest async(Boolean async);
    }
    
    /**
     * Remove a bookmark.
     * 
     * An example for removing a bookmark:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/bookmarks/123
     * ----
     */
    public interface RemoveResponse extends Response {
    }
    
    /**
     * Remove a bookmark.
     * 
     * An example for removing a bookmark:
     * 
     * [source]
     * ----
     * DELETE /ovirt-engine/api/bookmarks/123
     * ----
     */
    RemoveRequest remove();
    
    /**
     * Update a bookmark.
     * 
     * An example for updating a bookmark:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/bookmarks/123
     * ----
     * 
     * With the request body:
     * 
     * [source,xml]
     * ----
     * <bookmark>
     *   <name>new_example_vm</name>
     *   <value>vm: name=new_example*</value>
     * </bookmark>
     * ----
     */
    public interface UpdateRequest extends Request<UpdateRequest, UpdateResponse> {
        /**
         * Indicates if the update should be performed asynchronously.
         */
        UpdateRequest async(Boolean async);
        UpdateRequest bookmark(Bookmark bookmark);
        /**
         * The updated bookmark.
         */
        UpdateRequest bookmark(BookmarkBuilder bookmark);
    }
    
    /**
     * Update a bookmark.
     * 
     * An example for updating a bookmark:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/bookmarks/123
     * ----
     * 
     * With the request body:
     * 
     * [source,xml]
     * ----
     * <bookmark>
     *   <name>new_example_vm</name>
     *   <value>vm: name=new_example*</value>
     * </bookmark>
     * ----
     */
    public interface UpdateResponse extends Response {
        /**
         * The updated bookmark.
         */
        Bookmark bookmark();
    }
    
    /**
     * Update a bookmark.
     * 
     * An example for updating a bookmark:
     * 
     * [source]
     * ----
     * PUT /ovirt-engine/api/bookmarks/123
     * ----
     * 
     * With the request body:
     * 
     * [source,xml]
     * ----
     * <bookmark>
     *   <name>new_example_vm</name>
     *   <value>vm: name=new_example*</value>
     * </bookmark>
     * ----
     */
    UpdateRequest update();
    
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

