/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.services;

import java.io.IOException;
import java.math.BigInteger;
import java.util.List;
import org.ovirt.engine.sdk4.Request;
import org.ovirt.engine.sdk4.Response;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.builders.BookmarkBuilder;
import org.ovirt.engine.sdk4.types.Bookmark;

/**
 * A service to manage bookmarks.
 */
public interface BookmarksService extends Service {
    /**
     * Adding a new bookmark.
     * 
     * Example of adding a bookmark:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/bookmarks
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmark>
     *   <name>new_example_vm</name>
     *   <value>vm: name=new_example*</value>
     * </bookmark>
     * ----
     */
    public interface AddRequest extends Request<AddRequest, AddResponse> {
        AddRequest bookmark(Bookmark bookmark);
        /**
         * The added bookmark.
         */
        AddRequest bookmark(BookmarkBuilder bookmark);
    }
    
    /**
     * Adding a new bookmark.
     * 
     * Example of adding a bookmark:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/bookmarks
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmark>
     *   <name>new_example_vm</name>
     *   <value>vm: name=new_example*</value>
     * </bookmark>
     * ----
     */
    public interface AddResponse extends Response {
        /**
         * The added bookmark.
         */
        Bookmark bookmark();
    }
    
    /**
     * Adding a new bookmark.
     * 
     * Example of adding a bookmark:
     * 
     * [source]
     * ----
     * POST /ovirt-engine/api/bookmarks
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmark>
     *   <name>new_example_vm</name>
     *   <value>vm: name=new_example*</value>
     * </bookmark>
     * ----
     */
    AddRequest add();
    
    /**
     * Listing all the available bookmarks.
     * 
     * Example of listing bookmarks:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/bookmarks
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmarks>
     *   <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
     *     <name>database</name>
     *     <value>vm: name=database*</value>
     *   </bookmark>
     *   <bookmark href="/ovirt-engine/api/bookmarks/456" id="456">
     *     <name>example</name>
     *     <value>vm: name=example*</value>
     *   </bookmark>
     * </bookmarks>
     * ----
     * 
     * The order of the returned bookmarks isn't guaranteed.
     */
    public interface ListRequest extends Request<ListRequest, ListResponse> {
        /**
         * Sets the maximum number of bookmarks to return. If not specified all the bookmarks are returned.
         */
        ListRequest max(Integer max);
        /**
         * Sets the maximum number of bookmarks to return. If not specified all the bookmarks are returned.
         */
        ListRequest max(Long max);
        /**
         * Sets the maximum number of bookmarks to return. If not specified all the bookmarks are returned.
         */
        ListRequest max(BigInteger max);
    }
    
    /**
     * Listing all the available bookmarks.
     * 
     * Example of listing bookmarks:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/bookmarks
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmarks>
     *   <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
     *     <name>database</name>
     *     <value>vm: name=database*</value>
     *   </bookmark>
     *   <bookmark href="/ovirt-engine/api/bookmarks/456" id="456">
     *     <name>example</name>
     *     <value>vm: name=example*</value>
     *   </bookmark>
     * </bookmarks>
     * ----
     * 
     * The order of the returned bookmarks isn't guaranteed.
     */
    public interface ListResponse extends Response {
        /**
         * The list of available bookmarks.
         */
        List<Bookmark> bookmarks();
    }
    
    /**
     * Listing all the available bookmarks.
     * 
     * Example of listing bookmarks:
     * 
     * [source]
     * ----
     * GET /ovirt-engine/api/bookmarks
     * ----
     * 
     * [source,xml]
     * ----
     * <bookmarks>
     *   <bookmark href="/ovirt-engine/api/bookmarks/123" id="123">
     *     <name>database</name>
     *     <value>vm: name=database*</value>
     *   </bookmark>
     *   <bookmark href="/ovirt-engine/api/bookmarks/456" id="456">
     *     <name>example</name>
     *     <value>vm: name=example*</value>
     *   </bookmark>
     * </bookmarks>
     * ----
     * 
     * The order of the returned bookmarks isn't guaranteed.
     */
    ListRequest list();
    
    /**
     * A reference to the service managing a specific bookmark.
     */
    BookmarkService bookmarkService(String id);
    /**
     * Service locator method, returns individual service on which the URI is dispatched.
     */
    Service service(String path);
}

