/*
Copyright (c) 2015 Red Hat, Inc.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

  http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package org.ovirt.engine.sdk4.internal.services;

import java.io.IOException;
import java.net.URISyntaxException;
import org.apache.http.HttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.util.EntityUtils;
import org.ovirt.api.metamodel.runtime.xml.XmlReader;
import org.ovirt.engine.sdk4.Error;
import org.ovirt.engine.sdk4.Service;
import org.ovirt.engine.sdk4.internal.HttpConnection;
import org.ovirt.engine.sdk4.internal.xml.XmlDomainReader;
import org.ovirt.engine.sdk4.services.DomainGroupsService;
import org.ovirt.engine.sdk4.services.DomainService;
import org.ovirt.engine.sdk4.services.DomainUsersService;
import org.ovirt.engine.sdk4.types.Domain;

public class DomainServiceImpl extends ServiceImpl implements DomainService {
    
    public DomainServiceImpl(HttpConnection connection, String path) {
        super(connection, path);
    }
    
    private class GetRequestImpl implements GetRequest {
        public GetResponse send() {
            HttpGet request = null;
            try {
                URIBuilder uriBuilder = new URIBuilder(getConnection().getUrl() + getPath());
                request = new HttpGet(uriBuilder.build());
            }
            catch (URISyntaxException ex) {
                throw new Error("Failed to build URL", ex);
            }
            
            HttpResponse response = getConnection().send(request);
            if (
              response.getStatusLine().getStatusCode() == 200
            ) {
                try (
                  XmlReader reader = new XmlReader(response.getEntity().getContent())
                ) {
                    return new GetResponseImpl(XmlDomainReader.readOne(reader));
                }
                catch (IOException ex) {
                    throw new Error("Failed to read response", ex);
                }
                finally {
                    EntityUtils.consumeQuietly(response.getEntity());
                }
            }
            else {
                checkFault(response);
                return new GetResponseImpl(null);
            }
        }
        
    }
    
    private class GetResponseImpl implements GetResponse {
        private Domain domain;
        
        public GetResponseImpl(Domain domain) {
            this.domain = domain;
        }
        
        public Domain domain() {
            return domain;
        }
        
    }
    
    public GetRequest get() {
        return new GetRequestImpl();
    }
    
    public DomainGroupsService groupsService() {
        return new DomainGroupsServiceImpl(getConnection(), getPath() + "/groups");
    }
    
    public DomainUsersService usersService() {
        return new DomainUsersServiceImpl(getConnection(), getPath() + "/users");
    }
    
    public Service service(String path) {
        if (path == null) {
            return this;
        }
        
        if (path.equals("groups")) {
            return groupsService();
        }
        if (path.startsWith("groups/")) {
            return groupsService().service(path.substring(7));
        }
        if (path.equals("users")) {
            return usersService();
        }
        if (path.startsWith("users/")) {
            return usersService().service(path.substring(6));
        }
        throw new Error("The path " + path + " doesn't correspond to any service");
    }
    
    public String toString() {
        return "DomainService:" + getPath();
    }
    
}

