--
-- This file is part of TALER
-- Copyright (C) 2024 Taler Systems SA
--
-- TALER is free software; you can redistribute it and/or modify it under the
-- terms of the GNU General Public License as published by the Free Software
-- Foundation; either version 3, or (at your option) any later version.
--
-- TALER is distributed in the hope that it will be useful, but WITHOUT ANY
-- WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
-- A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
--
-- You should have received a copy of the GNU General Public License along with
-- TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
--

CREATE FUNCTION create_table_legitimization_outcomes(
  IN partition_suffix TEXT DEFAULT NULL
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
BEGIN
  PERFORM create_partitioned_table(
    'CREATE TABLE %I'
      '(outcome_serial_id BIGINT GENERATED BY DEFAULT AS IDENTITY'
      ',h_payto BYTEA NOT NULL CHECK (LENGTH(h_payto)=32)'
      ',decision_time INT8 NOT NULL'
      ',expiration_time INT8 NOT NULL'
      ',jproperties TEXT'
      ',new_measure_name TEXT'
      ',to_investigate BOOL NOT NULL'
      ',is_active BOOL NOT NULL DEFAULT(TRUE)'
      ',jnew_rules TEXT'
    ') %s ;'
    ,'legitimization_outcomes'
    ,'PARTITION BY HASH (h_payto)'
    ,partition_suffix
  );
  PERFORM comment_partitioned_table(
     'Outcomes of legitimization processes by account'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'unique ID for this legitimization outcome at the exchange'
    ,'outcome_serial_id'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'hash of the payto://-URI this outcome is about; foreign key linking the entry to the wire_targets table, NOT a primary key (multiple outcomes are possible per account over time)'
    ,'h_payto'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'when was this outcome decided, rounded timestamp'
    ,'decision_time'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'space-separated list of names of measures to trigger immediately, NULL for none, prefixed with a "+" to indicate AND combination for the measures'
    ,'new_measure_name'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'time when the decision expires and the expiration jnew_rules should be applied'
    ,'expiration_time'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'JSON object of type AccountProperties, such as PEP status, business domain, risk assessment, etc.'
    ,'jproperties'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'AML staff should investigate the activity of this account'
    ,'to_investigate'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'TRUE if this is the current authoritative legitimization outcome'
    ,'is_active'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
  PERFORM comment_partitioned_column(
     'JSON object of type LegitimizationRuleSet with rules to apply to the various operation types for this account; all KYC checks should first check if active new rules for a given account exist in this table (and apply specified measures); if not, it should check the default rules to decide if a measure is required; NULL if the default rules apply'
    ,'jnew_rules'
    ,'legitimization_outcomes'
    ,partition_suffix
  );
END
$$;


CREATE FUNCTION constrain_table_legitimization_outcomes(
  IN partition_suffix TEXT
)
RETURNS VOID
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'legitimization_outcomes';
BEGIN
  table_name = concat_ws('_', table_name, partition_suffix);

  EXECUTE FORMAT (
    'CREATE INDEX ' || table_name || '_by_target_token'
    ' ON ' || table_name ||
    ' (h_payto)'
    ' WHERE is_active' ||
    ';'
  );
END
$$;


CREATE FUNCTION foreign_table_legitimization_outcomes()
RETURNS void
LANGUAGE plpgsql
AS $$
DECLARE
  table_name TEXT DEFAULT 'legitimization_outcomes';
BEGIN

  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_serial_id_key'
    ' UNIQUE (outcome_serial_id)');

  EXECUTE FORMAT (
    'ALTER TABLE ' || table_name ||
    ' ADD CONSTRAINT ' || table_name || '_foreign_key_h_payto'
    ' FOREIGN KEY (h_payto)'
    ' REFERENCES wire_targets (wire_target_h_payto) ON DELETE CASCADE');
END
$$;


INSERT INTO exchange_tables
    (name
    ,version
    ,action
    ,partitioned
    ,by_range)
  VALUES
    ('legitimization_outcomes'
    ,'exchange-0005'
    ,'create'
    ,TRUE
    ,FALSE),
    ('legitimization_outcomes'
    ,'exchange-0005'
    ,'constrain'
    ,TRUE
    ,FALSE),
    ('legitimization_outcomes'
    ,'exchange-0005'
    ,'foreign'
    ,TRUE
    ,FALSE);
