/* -*- c++ -*- */
/*
 * Copyright 2004 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef INCLUDED_USRP0_SINK_C_H
#define INCLUDED_USRP0_SINK_C_H

#include <gr_sync_block.h>
#include <stdexcept>

class usrp0_tx;


class usrp0_sink_c;
typedef boost::shared_ptr<usrp0_sink_c> usrp0_sink_c_sptr;


// public shared_ptr constructor

usrp0_sink_c_sptr
usrp0_make_sink_c (int which_board, unsigned int interp_rate) throw (std::runtime_error);


/*!
 * \brief interface to Universal Software Radio Peripheral Tx path (Rev 0)
 * \ingroup sink
 */
class usrp0_sink_c : public gr_sync_block {
 private:
  usrp0_tx		*d_usrp;
  int			 d_nunderruns;
  
  friend usrp0_sink_c_sptr
  usrp0_make_sink_c (int which_board, unsigned int interp_rate) throw (std::runtime_error);

 protected:
  usrp0_sink_c (int which_board, unsigned int interp_rate) throw (std::runtime_error);

 public:
  ~usrp0_sink_c ();

  int work (int noutput_items,
	    gr_vector_const_void_star &input_items,
	    gr_vector_void_star &output_items);

  /*!
   * \brief Set interpolator rate.  \p rate must be in [2, 256]
   *
   * The final complex sample rate across the USB is
   *   dac_freq () / interp_rate ()
   */
  bool set_interp_rate (unsigned int rate);

  /*!
   * \brief set the frequency of the digital up converter.
   *
   * \p channel must be 0.  \p freq is the center frequency in Hz.
   * It must be in the range [-FIXME, FIXME].  The frequency specified is
   * quantized.  Use tx_freq to retrieve the actual value used.
   */
  bool set_tx_freq (int channel, double freq);

  // ACCESSORS

  long dac_freq () const;
  unsigned int interp_rate () const;
  double tx_freq (int channel) const;
  int nunderruns () const { return d_nunderruns; }
};

#endif /* INCLUDED_USRP0_SINK_C_H */
