/* -*- Mode: c++ -*- 
 *
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */


#ifndef _VrCORR_H_
#define _VrCORR_H_

#define SIZE 4096

#include <VrDecimatingSigProc.h>
#include <fstream>


// ********************************************************


template<class iType,class oType> 
class VrCorr : public VrDecimatingSigProc<iType,oType> {
protected:

  float result, old_result, *sqr, weight_sign, max;
  u_int first, first_high, first_low, current, mask;
  int corr_on, differential_offset;
  int low_interval, high_interval, max_id;
  int sync_period, corr_period, corr_count, pattern_length, pattern_count, corr_init;
  
public: 
  virtual const char *name() { return "VrCorr"; }
  virtual int forecast(VrSampleRange output,
		       VrSampleRange inputs[]);
  virtual int work(VrSampleRange output, void *o[],
		   VrSampleRange inputs[], void *i[]);
  virtual void initialize();
  virtual void setSync_Period(int per) { sync_period = per;}
  virtual void setCorr_Period(int per) { corr_period = per + pattern_length;}

  VrCorr(int, int, int, int);
  ~VrCorr();
};

template<class iType,class oType> int
VrCorr<iType,oType>::work(VrSampleRange output, void *ao[],
				   VrSampleRange inputs[], void *ai[]) {



  /*   Correlator that looks for "low-high-low" shape in symbol sequence 
   *   
   *                                       |------------|
   *                                       |            |
   *                                       |            |
   *                                       |            |
   *      |--------------------------------|            |----------|
   *
   *      ^                 ^                                      ^
   *      |    Initialize   |       Produce output values          |  OFF
   *        
   *      |--pattern length-| <---- desired number of outputs ---->|
   *      
   *
   *      |<-- corr_init -->|
   *      |<-----  corr_on --------------------------------------->|

   */


  iType **i = (iType **)ai;
  oType **o = (oType **)ao;
  int size;
  iType input;
  float x, y, sign;

  size = output.size;
  //cout << "Corr with size: "<< size << endl;

  while(size--) {

    input = *i[0]++;

    corr_count++;

    if (corr_count > sync_period-1) {
      corr_count = 0;
      corr_on = 1;
      corr_init = 1;
    }

    if (corr_count == corr_period) {
      corr_on = 0;
      cout << "Max is " << max << " at " << max_id << endl; 
    }

    if (corr_on){

      x = real(input);
      y = imag(input);
      sign = (float) (2 * (x>0.0))-1;
      
      sqr[(current)&mask] =  sign*(x*x + y*y);

      if (corr_init) {    // initialize array used for correlation
	
	pattern_count++;

	if (pattern_count == 1){
	  //cout << "count =1 " << pattern_count << endl;
	  old_result = 0.0; 
	  weight_sign = -1.0;
	  max = 0;
	}
						
        if (pattern_count == low_interval + 1){
	  //cout << "count = low " << pattern_count << endl;
	  weight_sign =  1.0;
	}

	if (pattern_count == low_interval + high_interval + 1) { 
	  //cout << "count = high " << pattern_count << endl;
	  weight_sign = -1.0;
	}
	
	old_result += weight_sign * sign*(x*x + y*y);
	
	*o[0]++ = (oType)0.0;
	current++;
      
	if (pattern_count == pattern_length){ 
	  pattern_count = 0;
	  corr_init = 0;
	  first = current-pattern_length;
	  first_high = first + low_interval;
	  first_low = first_high + high_interval;
	}
      
      } else {           // now start producing output values

	result = old_result;
      
	result += sqr[(first++)&mask];
	result -= (2 * sqr[(first_high++)&mask]);
	result += (2 * sqr[(first_low++)&mask]);
	result -= sqr[(current++)&mask];
	
	if (result > max) {
	  max = result;
	  max_id = corr_count;
	}
	old_result = result;
	//cout << 2.0* result <<  " written to " << (int) o[0] << endl;
	
	*o[0]++ = (oType) (2.0 * result);
      }
      
    } else {    // output zero when not performing correlation
    
      *o[0]++ = (oType) 0.0;
    }
  }  
  return output.size;
  
}

template<class iType,class oType> 
VrCorr<iType,oType>::VrCorr(int dec, int pts_per_symbol, int low_int, int high_int)
  :VrDecimatingSigProc<iType,oType>(1,dec),low_interval(low_int * pts_per_symbol),
        high_interval(high_int * pts_per_symbol)
  {
}

template<class iType, class oType> int
VrCorr<iType,oType>::forecast(VrSampleRange output,
					   VrSampleRange inputs[]) {
  /* dec:1 ratio with history */
  for(unsigned int i=0;i<numberInputs;i++) {
    inputs[i].index=output.index*decimation+ differential_offset; /* ! do not subtract history ! */
    inputs[i].size=output.size*decimation + history-1;
//jca printf ("VrCorr forecast[i] ts %lld size %ld output.index %lld dec %d off %d\n", inputs[i].index, inputs[i].size,
	//jca output.index, decimation, differential_offset);
  }
  return 0;
}  


template<class iType,class oType> 
void VrCorr<iType,oType>::initialize()
{

  sqr = new float[SIZE];
  mask = (u_int) (SIZE-1);
  differential_offset = 0;


  first = 0;
  first_high = low_interval;
  first_low = first_high + high_interval;
#if 0
  current = first_low + low_interval;
#endif

  current = 0;
  corr_on = 1;
  corr_init = 1;
  corr_count = 0;
  sync_period = 500;
  corr_period = 400;
  pattern_length = low_interval + high_interval + low_interval;
  pattern_count = 0;
}

template<class iType,class oType> 
VrCorr<iType,oType>::~VrCorr()
{
  delete sqr;
}

#endif









