/* -*- Mode: c++ -*- */
/*
 * Copyright 2001 Free Software Foundation, Inc.
 * 
 * This file is part of GNU Radio
 * 
 * GNU Radio is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * GNU Radio is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with GNU Radio; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/*
 *  Copyright 1997 Massachusetts Institute of Technology
 * 
 *  Permission to use, copy, modify, distribute, and sell this software and its
 *  documentation for any purpose is hereby granted without fee, provided that
 *  the above copyright notice appear in all copies and that both that
 *  copyright notice and this permission notice appear in supporting
 *  documentation, and that the name of M.I.T. not be used in advertising or
 *  publicity pertaining to distribution of the software without specific,
 *  written prior permission.  M.I.T. makes no representations about the
 *  suitability of this software for any purpose.  It is provided "as is"
 *  without express or implied warranty.
 * 
 */

#ifndef _VRSIGPROC_H_
#define _VRSIGPROC_H_

#include <VrTypes.h>
#include <VrPerfGraph.h>

extern unsigned int cacheSize;
extern float maxLatency;
class VrBuffer;
class VrConnect;
class VrMultiTask;

struct writerLL {
  //everything from current thread's desiredWP to maxValid is valid
  VrSampleIndex maxValid;
  struct writerLL *next;
};

struct readerLL {
  VrSampleIndex index;
  struct readerLL *next;
};

//! The base class for all signal processing modules.
/*!
  All signal processing modules are descendants from VrSigProc.
  In the simplest case a user defines a subclass of VrSigProc and
  overrides the forecast and work methods.
 */

class VrSigProc {
  friend class VrMultiTask;

private:
  /***********************************************************/
  /*** "Internal" methods -- only called from core modules ***/
  /***********************************************************/

  int uses_sync; 
  double proc_samplingFrequency;

  //! data before this has been written 	
  volatile VrSampleIndex WP; 

  //! data before this has been marked and will be written
  volatile VrSampleIndex markedWP; 

  //! maximum size that we should ever output at once.
  /*!
    due to cache size and latency contraints we
    may actually output anywhere < 2*maxOutSize
    to prevent unnecessarily small chunks from being marked
  */
  unsigned int maxOutSize; 

  /*** Performance monitoring data ***/
#ifdef PERFMON
  VrCycleCount *cycles;
#endif
  /*** Input Connectors ***/
  VrConnect ** inputConn;
  unsigned int setupCalled;
  unsigned int outputSize; 
  unsigned int type_size;
  unsigned int itype_size;
  unsigned int initializeCalled;
#ifdef THREADS
  pthread_key_t myMarkedData; //pointer to VrSampleRange
  pthread_key_t inputs_forecasted; //array of VrSampleRange
  pthread_key_t myWriterLL; //pointer to writerLL
  pthread_key_t myReaderLLs; //array of readerLLs
#else
  VrSampleRange myMarkedData;
  VrSampleRange *inputs_forecasted;
  writerLL *myWriterLL;
  readerLL *myReaderLLs; //array of size numberInputs
#endif

  /* Linked list of writing threads */
  /*   each writerLL structure indicated where the thread started writing */
  volatile writerLL *first, *last;
  MUTEX_DECLARE(mutex);
  void attach_writer(writerLL *r);
  void detach_writer(writerLL *r);

  /*!  \brief Cause initialize methods to be called.
   * This is called on all sinks for which isConnectedToSource is
   * true, and results in the pre_initialize and initialize methods
   * being called in the order source to sink.
   */
  void init_base();

  /*!
   * \brief Set sampling frequency based on upstream sampling freq.
   *
   * This method handles automatically setting the sampling frequency
   * for this module based on the sampling frequency of the upstream
   * modules.  For most modules, our sampling frequency is the same as
   * our upstream neighbor.  VrInterpolatingSigProc and
   * VrDecimatingSigProc override this.
   *
   * This method is called immediately prior to initialize.
   */
  virtual void pre_initialize();

  //! Initialize any local state that is dependent on sampling frequency
  virtual void initialize() {};

  virtual bool isConnectedToSource();

  int minwritespace(VrSampleIndex newWP, unsigned int desired); 

  //These work best in the constructor, but will work
  //  in initialize if the SigProc has ONLY ONE output buffer
  void initOutputBuffers(int n);
  virtual void initOutputBuffer(int n); //create a particular buffer
  void initMarkedData();
  virtual unsigned int mapSizeUp(int i, unsigned int size);

  //!is data marked to be computed?
  virtual bool dataMarked(VrSampleRange r); 


  //! figure out what data to compute
  virtual int markData(VrSampleRange r);
  static const int MARK_ALREADY = 2;  	  //!< all data is already computed
  static const int MARK_READY = 1;    	  //!< data is ready to be computed (and has been marked)
  static const int MARK_READY_NO_MARK = 0;//!< data is ready on this level or above, 
					  //   but no further data should be marked
  static const int MARK_NO_READY = -1;	  //!< no data is ready to be computed
  static const int MARK_THREAD = -2;  	  //!< data is marked by another thread
  static const int MARK_continue = -3;    //!< flag distinct from all others (keep last)

  virtual int VrSigProc::markDataUpstream (VrSampleRange *inputs,
					   bool *dataMarkedUpstream);
  //!compute the marked data
  virtual bool compute();

protected:

  VrBuffer** outBuffer;

  //! max Down Stream Read Size.  How big a read we might get.  maxOutSize should never exceed this.
  unsigned int maxDSReadSize; 

  unsigned int numberInputs;
  unsigned int numberOutputs; 
  virtual VrSigProc *getUpstreamModuleN(port p);
  double getInputSamplingFrequencyN(port p);

  //! set required multiple for number of items produced
  void setOutputSize(int o) { outputSize=o; }
  unsigned int getOutputSize() {return outputSize;}

  void setup_upstream();
  int getNumberInputs() {return numberInputs;}

  /*!
   * \returns true if we're working on the current output block, else
   * false, indicating that we're working on a future output block
   */
  bool is_synced (VrSampleIndex arg_index) {
    return WP >= arg_index;
  }
  
  //! force rest of procedure to run sequentially
  void sync (VrSampleIndex arg_index);

  VrSampleIndex proc_minRP();

public:    
  VrSigProc(int number_of_outputs, unsigned int arg_itype_size, unsigned int arg_type_size);
  virtual ~VrSigProc();

  //! connect our next input to \p proc's nth output
  bool connect_proc(VrSigProc* proc, port n);

  /*******************************************************************/
  /*** Methods you should override in processing modules ***/
  /*******************************************************************/

  //! default name for a module (override with actual name)

  virtual const char *name() { return "VrSigProc"; }

  //! map output range to required input ranges
  //  0=OK results and -1=don't know what I need yet

  virtual int forecast(VrSampleRange output, VrSampleRange inputs[]); 

  /*!
   * Returns the total number of bytes read (ignores written bytes)
   * per second by the upstream chain ending with this module
   *  zero has a special meaning (no outputs in cache)
   */

  virtual float memoryTouched(); 

  /*!
   * \brief Generate up to output.size output points at o[output#][sample#]
   *  using data at the inputptrs[input#][sample#]
   *
   *  Pointers start at corresponding VrSampleIndex in the range params.
   *  output.size is a multiple of outputSize.
   *  Returns the number of outputs (<= output.size) that it was
   *  able to compute with the given input ranges.  If this value
   *  is < output.size, forecast() will need to be called again.
   */

  virtual int work(VrSampleRange output, void *o[],
		   VrSampleRange inputs[], void *i[]) = 0; 

  //what fraction of the input samples (from input n) does this module look
  //at? 1.0=all 0.0=none <1.0=some fraction
  // >1.0 -- very very rare, means looks at proportionally more
  //         data (e.g. always twice the input data) --
  //         this isn't taps or any constant extra data

  virtual float averageInputUse(int n) {return 1.0;} 

  virtual int checkOutputSamplingFrequency(float) { return 0;} 

  //! return true iff this VrSigProc is a sink
  bool isSink ();
  
  /**********************************************************/
  /*** "External" methods -- call from out-of-band script ***/
  /**********************************************************/
  virtual int setSamplingFrequency(double sf); 

  //has data already been computed?
  virtual bool dataReady(VrSampleRange r); 
  virtual void size_setup(unsigned int size);
  virtual void setup();
  VrSampleIndex getMarkedWP() {return markedWP;}

#ifdef PERFMON
  /*** Performance monitoring procedures ***/
  unsigned int num_print_stats;
  long long getTotalCycles();
  long long getTotalCycles(int m);
  long getTotalSamples();
  long long getCyclesPerSample();
  long long getCyclesPerSample(int m);
  void addToGraph(VrPerfGraph *g);
  void print_stats();
#endif
  unsigned int getMaxOutputSize() {return maxOutSize;}
  VrSampleIndex getWP() {return WP;}
  double getSamplingFrequency() { return proc_samplingFrequency; }
  virtual int callback(int attribute_number, float value) { return 0; }
};

// "New World Order" connect macro...
// connect a's nth output to b's next input
// Ahhhh, doesn't that feel better.
// #define	NWO_CONNECTN(a, n, b) (b)->connect_proc (a, n)
// #define	NWO_CONNECT(a, b)     NWO_CONNECTN (a, 0, b)

static inline void
NWO_CONNECTN (VrSigProc *src, int n, VrSigProc *dst)
{
  dst->connect_proc (src, n);
}

static inline void
NWO_CONNECT (VrSigProc *src, VrSigProc *dst)
{
  NWO_CONNECTN (src, 0, dst);
}

// Macro to connect_proc (next) input of a to (1st) output of B.
// (deprecated)
#define CONNECTN(a, b, n, sf, bps) NWO_CONNECTN (b, n, a)
#define CONNECT(a, b, sf, bps)	   NWO_CONNECT (b, a)

#endif
