;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: OPAL; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;         The Garnet User Interface Development Environment.      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; This code was written as part of the Garnet project at          ;;;
;;; Carnegie Mellon University, and has been placed in the public   ;;;
;;; domain.  If you are using this code or any part of Garnet,      ;;;
;;; please contact garnet@cs.cmu.edu to be put on the mailing list. ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Changes:
;;; 26-May-92 ecp  Added new constant number-of-slots-of-update-info-struct.
;;; 01-Apr-92 amickish  Added copy-bbox-fn
;;; 11-Mar-92 ecp  Added two new fields width and height to win-update-info.
;;; 19-Feb-92 koz  Eliminated extra non-printable object at end of 
;;;		   invalid-objects list.
;;;  9-Dec-91 ecp  opal-gc has new stored-clip-mask field
;;; 26-Nov-91 ecp  changed erase-bbox to correctly handle double-buffered
;;;		   windows with background color.
;;; 26-Mar-91 sylvain  #+kcl patches
;;;  1-Oct-90 ecp  Made the print function for opal-gc not print font.
;;; 18-Jun-90 ecp  Added *clear* for erasing buffers.
;;;  5-Jun-90 dzg  Changed update-info structure to reduce storage allocation.
;;;  4-Jun-90 ecp  Altered erase-bbox to handle double buffering.
;;; 16-Apr-90 ecp  Moved defun of font-to-xfont from new-defs.lisp to
;;;		   create-instances2.lisp
;;; 19-Mar-90  Changed tile to stipple
;;;  9-Mar-90  Moved a bunch of defvars to defs.lisp.
;;;  5-Dec-89  Moved definition of new-garnet-window-name here from windows.lisp,
;;;            Added "#-cmu nil" to fix-font-path.
;;;             
(in-package "OPAL" :use '("LISP" "KR"))

(defstruct bbox
	x1
	y1
	x2
	y2
	valid-p)


;; Force-Computation-P is necessary since if an object R is in an aggregate A,
;; and you add-component that aggregate into another (visible) aggregate, then
;; R will be marked dirty, but it will not be added to the invalid-objects list
;; of the window, so at update time all its values in :update-slots-values will
;; be incorrect and will need to be recomputed.  Obtuse, but this works!

(defstruct (update-info (:print-function update-info-print-function))
	window
	old-bbox
	bits)

;;; This constant is used in debug/objsize.lisp to determine the
;;; size in bytes of an update-info structure.
;;; NOTE: IF YOU CHANGE THE DEFINITION OF UPDATE-INFO, BE SURE
;;; TO CHANGE THE VALUE OF THIS CONSTANT.
(defconstant number-of-slots-of-update-info-struct 3)

;;; The update-info-bits field is used to encode the following:
;;;   dirty-p
;;;   aggregate-p
;;;   invalid-p
;;;   force-computation-p
;;;   on-fastdraw-list-p

(defmacro bit-setter (object position value)
  (cond ((eq value T)
	 ;; Value is T at compile time.
	 `(setf (update-info-bits ,object)
		(logior (update-info-bits ,object) ,(ash 1 position))))
	((null value)
	 ;; Value is NIL at compile time.	 
	 `(setf (update-info-bits ,object)
		(logand (update-info-bits ,object)
			,(lognot (ash 1 position)))))
	(t
	 ;; Value is not known at compile time
	 `(if ,value
	      (setf (update-info-bits ,object)
		    (logior (update-info-bits ,object) ,(ash 1 position)))
	      (setf (update-info-bits ,object)
		    (logand (update-info-bits ,object)
			    ,(lognot (ash 1 position))))))))

(defmacro update-info-dirty-p (object)
  `(logbitp 0 (update-info-bits ,object)))

(defsetf update-info-dirty-p (object) (value)
  `(bit-setter ,object 0 ,value))


(defmacro update-info-aggregate-p (object)
  `(logbitp 1 (update-info-bits ,object)))

(defsetf update-info-aggregate-p (object) (value)
  `(bit-setter ,object 1 ,value))


(defmacro update-info-invalid-p (object)
  `(logbitp 2 (update-info-bits ,object)))

(defsetf update-info-invalid-p (object) (value)
  `(bit-setter ,object 2 ,value))


(defmacro update-info-force-computation-p (object)
  `(logbitp 3 (update-info-bits ,object)))

(defsetf update-info-force-computation-p (object) (value)
  `(bit-setter ,object 3 ,value))


(defmacro update-info-on-fastdraw-list-p (object)
  `(logbitp 4 (update-info-bits ,object)))

(defsetf update-info-on-fastdraw-list-p (object) (value)
  `(bit-setter ,object 4 ,value))
	 

(defun update-info-print-function (struct stream depth)
  (declare (ignore depth))
  (format stream "#<Update-Info dirty-p ~A invalid-p ~A>"
	(update-info-dirty-p struct)
	(update-info-invalid-p struct)))

(defstruct (win-update-info (:print-function win-update-info-print-function))
	invalid-objects
	last-invalid-obj
	invalid-slots
	new-bbox
	clip-mask-1
	clip-mask-2
	old-aggregate
        width
        height
)
#|
(defun win-update-info-print-function (struct stream depth)
  (declare (ignore depth))
  (format stream "#<Win-Update-Info invalid-objects ")
  (let* ((invalid-objs (win-update-info-invalid-objects struct))
 	 (last-invalid-obj (win-update-info-last-invalid-obj struct))
	 (cdr-of-lio (cdr last-invalid-obj)))
    (if invalid-objs
	(progn
	  (setf (cdr last-invalid-obj) NIL)
	  (format stream "(")
	  (dolist (obj invalid-objs)
		(format stream "~A " obj))
	  (format stream ")")
	  (setf (cdr last-invalid-obj) cdr-of-lio))
	(format stream "NIL"))
    (format  stream " invalid-slots ~A>"
	(win-update-info-invalid-slots struct))))
|#

;;; The invalid objects slot used to be unprintable because it had
;;; an extra item at the end, but that has been eliminated.
(defun win-update-info-print-function (struct stream depth)
  (declare (ignore depth))
  (format stream "#<Win-Update-Info invalid-objects ~A invalid-slots ~A>"
     (win-update-info-invalid-objects struct)
     (win-update-info-invalid-slots struct)))

(defstruct (opal-gc (:print-function opal-gc-print-function))
	gcontext
	opal-style		; This is either a line or filling style

	function
	foreground
	background
	line-width
	line-style
	cap-style
	join-style
	dashes		;; do not set to NIL
	font		;; do not set to NIL
	fill-style
	fill-rule
	stipple
	clip-mask
        stored-clip-mask  ;; The clip-mask actually stored in
                          ;; the xlib:gcontext -- except if the
                          ;; clip-mask is :none, in which case
                          ;; this contains a list like '(nil 0 0 0)
                          ;; (to avoid unnecessary consing)
)

(defun opal-gc-print-function (gc stream depth)
  (declare (ignore depth))
  (format stream "#<Opal-GC function ~A clip-mask ~A>"
	  (opal-gc-function gc)
	  (opal-gc-clip-mask gc)))

(defvar *free-cons* NIL)

(defvar *font-hash-table* (make-hash-table :test #'equal))

;; This exists expressly to convert paths using CMU's
;; ext:search-list keys into normal paths.  Not robust, but better than
;; what used to be done...
(defun fix-font-path (path-argument)
  (when path-argument
    (let* ((path path-argument)
	   (colon-posn (position #\: path))
	   (search-path (when colon-posn
			  #+cmu (ext:search-list
				 (subseq path 0 (1+ colon-posn)))
			  #-cmu nil)))
      (if search-path
	  (concatenate 'string (car search-path) (subseq path (1+ colon-posn)))
	  (if (eq (position #\/ path :from-end t) (1- (length path)))
	      path
	      (concatenate 'string path "/"))))))

;; Hack used in font-to-xfont to counteract ridiculous tendency
;; of CLX to tack on #\null characters at the end of font paths.
(defun remove-null-char (s)
  #+kcl (remove #\^@   s :start (1- (length s)))
  #-kcl (remove #\null s :start (1- (length s))))

(defmacro merge-bbox (dest-bbox source-bbox)
  `(when (bbox-valid-p ,source-bbox)
     (if (bbox-valid-p ,dest-bbox)
      (progn
	(setf (bbox-x1 ,dest-bbox)
		(MIN (bbox-x1 ,dest-bbox) (bbox-x1 ,source-bbox)))
	(setf (bbox-y1 ,dest-bbox)
		(MIN (bbox-y1 ,dest-bbox) (bbox-y1 ,source-bbox)))
	(setf (bbox-x2 ,dest-bbox)
		(MAX (bbox-x2 ,dest-bbox) (bbox-x2 ,source-bbox)))
	(setf (bbox-y2 ,dest-bbox)
		(MAX (bbox-y2 ,dest-bbox) (bbox-y2 ,source-bbox))))
      (progn
	(setf (bbox-x1 ,dest-bbox) (bbox-x1 ,source-bbox))
	(setf (bbox-y1 ,dest-bbox) (bbox-y1 ,source-bbox))
	(setf (bbox-x2 ,dest-bbox) (bbox-x2 ,source-bbox))
	(setf (bbox-y2 ,dest-bbox) (bbox-y2 ,source-bbox))
	(setf (bbox-valid-p ,dest-bbox) T)))))

;;; Leaves the bboxes valid-p bits alone.  Only copies the dimensions.
(defmacro copy-bbox-dims (dest-bbox source-bbox)
  `(progn
	(setf (bbox-x1 ,dest-bbox) (bbox-x1 ,source-bbox))
	(setf (bbox-y1 ,dest-bbox) (bbox-y1 ,source-bbox))
	(setf (bbox-x2 ,dest-bbox) (bbox-x2 ,source-bbox))
	(setf (bbox-y2 ,dest-bbox) (bbox-y2 ,source-bbox))))

;;; Performs the same function as copy-bbox-dims *AND* copies the valid-p bit
(defun copy-bbox-fn (dest-bbox source-bbox)
  (copy-bbox-dims dest-bbox source-bbox)
  (setf (bbox-valid-p dest-bbox) (bbox-valid-p source-bbox)))

;; Returns T iff the dimensions of two bboxes are different. Ignores valid-p.
(defmacro bbox-dims-differ (bb1 bb2)
  `(not (and
	  (= (bbox-x1 ,bb1) (bbox-x1 ,bb2))
	  (= (bbox-y1 ,bb1) (bbox-y1 ,bb2))
	  (= (bbox-x2 ,bb1) (bbox-x2 ,bb2))
	  (= (bbox-y2 ,bb1) (bbox-y2 ,bb2)))))

;;; Updates the bbox given (probably the object's :old-bbox slot value) with
;;; the values from the object.  This *presumes* that the object is visible!
(defmacro update-bbox (object bbox)
    `(let ((left (g-value ,object :left))
	   (top  (g-value ,object :top )))
	(setf (bbox-x1 ,bbox) left)
	(setf (bbox-y1 ,bbox) top)
	(setf (bbox-x2 ,bbox) (+ left (g-value ,object :width )))
	(setf (bbox-y2 ,bbox) (+ top  (g-value ,object :height)))
	(setf (bbox-valid-p ,bbox) T)))

;;; Returns true if they intersect (ignores the valid bit!)
(defmacro bbox-intersect-p (bb1 bb2)
 `(and (<= (bbox-x1 ,bb1) (bbox-x2 ,bb2))   ;; 1 not right of 2
       (<= (bbox-x1 ,bb2) (bbox-x2 ,bb1))   ;; 2 not right of 1
       (<= (bbox-y1 ,bb1) (bbox-y2 ,bb2))   ;; 1 not below 2
       (<= (bbox-y1 ,bb2) (bbox-y2 ,bb1)))) ;; 2 not below 1

;;; Returns true iff bbox intersects either bb1 or bb2.  This will check if
;;; bb2 is NIL, but if bb1 is NIL this will crash.
(defmacro bbox-intersects-either-p (bbox bb1 bb2)
  `(or (bbox-intersect-p ,bbox ,bb1)
       (and ,bb2 (bbox-intersect-p ,bbox ,bb2))))

;; Erases this bbox from this window (or its buffer). Ignores valid bit.
(defun erase-bbox (bb drawable buffer buffer-gc)
  (if buffer
      (let ((background (xlib:gcontext-background buffer-gc)))
        (xlib:with-gcontext (buffer-gc :function opal::*copy*
				       :foreground background)
	  (xlib:draw-rectangle buffer buffer-gc
			       (bbox-x1 bb)
			       (bbox-y1 bb)
			       (- (bbox-x2 bb) (bbox-x1 bb))
			       (- (bbox-y2 bb) (bbox-y1 bb))
			       t)))
      (xlib:clear-area drawable
		       :x (bbox-x1 bb)
		       :y (bbox-y1 bb)
		       :width  (- (bbox-x2 bb) (bbox-x1 bb))
		       :height (- (bbox-y2 bb) (bbox-y1 bb)))))

#|
;;; NOTE THAT THE STUFF ABOUT THE LET IS CACA.
;; Returns True if slot does not have a formula and is non-NIL or if
;; it has a formula with a valid cached value.  NOTE:  this uses the
;; special variable "***formula".  This will be a global unless you place it
;; inside a LET (which is how it is normally called!)
(defmacro slot-is-valid (schema slot)
  `(if (formula-p (setq ***formula (g-local-value ,schema ,slot)))
	(kr::cache-is-valid (g-local-value ***formula :cached-value))
	***formula))

;; Returns True if the :top, :left, :width, and :height are either non-NIL
;; non-formulas or valid cached-values of formulas.  It is used so that if
;; an aggregate's dimensions are not valid, instead of computing them we'll
;; just go ahead and update it!
(defmacro dims-are-valid (schema)
  `(let (***formula)
	(and (slot-is-valid ,schema :left)
	     (slot-is-valid ,schema :top)
	     (slot-is-valid ,schema :width)
	     (slot-is-valid ,schema :height))))
|#

;; Takes a bbox and a clip mask, and goes through and sets the fields properly
;; within the clip mask.  Ignores valid bit.
(defmacro bbox-to-clip-mask (bb clip-mask)
  `(let ((cm ,clip-mask))
     (setf (car cm) (bbox-x1 ,bb))
     (setf (car (setq cm (cdr cm))) (bbox-y1 ,bb))
     (setf (car (setq cm (cdr cm))) (- (bbox-x2 ,bb) (bbox-x1 ,bb)))
     (setf (cadr cm) (- (bbox-y2 ,bb) (bbox-y1 ,bb)))))

;; propagate dirty bit of T from this object up towards root
;; this will do ugly things if called with object == NULL.
(defmacro propagate-dirty-bit (object update-info)
   `(unless (update-info-dirty-p ,update-info)
      (let ((temp ,object) (temp-update-info ,update-info))
        (loop
	  (setf (update-info-dirty-p temp-update-info) T)
	  (if (or (null (setq temp (g-local-value temp :parent)))
		  (update-info-dirty-p
		     (setq temp-update-info
			   (g-local-value temp :update-info))))
		(return))))))

;; this adds the object to the window's invalid-objects entry in its
;; :win-update-info slot and then sets the object's invalid-p to T.
(defmacro make-object-invalid (gob gob-update-info the-window)
  `(let* ((w-info (g-local-value ,the-window :win-update-info))
	  (last-invalid (win-update-info-last-invalid-obj w-info)))
    (if *free-cons*
      (progn
	(if last-invalid
	    				;; some free cons'es, Already entries
		(setf (win-update-info-last-invalid-obj w-info)
		   (setf last-invalid
			(setf (cdr last-invalid) *free-cons*)))

	     				;; some free cons'es, No entries
		(setf (win-update-info-invalid-objects w-info)
		    (setf (win-update-info-last-invalid-obj w-info)
			(setf last-invalid
			   *free-cons*))))
	(setf *free-cons* (cdr *free-cons*))
        (setf (cdr last-invalid) nil) ;;; make invalid-objects lists be
				      ;;; genuine Lisp lists.
	(setf (car last-invalid) ,gob))

    ;else no *free-cons* cells...
	(if last-invalid
	    				;; no free cons'es, Already entries
		(setf (win-update-info-last-invalid-obj w-info)
		   (setf (cdr last-invalid)
			(list ,gob)))

	     				;; no free cons'es, No entries
		(setf (win-update-info-invalid-objects w-info)
		    (setf (win-update-info-last-invalid-obj w-info)
			(list ,gob)))))
    (setf (update-info-invalid-p ,gob-update-info) T)))

#|
	The following code is no longer used by the update algorithm!

;;; This is almost exactly a clone of the previous macro, but it deals
;;; with adding an object to a window's fastdraw-objects list...  It is a
;;; support macro for "add-to-fastdraw-list"
(defmacro add-obj-to-fastdraw-list (obj w-info)
  `(let ((last-fastdraw (win-update-info-last-fastdraw-obj ,w-info)))
    (if *free-cons*
      (progn
	(if last-fastdraw
	    				;; some free cons'es, Already entries
		(setf (win-update-info-last-fastdraw-obj ,w-info)
		   (setf last-fastdraw
			(setf (cdr last-fastdraw) *free-cons*)))

	     				;; some free cons'es, No entries
		(setf (win-update-info-fastdraw-objects ,w-info)
		    (setf (win-update-info-last-fastdraw-obj ,w-info)
			(setf last-fastdraw
			   *free-cons*))))
	(setf *free-cons* (cdr *free-cons*))
	(setf (car last-fastdraw) ,obj))

    ;else no *free-cons* cells...
	(if last-fastdraw
	    				;; no free cons'es, Already entries
		(setf (win-update-info-last-fastdraw-obj ,w-info)
		   (setf (cdr last-fastdraw)
			(list ,obj)))

	     				;; no free cons'es, No entries
		(setf (win-update-info-fastdraw-objects ,w-info)
		    (setf (win-update-info-last-fastdraw-obj ,w-info)
			(list ,obj)))))
))

(defmacro add-to-fastdraw-list (gob first-changed w-info)
  `(progn
    (add-obj-to-fastdraw-list ,gob ,w-info)
    (add-obj-to-fastdraw-list ,first-changed ,w-info)
    (setf (update-info-on-fastdraw-list-p (g-local-value ,gob :update-info))
	T)))
|#

(defun new-garnet-window-name ()
  (let ((*print-base* 10))
    (format nil "Opal ~S" (incf *opal-window-count*))))


;;; Wonderful hack that allows one to use update-all to update even
;;; those windows that have never been updated, and thus have an empty
;;; :drawable slot and do not appear in the hash-table
;;; opal::*drawable-to-window-mapping*.  I include NIL at the head of
;;; the list so that Common Lisp's delete will always work.
(defvar *windows-that-have-never-been-updated* '(NIL))
