/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"gleptree.h"
#include	"glegcontainer.h"
#include	"glercargs.h"
#include	"glednd.h"
#include	<gdk/gdk.h>
#include	<string.h>
#include	<stdio.h>


typedef struct
{
  GlePTree	*ptree;
  GtkCTreeNode	*parent;
  GtkCTreeNode	*sibling;
  guint		 depth;
} GlePTreeAppendData;


/* --- signals --- */
enum {
  SIGNAL_BUTTON_CLICK,
  SIGNAL_LAST
};
typedef	gboolean	(*SignalButtonClick)	(GtkObject  *object,
						 GleGWidget *click_widget,
						 guint	     root_x,
						 guint	     root_y,
						 guint	     button,
						 guint	     time,
						 gpointer    func_data);


/* --- prototypes --- */
static void		gle_ptree_class_init		(GlePTreeClass	*class);
static void		gle_ptree_init			(GlePTree	*ptree);
static void		gle_ptree_destroy		(GtkObject	*object);
static void		gle_ptree_append_gwidget	(GleGWidget	*gwidget,
							 GlePTreeAppendData*data);
static void		gle_ptree_row_data_update	(GlePTreeRowData*row_data);
static void		gle_ptree_row_data_destroyed	(GlePTreeRowData*row_data);
static void		gle_ptree_row_data_free		(GlePTreeRowData*row_data);
static void		gle_ptree_gcontainer_add	(GleGContainer  *gcontainer,
							 GlePTreeRowData*row_data,
							 GleGWidget	*gchild);
static void		gle_ptree_gcontainer_remove	(GleGContainer  *gcontainer,
							 GlePTreeRowData*row_data,
							 GleGWidget	*gchild);
static gint		gle_ptree_button_press_event	(GtkWidget	*widget,
							 GdkEventButton *event);
static GleGObject*	gle_ptree_dnd_start		(GtkWidget      *widget,
							 gint            root_x,
							 gint            root_y);
static GleGObject*	gle_ptree_dnd_check		(GtkWidget	*widget,
							 GtkType	 drag_type,
							 gint            root_x,
							 gint            root_y);


/* --- variables --- */
static guint	ptree_init_col_width[GLE_PTREE_N_COLS] =
{
  150	/* GLE_PTREE_COL_OTYPE */,
  50	/* GLE_PTREE_COL_STATE */,
  50	/* GLE_PTREE_COL_GNAME */,
  50	/* GLE_PTREE_COL_NAME */,
};
static GtkCTreeClass	*parent_class = NULL;
static GlePTreeClass	*gle_ptree_class = NULL;
static gint		 gle_row_data_counter = 0;
static guint		 ptree_signals[SIGNAL_LAST] = { 0 };


/* --- functions --- */
GtkType
gle_ptree_get_type (void)
{
  static GtkType ptree_type = 0;
  
  if (!ptree_type)
    {
      GtkTypeInfo ptree_info =
      {
	"GlePTree",
	sizeof (GlePTree),
	sizeof (GlePTreeClass),
	(GtkClassInitFunc) gle_ptree_class_init,
	(GtkObjectInitFunc) gle_ptree_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      
      ptree_type = gtk_type_unique (GTK_TYPE_CTREE, &ptree_info);
    }
  
  return ptree_type;
}

static void
gle_ptree_marshal_button_click (GtkObject      *object,
				GtkSignalFunc  func,
				gpointer       func_data,
				GtkArg	       *args)
{
  gboolean *handled;
  SignalButtonClick sfunc = (SignalButtonClick) func;
  
  handled = GTK_RETLOC_BOOL (args[5]);
  
  *handled = sfunc (object,
		    GTK_VALUE_POINTER (args[0]),
		    GTK_VALUE_UINT (args[1]),
		    GTK_VALUE_UINT (args[2]),
		    GTK_VALUE_UINT (args[3]),
		    GTK_VALUE_UINT (args[4]),
		    func_data);
}

static void
gle_ptree_class_init (GlePTreeClass  *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  
  gle_ptree_class = class;
  parent_class = gtk_type_class (GTK_TYPE_CTREE);
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  
  ptree_signals[SIGNAL_BUTTON_CLICK] =
    gtk_signal_new ("button-click",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GlePTreeClass, button_click),
		    gle_ptree_marshal_button_click,
		    GTK_TYPE_BOOL, 5,
		    GLE_TYPE_GWIDGET,
		    GTK_TYPE_UINT,
		    GTK_TYPE_UINT,
		    GTK_TYPE_UINT,
		    GTK_TYPE_UINT);
  gtk_object_class_add_signals (object_class, ptree_signals, SIGNAL_LAST);
  
  object_class->destroy = gle_ptree_destroy;
  
  widget_class->button_press_event = gle_ptree_button_press_event;
  
  gle_row_data_counter = 0;
  class->row_data_mem_chunk_ref = 0;
  class->row_data_mem_chunk = NULL;
  class->button_click = NULL;
}

static void
gle_ptree_init (GlePTree *ptree)
{
  guint i;
  gchar *titles[GLE_PTREE_N_COLS] =
  {
    "Type"	/* GLE_PTREE_COL_OTYPE */,
    "State"	/* GLE_PTREE_COL_STATE */,
    "GleName"	/* GLE_PTREE_COL_GNAME */,
    "Name"	/* GLE_PTREE_COL_NAME */,
  };
  
  ptree->gwidget = NULL;
  ptree->shown_row_data = NULL;
  
  /* setup ctree
   */
  gtk_ctree_construct (GTK_CTREE (ptree), GLE_PTREE_N_COLS, 0, titles);
  gtk_clist_set_selection_mode (GTK_CLIST (ptree), GTK_SELECTION_EXTENDED);
  
  /* setup clist
   */
  gtk_clist_column_titles_passive (GTK_CLIST (ptree));
  for (i = 0; i < GLE_PTREE_N_COLS; i++)
    gtk_clist_set_column_width (GTK_CLIST (ptree), i, ptree_init_col_width[i]);
  gtk_clist_set_column_justification (GTK_CLIST (ptree), GLE_PTREE_COL_GNAME, GTK_JUSTIFY_RIGHT);
  
  /* setup widget
   */
  gtk_widget_set_usize (GTK_WIDGET (ptree),
			GLE_RCVAL_PTREE_WIDTH > 0 ? GLE_RCVAL_PTREE_WIDTH : -1,
			GLE_RCVAL_PTREE_HEIGHT > 0 ? GLE_RCVAL_PTREE_HEIGHT : -1);

  /* implement dnd facilities
   */
  gle_dnd_add_source (GTK_WIDGET (ptree), 2, gle_ptree_dnd_start, NULL);
  gle_dnd_add_target (GTK_WIDGET (ptree), gle_ptree_dnd_check);
  gle_dnd_add_notifier (GTK_WIDGET (ptree), (GleDndNotify) gle_ptree_mark_gwidget);

  /* row data memchunks
   */
  if (!gle_ptree_class->row_data_mem_chunk)
    gle_ptree_class->row_data_mem_chunk =
      g_mem_chunk_new ("GlePTreeRowData mem chunks",
		       sizeof (GlePTreeRowData),
		       sizeof (GlePTreeRowData) * 128,
		       G_ALLOC_AND_FREE);
  gle_ptree_class->row_data_mem_chunk_ref += 1;
}

GtkWidget*
gle_ptree_new (void)
{
  GlePTree *ptree;
  
  ptree = gtk_type_new (GLE_TYPE_PTREE);
  
  return GTK_WIDGET (ptree);
}

static void
gle_ptree_destroy (GtkObject *object)
{
  GlePTree *ptree;
  
  ptree = GLE_PTREE (object);
  
  gle_ptree_set_gwidget (ptree, NULL);
  
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
  
  gle_ptree_class->row_data_mem_chunk_ref -= 1;
  if (gle_ptree_class->row_data_mem_chunk_ref == 0)
    {
      g_mem_chunk_destroy (gle_ptree_class->row_data_mem_chunk);
      gle_ptree_class->row_data_mem_chunk = NULL;
    }
}

static void
gle_ptree_gwidget_destroyed (GlePTree *ptree)
{
  ptree->gwidget = NULL;
  gle_ptree_rebuild (ptree);
}

void
gle_ptree_set_gwidget (GlePTree	  *ptree,
		       GleGWidget *gwidget)
{
  g_return_if_fail (ptree != NULL);
  g_return_if_fail (GLE_IS_PTREE (ptree));
  
  if (gwidget)
    {
      g_return_if_fail (GLE_IS_GWIDGET (gwidget));
      
      if (GLE_GOBJECT_DESTROYED (gwidget))
	gwidget = NULL;
    }
  
  if (gwidget != ptree->gwidget)
    {
      if (ptree->gwidget)
	GLE_NOTIFIER_REMOVE_FD (ptree->gwidget, gle_ptree_gwidget_destroyed, ptree);
      ptree->gwidget = gwidget;
      if (ptree->gwidget)
	GLE_NOTIFIER_INSTALL_DATA (ptree->gwidget,
				   "pre_destroy",
				   gle_ptree_gwidget_destroyed,
				   ptree);
      
      gle_ptree_rebuild (ptree);
    }
}

static void
gle_ptree_foreach_ctree_row_update (GtkCTree	 *ctree,
				    GtkCTreeNode *cnode,
				    gpointer	  data)
{
  GlePTreeRowData *row_data;
  
  row_data = gtk_ctree_node_get_row_data (ctree, cnode);
  gle_ptree_row_data_update (row_data);
}

void
gle_ptree_rebuild (GlePTree *ptree)
{
  g_return_if_fail (ptree != NULL);
  g_return_if_fail (GLE_IS_PTREE (ptree));
  
  gtk_clist_freeze (GTK_CLIST (ptree));
  
  gtk_clist_clear (GTK_CLIST (ptree));
  if (GLE_RCVAL_PTREE_RESET_COLUMNS)
    {
      guint i;
      
      for (i = 0; i < GLE_PTREE_N_COLS; i++)
	gtk_clist_set_column_width (GTK_CLIST (ptree), i, ptree_init_col_width[i]);
    }
  
  if (ptree->gwidget)
    {
      GlePTreeAppendData data;
      
      data.ptree = ptree;
      data.parent = NULL;
      data.sibling = NULL;
      data.depth = 0;
      gle_ptree_append_gwidget (ptree->gwidget, &data);
    }
  
  gtk_clist_thaw (GTK_CLIST (ptree));
}

void
gle_ptree_update (GlePTree *ptree)
{
  g_return_if_fail (ptree != NULL);
  g_return_if_fail (GLE_IS_PTREE (ptree));
  
  gtk_clist_freeze (GTK_CLIST (ptree));
  
  gtk_ctree_post_recursive (GTK_CTREE (ptree), NULL, gle_ptree_foreach_ctree_row_update, NULL);
  
  gtk_clist_thaw (GTK_CLIST (ptree));
}

static void
gle_ptree_foreach_find (GtkCTree     *ctree,
			GtkCTreeNode *cnode,
			gpointer      data_p)
{
  GlePTreeRowData *row_data;
  gpointer *data = data_p;
  
  row_data = gtk_ctree_node_get_row_data (ctree, cnode);
  if (row_data && row_data->gwidget == data[0])
    data[1] = row_data;
}

GlePTreeRowData*
gle_ptree_row_data_from_gwidget (GlePTree       *ptree,
				 GleGWidget     *gwidget)
{
  gpointer data[2];

  g_return_val_if_fail (ptree != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PTREE (ptree), NULL);
  g_return_val_if_fail (gwidget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), NULL);
  
  data[0] = gwidget;
  data[1] = NULL;
  gtk_ctree_post_recursive (GTK_CTREE (ptree), NULL, gle_ptree_foreach_find, data);

  return data[1];
}

void
gle_ptree_insert_gwidget (GlePTree   *ptree,
			  GleGWidget *gwidget)
{
  GleGWidget *parent;
  GlePTreeRowData *row_data;
  GlePTreeAppendData data;

  g_return_if_fail (ptree != NULL);
  g_return_if_fail (GLE_IS_PTREE (ptree));
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));

  if (gle_ptree_row_data_from_gwidget (ptree, gwidget))
    {
      g_warning ("PTree already contains `%s' proxy %s \"%s\"",
		 gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)),
		 gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gwidget)),
		 GLE_GOBJECT (gwidget)->gname);
      return;
    }
  parent = gwidget->parent;
  if (parent)
    row_data = gle_ptree_row_data_from_gwidget (ptree, parent);
  else
    row_data = NULL;
  if (!parent || !row_data)
    {
      g_warning ("PTree does not contain parent of `%s' proxy %s \"%s\"",
		 gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)),
		 gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gwidget)),
		 GLE_GOBJECT (gwidget)->gname);
      return;
    }

  data.ptree = ptree;
  data.depth = 0;
  data.parent = row_data->ct_node;
  
  if (ptree->gwidget)
    {
      data.sibling = NULL;
      gle_ptree_append_gwidget (gwidget, &data);
    }
}

gboolean
gle_ptree_mark_gwidget (GlePTree   *ptree,
			GleGWidget *gwidget)
{
  GlePTreeRowData *row_data;
  
  g_return_val_if_fail (ptree != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_PTREE (ptree), FALSE);
  
  if ((ptree->shown_row_data == NULL && gwidget == NULL) ||
      (ptree->shown_row_data && gwidget == ptree->shown_row_data->gwidget))
    return gwidget != NULL;
  
  if (ptree->shown_row_data)
    {
      row_data = ptree->shown_row_data;
      ptree->shown_row_data = NULL;
      gle_ptree_row_data_update (row_data);
    }
  
  if (gwidget)
    {
      row_data = gle_ptree_row_data_from_gwidget (ptree, gwidget);
      
      if (row_data)
	{
	  GtkCTreeNode *cnode;
	  gint row;
	  
	  ptree->shown_row_data = row_data;
	  gle_ptree_row_data_update (row_data);
	  
	  cnode = GTK_CTREE_ROW (row_data->ct_node)->parent;
	  while (cnode)
	    {
	      gtk_ctree_expand (GTK_CTREE (ptree), cnode);
	      cnode = GTK_CTREE_ROW (cnode)->parent;
	    }
	  
	  row = g_list_position (GTK_CLIST (ptree)->row_list, (GList*) row_data->ct_node);
	  if (gtk_clist_row_is_visible (GTK_CLIST (ptree), row) != GTK_VISIBILITY_FULL)
	    gtk_ctree_node_moveto (GTK_CTREE (ptree),
				   row_data->ct_node,
				   -1, 0.5, 0);
	}
      else
	gwidget = NULL;
    }
  
  return gwidget != NULL;
}

GleGWidget*
gle_ptree_gwidget_from_node (GlePTree	   *ptree,
			     GtkCTreeNode   *ct_node)
{
  GlePTreeRowData *row_data;
  
  g_return_val_if_fail (ptree != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PTREE (ptree), NULL);
  g_return_val_if_fail (ct_node != NULL, NULL);
  
  row_data = gtk_ctree_node_get_row_data (GTK_CTREE (ptree), ct_node);
  if (row_data && row_data->ptree == ptree)
    return row_data->gwidget;
  
  return NULL;
}

GSList*
gle_ptree_selected_gwidgets (GlePTree *ptree)
{
  GList *list;
  GSList *selection;
  
  g_return_val_if_fail (ptree != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PTREE (ptree), NULL);
  
  selection = NULL;
  for (list = GTK_CLIST (ptree)->selection; list; list = list->next)
    {
      GleGWidget *gwidget;
      
      gwidget = gle_ptree_gwidget_from_node (ptree, list->data);
      selection = g_slist_prepend (selection, gwidget);
    }
  
  return selection;
}

static void
gle_ptree_connect_row_data (GlePTreeRowData *row_data)
{
  GtkObject *object;

  g_return_if_fail (row_data->is_connected == FALSE);
  
  row_data->is_connected = TRUE;

  object = GLE_GWIDGET_OBJECT (row_data->gwidget);
  
  gtk_signal_connect_object_after (object, "hide", gle_ptree_row_data_update, (GtkObject*) row_data);
  gtk_signal_connect_object_after (object, "show", gle_ptree_row_data_update, (GtkObject*) row_data);
  gtk_signal_connect_object_after (object, "state-changed", gle_ptree_row_data_update, (GtkObject*) row_data);
}

static void
gle_ptree_disconnect_row_data (GlePTreeRowData *row_data)
{
  GtkObject *object;
  
  g_return_if_fail (row_data->is_connected == TRUE);

  row_data->is_connected = FALSE;

  object = GLE_GWIDGET_OBJECT (row_data->gwidget);
  
  gtk_signal_disconnect_by_func (object, gle_ptree_row_data_update, row_data);
}

static void
gle_ptree_gcontainer_add (GleGContainer   *gcontainer,
			  GlePTreeRowData *row_data,
			  GleGWidget	  *gchild)
{
  gle_ptree_insert_gwidget (row_data->ptree, gchild);
}

static void
gle_ptree_gcontainer_remove (GleGContainer   *gcontainer,
			     GlePTreeRowData *container_row_data,
			     GleGWidget	     *gchild)
{
  GlePTreeRowData *row_data;

  row_data = gle_ptree_row_data_from_gwidget (container_row_data->ptree, gchild);
  if (row_data)
    gtk_ctree_remove_node (GTK_CTREE (container_row_data->ptree), row_data->ct_node);
}

static void
gle_ptree_append_gwidget (GleGWidget	     *gwidget,
			  GlePTreeAppendData *data)
{
  GlePTree *ptree;
  gchar *text[GLE_PTREE_N_COLS];
  GlePTreeRowData *row_data;
  guint i;
  
  ptree = data->ptree;
  
  if (GLE_GOBJECT_DESTROYED (gwidget))
    return;
  
  data->depth += 1;
  
  for (i = 0; i < GLE_PTREE_N_COLS; i++)
    text[i] = "x";
  
  text[GLE_PTREE_COL_OTYPE] = gtk_type_name (GLE_GOBJECT_OTYPE (gwidget));
  
  row_data = g_mem_chunk_alloc (gle_ptree_class->row_data_mem_chunk);
  row_data->ptree = ptree;
  row_data->gwidget = gwidget;
  row_data->is_connected = FALSE;
  row_data->ct_node = gtk_ctree_insert_node (GTK_CTREE (ptree),
					     data->parent,
					     NULL /* data->sibling */,
					     text, 0,
					     NULL, NULL, NULL, NULL,
					     !GLE_IS_GCONTAINER (gwidget),
					     data->depth <= GLE_RCVAL_PTREE_EXPAND_DEPTH);
  gtk_ctree_node_set_row_data_full (GTK_CTREE (ptree),
				    row_data->ct_node,
				    row_data,
				    (GtkDestroyNotify) gle_ptree_row_data_free);
  
  GLE_NOTIFIER_INSTALL_DATA (gwidget, "pre_destroy", gle_ptree_row_data_destroyed, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gwidget, "set_gname", gle_ptree_row_data_update, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gwidget, "associate", gle_ptree_row_data_update, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gwidget, "disassociate", gle_ptree_row_data_update, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gwidget, "associate", gle_ptree_connect_row_data, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gwidget, "pre_disassociate", gle_ptree_disconnect_row_data, row_data);
  if (GLE_IS_GCONTAINER (gwidget))
    {
      GLE_NOTIFIER_INSTALL (gwidget, "add_child", gle_ptree_gcontainer_add, row_data);
      GLE_NOTIFIER_INSTALL (gwidget, "remove_child", gle_ptree_gcontainer_remove, row_data);
    }
  if (GLE_GOBJECT_IS_INSTANTIATED (gwidget))
    gle_ptree_connect_row_data (row_data);
  
  if (GLE_IS_GCONTAINER (gwidget))
    {
      GtkCTreeNode *old_parent;
      GList *list;
      
      old_parent = data->parent;
      data->parent = row_data->ct_node;
      data->sibling = NULL;
      for (list = GLE_GCONTAINER (gwidget)->children; list; list = list->next)
	if (!GLE_GWIDGET_INTERNAL (list->data) && !GLE_GWIDGET_COMPOSITE (list->data))
	  gle_ptree_append_gwidget (list->data, data);
      for (list = GLE_GCONTAINER (gwidget)->children; list; list = list->next)
	if (GLE_GWIDGET_INTERNAL (list->data) || GLE_GWIDGET_COMPOSITE (list->data))
	  gle_ptree_append_gwidget (list->data, data);
      data->parent = old_parent;
    }
  
  data->sibling = row_data->ct_node;
  data->depth -= 1;

  gle_ptree_row_data_update (row_data);
}

static void
gle_ptree_row_data_update (GlePTreeRowData *row_data)
{
  if (row_data)
    {
      GlePTree *ptree;
      GleGWidget *gwidget;
      gchar *text[GLE_PTREE_N_COLS] = { NULL };
      guint i;
      gchar *s;
      
      ptree = row_data->ptree;
      gwidget = row_data->gwidget;
      
      if (GLE_GOBJECT_DESTROYED (gwidget))
	{
	  gle_ptree_row_data_destroyed (row_data);
	  return;
	}
      
      gtk_ctree_node_set_foreground (GTK_CTREE (ptree),
				     row_data->ct_node,
				     (ptree->shown_row_data == row_data ?
				      &GTK_WIDGET (ptree)->style->base[GTK_STATE_PRELIGHT] :
				      NULL));

      if (GLE_GWIDGET_COMPOSITE (gwidget) || GLE_GWIDGET_INTERNAL (gwidget))
	{
	  text[GLE_PTREE_COL_OTYPE] = g_strconcat ("_",
						   gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)),
						   NULL);
	  if (GLE_GWIDGET_COMPOSITE (gwidget))
	    {
	      if (GLE_GWIDGET_INTERNAL (gwidget))
		text[GLE_PTREE_COL_OTYPE][0] = '?';
	      else
		text[GLE_PTREE_COL_OTYPE][0] = '*';
	    }
	  else
	      text[GLE_PTREE_COL_OTYPE][0] = '#';
	}
      else
	text[GLE_PTREE_COL_OTYPE] = g_strdup (gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)));
      
      
      s = GLE_GOBJECT (gwidget)->gname;
      text[GLE_PTREE_COL_GNAME] = g_strconcat ("\"",
					       s ? s : "",
					       "\"",
					       NULL);
      
      if (GLE_GOBJECT_IS_INSTANTIATED (gwidget))
	text[GLE_PTREE_COL_NAME] = g_strconcat ("\"",
						(GLE_GWIDGET_WIDGET (gwidget)->name ?
						 GLE_GWIDGET_WIDGET (gwidget)->name : ""),
						"\"",
						NULL);
      
      if (GLE_GOBJECT_IS_INSTANTIATED (gwidget))
	{
	  switch (GTK_WIDGET_STATE (GLE_GWIDGET_WIDGET (gwidget)))
	    {
	    case GTK_STATE_ACTIVE:
	      text[GLE_PTREE_COL_STATE] = g_strdup ("(_) Active");
	      break;
	    case GTK_STATE_PRELIGHT:
	      text[GLE_PTREE_COL_STATE] = g_strdup ("(_) Prelight");
	      break;
	    case GTK_STATE_SELECTED:
	      text[GLE_PTREE_COL_STATE] = g_strdup ("(_) Selected");
	      break;
	    case GTK_STATE_INSENSITIVE:
	      text[GLE_PTREE_COL_STATE] = g_strdup ("(_) Insensitive");
	      break;
	    default:
	      text[GLE_PTREE_COL_STATE] = g_strdup ("(_) Normal");
	      break;
	    }
	  text[GLE_PTREE_COL_STATE][1] = GTK_WIDGET_VISIBLE (GLE_GWIDGET_WIDGET (gwidget)) ? 'V' : 'H';
	}
      else
	text[GLE_PTREE_COL_STATE] = g_strdup ("...empty...");
      
      for (i = GLE_PTREE_COL_OTYPE; i < GLE_PTREE_N_COLS; i++)
	{
	  if (GLE_RCVAL_PTREE_RESET_COLUMNS)
	    {
	      gint new_width;
	      
	      new_width = text[i] ? gdk_string_width (GTK_WIDGET (ptree)->style->font, text[i]) : 0;
	      if (GTK_CLIST (ptree)->column[i].width < new_width)
		gtk_clist_set_column_width (GTK_CLIST (ptree), i, new_width);
	    }
	  
	  gtk_ctree_node_set_text (GTK_CTREE (ptree), row_data->ct_node, i, text[i]);
	  
	  g_free (text[i]);
	}
    }
}

static void
gle_ptree_row_data_destroyed (GlePTreeRowData *row_data)
{
  GlePTree *ptree;
  GtkCTree *ctree;
  
  ptree = row_data->ptree;
  ctree = GTK_CTREE (ptree);
  
  gtk_clist_freeze (GTK_CLIST (ptree));
  
  gtk_ctree_unselect (ctree, row_data->ct_node);
  gtk_ctree_node_set_text (ctree, row_data->ct_node, GLE_PTREE_COL_STATE, "Destroyed");
  gtk_ctree_node_set_foreground (ctree, row_data->ct_node, &GTK_WIDGET (ptree)->style->fg[GTK_STATE_INSENSITIVE]);
  gtk_ctree_node_set_background (ctree, row_data->ct_node, &GTK_WIDGET (ptree)->style->bg[GTK_STATE_INSENSITIVE]);
  gtk_ctree_node_set_selectable (ctree, row_data->ct_node, FALSE);
  
  gtk_clist_thaw (GTK_CLIST (ptree));
  
  gle_ptree_row_data_free (row_data);
}

static void
gle_ptree_row_data_free (GlePTreeRowData *row_data)
{
  if (row_data)
    {
      GlePTree *ptree;
      
      ptree = row_data->ptree;
      
      if (row_data->is_connected)
	gle_ptree_disconnect_row_data (row_data);
      
      gtk_ctree_node_set_row_data (GTK_CTREE (ptree), row_data->ct_node, NULL);
      
      GLE_NOTIFIER_REMOVE_FD (row_data->gwidget, gle_ptree_row_data_destroyed, row_data);
      GLE_NOTIFIER_REMOVE_FD (row_data->gwidget, gle_ptree_row_data_update, row_data);
      GLE_NOTIFIER_REMOVE_FD (row_data->gwidget, gle_ptree_connect_row_data, row_data);
      GLE_NOTIFIER_REMOVE_FD (row_data->gwidget, gle_ptree_disconnect_row_data, row_data);
      if (GLE_IS_GCONTAINER (row_data->gwidget))
	{
	  GLE_NOTIFIER_REMOVE_FD (row_data->gwidget, gle_ptree_gcontainer_add, row_data);
	  GLE_NOTIFIER_REMOVE_FD (row_data->gwidget, gle_ptree_gcontainer_remove, row_data);
	}
      
      if (row_data->ptree->shown_row_data == row_data)
	row_data->ptree->shown_row_data = NULL;
      
      row_data->ptree = NULL;
      
      g_mem_chunk_free (gle_ptree_class->row_data_mem_chunk, row_data);
    }
}

static gint
gle_ptree_button_press_event (GtkWidget		*widget,
			      GdkEventButton	*event)
{
  GlePTree *ptree;
  gboolean handled;
  
  ptree = GLE_PTREE (widget);
  
  handled = FALSE;
  if (event->type == GDK_BUTTON_PRESS && event->button)
    {
      gint row;
      
      if (gle_clist_selection_info (GTK_CLIST (ptree),
				    event->window,
				    event->x, event->y,
				    &row, NULL))
	{
	  GlePTreeRowData *row_data;
	  
	  row_data = gtk_clist_get_row_data (GTK_CLIST (ptree), row);
	  if (row_data)
	    {
	      gint return_val = FALSE;
	      
	      gtk_signal_emit (GTK_OBJECT (ptree),
			       ptree_signals[SIGNAL_BUTTON_CLICK],
			       row_data->gwidget,
			       (guint) event->x_root,
			       (guint) event->y_root,
			       (guint) event->button,
			       (guint) event->time,
			       &return_val);
	      handled |= return_val != FALSE;
	    }
	}
    }
  if (!handled && GTK_WIDGET_CLASS (parent_class)->button_press_event)
    handled = GTK_WIDGET_CLASS (parent_class)->button_press_event (widget, event);
  
  gle_set_flash_widget (NULL);
  
  return handled;
}

static GleGObject*
gle_ptree_dnd_start (GtkWidget *widget,
		     gint       root_x,
		     gint       root_y)
{
  GlePTree *ptree;
  GtkCList *clist;
  gint x, y, row;
  
  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PTREE (widget), NULL);
  
  ptree = GLE_PTREE (widget);
  clist = GTK_CLIST (ptree);
  
  if (GTK_WIDGET_REALIZED (ptree) &&
      gdk_window_get_origin (clist->clist_window, &x, &y) &&
      gle_clist_selection_info (GTK_CLIST (ptree),
				clist->clist_window,
				root_x - x, root_y - y,
				&row, NULL))
    {
      GlePTreeRowData *row_data;
      
      row_data = gtk_clist_get_row_data (GTK_CLIST (ptree), row);
      if (row_data)
	return GLE_GOBJECT (row_data->gwidget);
    }
  
  return NULL;
}

static GleGObject*
gle_ptree_dnd_check (GtkWidget *widget,
		     GtkType	drag_type,
		     gint       root_x,
		     gint       root_y)
{
  GlePTree *ptree;
  GtkCList *clist;
  gint x, y, row;

  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PTREE (widget), NULL);
  
  ptree = GLE_PTREE (widget);
  clist = GTK_CLIST (ptree);

  if (GTK_WIDGET_REALIZED (ptree) &&
      gdk_window_get_origin (clist->clist_window, &x, &y) &&
      gle_clist_selection_info (GTK_CLIST (ptree),
				clist->clist_window,
				root_x - x, root_y - y,
				&row, NULL))
    {
      GlePTreeRowData *row_data;
      
      row_data = gtk_clist_get_row_data (clist, row);
      if (row_data)
	{
	  if (GLE_GOBJECT_IS_INSTANTIATED (row_data->gwidget))
	    {
	      GtkWidget *container;

	      if (GLE_IS_GCONTAINER (row_data->gwidget))
		container = GLE_GWIDGET_WIDGET (row_data->gwidget);
	      else
		container = GLE_GWIDGET_WIDGET (row_data->gwidget->parent);

	      while (container &&
		     !gtk_type_is_a (drag_type, gtk_container_child_type (GTK_CONTAINER (container))))
		container = container->parent;
	      
	      if (container)
		return gle_object_get_gobject (GTK_OBJECT (container));
	    }
	}
    }
  
  return NULL;
}
