/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"gleetils.h"
#include	"glepopups.h"
#include	"gleprivate.h"
#include	<gtk/gtkprivate.h>



/* --- variables --- */
static const gchar	*key_signal_list = GLE_PRIVATE_KEY (gle-signal-list);
static const gchar	*key_flag_list = GLE_PRIVATE_KEY (gle-flag-list);
static struct
{
  gchar *name;
  guint32 value;
} gle_widget_flags[] =
{
  /* flag section 0 */
  { "destroyed",	GTK_DESTROYED },
  { "floating",		GTK_FLOATING },
  { "reserved_1",	GTK_RESERVED_1 },
  { "reserved_2",	GTK_RESERVED_2 },
  /* flag section 1 */
  { "----------------",	0 },
  { "toplevel",		GTK_TOPLEVEL },
  { "no_window",	GTK_NO_WINDOW },
  { "realized",		GTK_REALIZED },
  { "mapped",		GTK_MAPPED },
  { "visible",		GTK_VISIBLE },
  { "sensitive",	GTK_SENSITIVE },
  { "parent_sensitive",	GTK_PARENT_SENSITIVE },
  { "can_focus",	GTK_CAN_FOCUS },
  { "has_focus",	GTK_HAS_FOCUS },
  { "can_default",	GTK_CAN_DEFAULT },
  { "has_default",	GTK_HAS_DEFAULT },
  { "has_grab",		GTK_HAS_GRAB },
  { "basic",		GTK_BASIC },
  /* flag section 2 */
  { "----------------",	0 },
  { "user_style",	PRIVATE_GTK_USER_STYLE },
  { "redraw_pending",	PRIVATE_GTK_REDRAW_PENDING },
  { "resize_pending",	PRIVATE_GTK_RESIZE_PENDING },
  { "resize_needed",	PRIVATE_GTK_RESIZE_NEEDED },
  { "leave_pending",	PRIVATE_GTK_LEAVE_PENDING },
  { "has_shape_mask",	PRIVATE_GTK_HAS_SHAPE_MASK },
  { "gtk_in_reparent",	PRIVATE_GTK_IN_REPARENT },
};
static guint	gle_n_widget_flag_names = sizeof (gle_widget_flags) / sizeof (gle_widget_flags[0]);


/* --- functions --- */
static void
gle_item_toggle_x_flag (GtkItem   *item,
			GdkEventButton *event,
			gpointer   func_data,
			gboolean   priv)
{
  guint f;
  GtkObject *object;
  GtkLabel *label;
  gchar *string;

  g_return_if_fail (item != NULL);
  g_return_if_fail (GTK_IS_ITEM (item));

  if (event->type != GDK_2BUTTON_PRESS)
    return;

  f = (guint) func_data;
  object = gtk_object_get_data (GTK_OBJECT (item), "gle-toggle-object");
  g_return_if_fail (object != NULL);

  label = GTK_LABEL (GTK_BIN (item)->child);

  if (priv)
    GTK_PRIVATE_FLAGS (object) ^= f;
  else
    GTK_OBJECT_FLAGS (object) ^= f;

  string = g_strdup (label->label);
  if (priv)
    string[0] = GTK_PRIVATE_FLAGS (object) & f ? '+' : '-';
  else
    string[0] = GTK_OBJECT_FLAGS (object) & f ? '+' : '-';
  gtk_label_set (label, string);
  g_free (string);
}

static void
gle_item_toggle_object_flag (GtkItem   *item,
			     GdkEventButton *event,
			     gpointer   func_data)
{
  gle_item_toggle_x_flag (item, event, func_data, 0);
}

static void
gle_item_toggle_private_flag (GtkItem   *item,
			      GdkEventButton *event,
			      gpointer   func_data)
{
  gle_item_toggle_x_flag (item, event, func_data, 1);
}

static void
gle_object_remove_key_flag_list (GtkObject *object)
{
  gtk_object_remove_data (object, key_flag_list);
}

GtkWindow*
gle_widget_popup_flag_list (GtkWidget        *widget)
{
  GtkWidget	*window;
  GList		*item_list;
  GtkWidget	*item;
  guint		 i;
  gint		 pos;
  gint		 flag_sect;
  gchar		*message;

  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

  if (gtk_object_get_data (GTK_OBJECT (widget), key_flag_list))
    return NULL;

  /* create list items
   */
  pos = -1;
  item_list = NULL;
  flag_sect = 0;
  for (i = 0; i < gle_n_widget_flag_names; i++)
    {
      gchar  *string;

      if (!gle_widget_flags[i].value)
	flag_sect++;
      
      string = g_strconcat (" ", gle_widget_flags[i].name, NULL);
      if (flag_sect > 1)
	string[0] = GTK_PRIVATE_FLAGS (widget) & gle_widget_flags[i].value ? '+' : '-';
      else
	string[0] = GTK_OBJECT_FLAGS (widget) & gle_widget_flags[i].value ? '+' : '-';
      item = gtk_list_item_new_with_label (string);
      g_free (string);
      /* gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &string);
       * gtk_object_set_user_data (GTK_OBJECT (item), string);
       */
      gtk_object_set_data (GTK_OBJECT (item), "gle-toggle-object", widget);
      gtk_widget_show (GTK_WIDGET (item));
      if (gle_widget_flags[i].value)
	gtk_signal_connect (GTK_OBJECT (item),
			    "button_press_event",
			    flag_sect > 1 ?
			    GTK_SIGNAL_FUNC (gle_item_toggle_private_flag) :
			    GTK_SIGNAL_FUNC (gle_item_toggle_object_flag),
			    (gpointer) gle_widget_flags[i].value);
      else
	gtk_widget_set_sensitive (item, FALSE);
      item_list = g_list_prepend (item_list, item);
    }
  item_list = g_list_reverse (item_list);
  if (pos != -1)
    pos = i - pos;

  /* pop up the flag list
   */
  message = g_strconcat (gtk_type_name (GTK_OBJECT (widget)->klass->type),
			 "\"",
			 widget->name ? widget->name : "",
			 "\"",
			 " flags:",
			 NULL);
  window = gle_popup_simple_list (NULL,
				  "GLE Flag List",
				  NULL,
				  NULL,
				  NULL,
				  FALSE,
				  item_list,
				  pos,
				  NULL,
				  FALSE,
				  message);
  g_free (message);
  gtk_signal_connect_object_while_alive (GTK_OBJECT (widget),
					 "destroy",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (window));

  gle_root_connect_object_life (GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (window),
			     "destroy",
			     GTK_SIGNAL_FUNC (gle_object_remove_key_flag_list),
			     GTK_OBJECT (widget));

  gtk_object_set_data (GTK_OBJECT (widget), key_flag_list, window);

  return GTK_WINDOW (window);
}

GtkWindow*
gle_widget_get_flag_list (GtkWidget      *widget)
{
  return (GtkWindow*) gtk_object_get_data ((GtkObject*) widget, key_flag_list);
}

static void
gle_object_remove_key_signal_list (GtkObject *object)
{
  gtk_object_remove_data (object, key_signal_list);
}

GtkWindow*
gle_object_popup_signal_list (GtkObject      *object)
{
  GtkObjectClass*klass;
  GtkWidget	*window;
  GList		*item_list;
  GtkWidget	*item;
  gint		 pos;
  gchar		*message;
  GtkType	ctype;
  guint		tmp_pos;
  gchar		*selected;

  g_return_val_if_fail (object != NULL, NULL);
  g_return_val_if_fail (GTK_IS_OBJECT (object), NULL);

  if (gtk_object_get_data (object, key_signal_list))
    return NULL;

  klass = object->klass;
  selected = "";

  /* create list items
   */
  pos = -1;
  item_list = NULL;
  ctype = klass->type;
  tmp_pos = 0;
  while (ctype)
    {
      GtkObjectClass *class;
      gint	      n;

      class = gtk_type_class (ctype);
	
      for (n = class->nsignals - 1; n >= 0; n--)
	{
	  GtkSignalQuery *query;
	  gchar *label;

	  query = gtk_signal_query (class->signals[n]);

	  if (!query)
	    {
	      if (n == 0)
		g_warning ("%s: signal slot 0 empty", gtk_type_name (ctype));
	      else
		g_warning ("%s: signal slot %d (after \"%s\") empty",
			   gtk_type_name (ctype),
			   n,
			   gtk_signal_name (class->signals[n - 1]));
	      continue;
	    }

	  if (g_str_equal ((gchar*) selected, (gchar*) query->signal_name) == 0)
	    pos = tmp_pos + n + 1;


	  label = g_strconcat (gtk_type_name (query->object_type),
			       "::",
			       query->signal_name,
			       "  (",
			       query->is_user_signal ? "U" : "-",
			       query->run_type & GTK_RUN_FIRST ? "F" : "-",
			       query->run_type == GTK_RUN_LAST ? "L" : "-",
			       ")  returns: ",
			       gtk_type_name (query->return_val),
			       NULL);
	  item = gtk_list_item_new_with_label (label);
	  g_free (label);
	  g_free (query);

	  gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &label);
	  gtk_object_set_user_data (GTK_OBJECT(item), label);
	  gtk_widget_show (GTK_WIDGET (item));
	  
	  item_list = g_list_prepend (item_list, item);
	}

      tmp_pos += class->nsignals;
      ctype = gtk_type_parent (ctype);
    }
  item_list = g_list_reverse (item_list);
  if (pos != -1)
    pos = tmp_pos - pos;

  /* pop up the signal_name list
   */
  message = g_strconcat (gtk_type_name (klass->type), " signals:", NULL);
  window = gle_popup_simple_list (NULL,
				  "GLE Signal List",
				  NULL,
				  NULL,
				  NULL,
				  FALSE,
				  item_list,
				  pos,
				  NULL,
				  TRUE,
				  message);
  g_free (message);
  gtk_signal_connect_object_while_alive (object,
					 "destroy",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (window));

  gle_root_connect_object_life (GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (window),
			     "destroy",
			     GTK_SIGNAL_FUNC (gle_object_remove_key_signal_list),
			     object);

  gtk_object_set_data (object, key_signal_list, window);

  return GTK_WINDOW (window);
}

GtkWindow*
gle_object_get_signal_list (GtkObject      *object)
{
  return (GtkWindow*) gtk_object_get_data (object, key_signal_list);
}
