/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"gleselector.h"



/* --- signals --- */
enum
{
  SIGNAL_CANDIDATE_CHECK,
  SIGNAL_CANDIDATE_SELECTED,
  SIGNAL_LAST
};
typedef	void	(*SignalCandidateCheck)		(GtkObject	*object,
						 GtkWidget	*new_candidate,
						 gint		*candidate_ok,
						 gpointer	 func_data);
typedef	void	(*SignalCandidateSelected)	(GtkObject	*object,
						 GtkWidget	*candidate,
						 gpointer	 func_data);


/* --- prototypes --- */
static void		gle_selector_class_init		(GleSelectorClass *class);
static void		gle_selector_init		(GleSelector	*selector);
static void		gle_selector_destroy		(GtkObject	*object);
static gint		gle_selector_event		(GtkWidget	*widget,
							 GdkEvent	*event);
static void		gle_selector_cleanup		(GleSelector	*selector);
static void		gle_selector_candidate_label_set(GleSelector	*selector,
							 GtkWidget	*candidate);


/* --- variables --- */
static GtkWindowClass	*parent_class = NULL;
static gint		 selector_signals[SIGNAL_LAST] = { 0 };
static const gchar	*pkey_eventmask = GLE_PRIVATE_KEY (gle-event-mask);


/* --- functions --- */
guint
gle_selector_get_type (void)
{
  static guint selector_type = 0;

  if (!selector_type)
    {
      GtkTypeInfo selector_info =
      {
	"GleSelector",
	sizeof (GleSelector),
	sizeof (GleSelectorClass),
	(GtkClassInitFunc) gle_selector_class_init,
	(GtkObjectInitFunc) gle_selector_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      selector_type = gtk_type_unique (gtk_window_get_type (), &selector_info);
    }

  return selector_type;
}

static void
gle_selector_marshal_candidate_check (GtkObject      *object,
				      GtkSignalFunc  func,
				      gpointer       func_data,
				      GtkArg         *args)
{
  SignalCandidateCheck sfunc = (SignalCandidateCheck) func;

  (* sfunc) (object, GTK_VALUE_WIDGET (args[0]), GTK_VALUE_POINTER (args[1]), func_data);
}

static void
gle_selector_marshal_candidate_selected (GtkObject      *object,
					 GtkSignalFunc  func,
					 gpointer       func_data,
					 GtkArg         *args)
{
  SignalCandidateSelected sfunc = (SignalCandidateSelected) func;

  (* sfunc) (object, GTK_VALUE_WIDGET (args[0]), func_data);
}

static void
gle_selector_class_init (GleSelectorClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;

  parent_class = gtk_type_class (gtk_window_get_type ());

  selector_signals[SIGNAL_CANDIDATE_CHECK] =
    gtk_signal_new ("candidate_check",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleSelectorClass, candidate_check),
		    gle_selector_marshal_candidate_check,
		    GTK_TYPE_NONE, 2,
		    GTK_TYPE_WIDGET,
		    GTK_TYPE_POINTER);
  selector_signals[SIGNAL_CANDIDATE_SELECTED] =
    gtk_signal_new ("candidate_selected",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleSelectorClass, candidate_selected),
		    gle_selector_marshal_candidate_selected,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_WIDGET);
  gtk_object_class_add_signals (object_class, selector_signals, SIGNAL_LAST);

  object_class->destroy = gle_selector_destroy;
  widget_class->event = gle_selector_event;

  class->candidate_check = NULL;
  class->candidate_selected = NULL;
}

static void
gle_selector_init (GleSelector *selector)
{
  GtkWidget *main_vbox;
  GtkWidget *label;
  GtkWidget *any;
  GtkWidget *button;

  selector->in_selection = FALSE;
  selector->candidate = NULL;
  selector->event_restore_list = NULL;
  
  gtk_widget_set (GTK_WIDGET (selector),
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::title", gtk_type_name (GTK_OBJECT_TYPE (selector)),
		  "GtkWindow::window_position", GTK_WIN_POS_CENTER,
		  "GtkObject::signal::delete_event", gtk_false, NULL,
		  "GtkWindow::allow_shrink", FALSE,
		  "GtkWindow::allow_grow", FALSE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 10,
		    "GtkWidget::parent", selector,
		    "GtkWidget::visible", TRUE,
		    NULL);
  label	= gtk_widget_new (gtk_label_get_type (),
			  "GtkLabel::label", "Select a window by clicking",
			  "GtkWidget::visible", TRUE,
			  NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), label, FALSE, FALSE, 10);
  any =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::shadow", GTK_SHADOW_ETCHED_OUT,
		    "GtkFrame::label_xalign", 0.5,
		    "GtkFrame::label", "Candidate",
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  selector->candidate_label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "<None>",
		    "GtkWidget::sensitive", FALSE,
		    "GtkWidget::parent", any,
		    "GtkWidget::visible", TRUE,
		    NULL);
  any	=
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Abort",
		    "GtkContainer::border_width", 5,
		    "GtkObject::object_signal::clicked", gle_selector_abort, selector,
		    "GtkWidget::can_default", TRUE,
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::has_default", TRUE,
		    NULL);
}

static void
gle_selector_destroy (GtkObject	*object)
{
  GleSelector *selector;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (object));

  selector = GLE_SELECTOR (object);

  if (selector->in_selection)
    gle_selector_abort (selector);

  gle_selector_reset (selector);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

void
gle_selector_abort (GleSelector	   *selector)
{
  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));

  if (selector->in_selection)
    {
      gle_selector_cleanup (selector);
      gle_selector_reset (selector);
    }
}

void
gle_selector_reset (GleSelector	   *selector)
{
  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));
  g_return_if_fail (selector->in_selection == FALSE);

  if (selector->candidate)
    {
      gtk_widget_unref (selector->candidate);
      selector->candidate = NULL;
      gle_selector_candidate_label_set (selector, NULL);
    }
}

GtkWidget*
gle_selector_make_selection (GleSelector    *selector)
{
  GtkWidget *result;
  
  g_return_val_if_fail (selector != NULL, NULL);
  g_return_val_if_fail (GLE_IS_SELECTOR (selector), NULL);
  g_return_val_if_fail (selector->in_selection == FALSE, NULL);

  gtk_widget_ref (selector);

  if (selector->candidate)
    gle_selector_reset (selector);

  gtk_container_need_resize (GTK_CONTAINER (selector));
  gtk_widget_show (GTK_WIDGET (selector));

  gtk_grab_add (GTK_WIDGET (selector));
  selector->in_selection = TRUE;
  gtk_main ();

  gtk_widget_hide (GTK_WIDGET (selector));

  if (selector->in_selection)
    {
      g_warning ("GleSelector: main gtk loop quited for unknown reason, aborting...");
      gle_selector_abort (selector);
    }

  result = selector->candidate;
  if (!GTK_OBJECT_DESTROYED (selector))
    gtk_signal_emit (GTK_OBJECT (selector), selector_signals[SIGNAL_CANDIDATE_SELECTED], result);
		     
  gtk_widget_unref (selector);

  return result;
}

static void
gle_selector_cleanup (GleSelector *selector)
{
  GSList *list;

  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));
  g_return_if_fail (selector->in_selection == TRUE);
  
  gtk_grab_remove (GTK_WIDGET (selector));
  selector->in_selection = FALSE;
  gtk_main_quit ();
  
  list = selector->event_restore_list;
  while (list)
    {
      GtkWidget *window;
      GdkEventMask *event_mask;
      
      window = list->data;
      event_mask = gtk_object_get_data (GTK_OBJECT (window), pkey_eventmask);
      if (event_mask)
	{
	  gtk_object_set_data (GTK_OBJECT (window), pkey_eventmask, NULL);
	  gdk_window_set_events (window->window, *event_mask);
	  g_free (event_mask);
	}
      gtk_widget_unref (window);
      list = list->next;
    }
  g_slist_free (selector->event_restore_list);
  selector->event_restore_list = NULL;
}

static void
gle_selector_candidate_label_set (GleSelector	*selector,
				  GtkWidget	*candidate)
{
  GtkLabel *label;
  gchar *string;

  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));

  label = GTK_LABEL (selector->candidate_label);

  if (!candidate)
    string = "<None>";
  else
    {
      if (GTK_CHECK_TYPE (candidate, gtk_window_get_type ()))
	{
	  if (GTK_WINDOW (candidate)->title)
	    string = GTK_WINDOW (candidate)->title;
	  else
	    string = "<NULL> Title";
	}
      else if (GTK_CHECK_TYPE (candidate, gtk_menu_get_type ()))
	string = "GtkMenu";
      else
	string = "<Unknown Candidate>";
    }
  if (!g_str_equal (label->label, string))
    gtk_label_set (label, string);
  gtk_widget_set_sensitive (GTK_WIDGET (label), candidate && candidate == selector->candidate);
}

static gint
gle_selector_check_candidate (GleSelector    *selector,
			      GtkWidget	     *new_candidate)
{
  gint candidate_ok;

  candidate_ok = new_candidate && new_candidate != GTK_WIDGET (selector);
  if (candidate_ok)
    gtk_signal_emit (GTK_OBJECT (selector),
		     selector_signals[SIGNAL_CANDIDATE_CHECK],
		     new_candidate,
		     &candidate_ok);
  if (selector->candidate)
    gtk_widget_unref (selector->candidate);
  if (candidate_ok)
    {
      selector->candidate = new_candidate;
      gtk_widget_ref (selector->candidate);
    }
  else
    selector->candidate = NULL;

  gle_selector_candidate_label_set (selector, new_candidate);
  
  return candidate_ok;
}

static gint
gle_selector_event (GtkWidget	       *widget,
		    GdkEvent	       *event)
{
  GleSelector *selector;
  GtkWidget *event_window;
  gboolean event_handled;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_SELECTOR (widget), FALSE);

  selector = GLE_SELECTOR (widget);
  if (!selector->in_selection)
    return FALSE;

  event_window = gtk_get_event_widget (event);

  if (event_window)
    while (event_window->parent)
      event_window = event_window->parent;

  event_handled = FALSE;
  switch (event->type)
    {
      GdkEventMask *event_mask;
      GdkWindow *pointer_window;
      
    case  GDK_ENTER_NOTIFY:
      if (gle_selector_check_candidate (selector, event_window))
	{
	  event_mask = gtk_object_get_data (GTK_OBJECT (event_window), pkey_eventmask);
	  if (!event_mask)
	    {
	      event_mask = g_new (GdkEventMask, 1);
	      *event_mask = gdk_window_get_events (event_window->window);
	      gtk_object_set_data (GTK_OBJECT (event_window), pkey_eventmask, event_mask);
	      gdk_window_set_events (event_window->window,
				     *event_mask |
				     GDK_BUTTON_RELEASE_MASK |
				     GDK_ENTER_NOTIFY_MASK |
				     GDK_LEAVE_NOTIFY_MASK);
	      selector->event_restore_list =
		g_slist_prepend (selector->event_restore_list, event_window);
	      gtk_widget_ref (event_window);
	    }
	}
      event_handled = TRUE;
      break;
      
    case  GDK_LEAVE_NOTIFY:
      if (event_window)
	pointer_window = gdk_window_get_pointer (event_window->window, NULL, NULL, NULL);
      else
	pointer_window = NULL;
      event_window = NULL;
      if (pointer_window)
	gdk_window_get_user_data (pointer_window, (gpointer*) &event_window);
      if (event_window)
	{
	  while (event_window->parent)
	    event_window = event_window->parent;
	}
      gle_selector_check_candidate (selector, event_window);
      event_handled = TRUE;
      break;
      
    case  GDK_BUTTON_RELEASE:
      if (event->button.button == 1 &&
	  gle_selector_check_candidate (selector, event_window))
	{
	  gle_selector_cleanup (selector);
	  gtk_signal_emit (GTK_OBJECT (selector),
			   selector_signals[SIGNAL_CANDIDATE_SELECTED],
			   selector->candidate);
	}
      event_handled = TRUE;
      break;

    default:
      break;
    }

  return event_handled;
}
