/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"gleargs.h"
#include	<string.h>
#include	<stdio.h>


/* --- variables --- */
static GMemChunk	*garg_mem_chunk = NULL;
static GHashTable	*gle_arg_ht = NULL;

/* --- functions --- */
GleGArg*
gle_garg_new (const GtkArg	*dfl_arg)
{
  GtkType	widget_type;
  GleGArg	*garg;

  g_return_val_if_fail (dfl_arg != NULL, NULL);
  g_return_val_if_fail (dfl_arg->type != GTK_TYPE_INVALID, NULL);
  g_return_val_if_fail (dfl_arg->type != GTK_TYPE_NONE, NULL);

  widget_type = gtk_type_from_name (dfl_arg->name);
  g_return_val_if_fail (gtk_type_is_a (widget_type, gtk_widget_get_type ()), NULL);

  if (!garg_mem_chunk)
    garg_mem_chunk = g_mem_chunk_new ("GLE Arguments mem chunk",
				      sizeof (GleGArg),
				      sizeof (GleGArg) * GLE_GARGS_PREALLOC,
				      G_ALLOC_AND_FREE);

  garg = g_chunk_new (GleGArg, garg_mem_chunk);
  garg->widget_type = widget_type;
  garg->arg_name = dfl_arg->name;
  garg->default_arg = dfl_arg;
  garg->saved_arg.type = GTK_TYPE_INVALID;
  garg->saved_arg.name = (gchar *) garg->arg_name;
  memset (&garg->saved_arg.d, 0, sizeof (garg->saved_arg.d));
  garg->widget_arg.type = GTK_TYPE_INVALID;
  garg->widget_arg.name = (gchar *) garg->arg_name;
  memset (&garg->widget_arg.d, 0, sizeof (garg->widget_arg.d));

  return garg;
}

void
gle_arg_reset (GtkArg *arg)
{
  g_return_if_fail (arg != NULL);

  if (arg->type != GTK_TYPE_INVALID)
    {
      if (arg->type == GTK_TYPE_STRING)
	g_free (arg->d.string_data);
      arg->type = GTK_TYPE_INVALID;
    }
}

void
gle_garg_reset (GleGArg	*garg)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (GLE_IS_GARG (garg));

  gle_arg_reset (&garg->saved_arg);
  gle_arg_reset (&garg->widget_arg);
}

void
gle_garg_init (GleGArg        *garg,
	       GtkWidget      *widget)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (GLE_IS_GARG (garg));
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_WIDGET (widget));

  gle_garg_reset (garg);

  gtk_widget_get (widget, &garg->saved_arg);
  g_memmove (&garg->widget_arg, &garg->saved_arg, sizeof (garg->saved_arg));
  if (garg->widget_arg.type == GTK_TYPE_STRING)
    garg->widget_arg.d.string_data = g_strdup (garg->saved_arg.d.string_data);
}

void
gle_garg_free (GleGArg*        garg)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (GLE_IS_GARG (garg));

  if (garg->saved_arg.type == GTK_TYPE_STRING)
    g_free (garg->saved_arg.d.string_data);
  if (garg->widget_arg.type == GTK_TYPE_STRING)
    g_free (garg->widget_arg.d.string_data);
  g_mem_chunk_free (garg_mem_chunk, garg);
}

const GtkArg*
gle_arg_get_default (const gchar    *arg_name)
{
  GtkArg *arg;
  
  g_return_val_if_fail (arg_name != NULL, NULL);
  
  if (!gle_arg_ht)
    {
      gle_arg_ht = g_hash_table_new (g_str_hash,
				     g_str_equal);
      arg = NULL;
    }
  else
    arg = g_hash_table_lookup (gle_arg_ht, (gpointer) arg_name);
  
  if (!arg)
    {
      GtkType		widget_type;
      GtkWidget		*widget;
      GtkArg		*args;
      guint		n_args;

      widget_type = gle_type_from_name (arg_name, NULL);
      g_return_val_if_fail (gtk_type_is_a (widget_type, gtk_widget_get_type ()), NULL);

      widget = gtk_widget_new (widget_type, NULL);
      n_args = 0;
      args = gtk_object_query_args (GTK_OBJECT_TYPE (widget), &n_args);
      gtk_widget_getv (widget, n_args, args);
      gtk_widget_destroy (widget);

      for (; n_args > 0; n_args--)
	if (args[n_args - 1].type)
	  g_hash_table_insert (gle_arg_ht, args[n_args - 1].name, &args[n_args - 1]);

      arg = g_hash_table_lookup (gle_arg_ht, (gpointer) arg_name);
    }

  return arg;
}

void
gle_arg_set_from_string (GtkArg         *arg,
			 const gchar    *value_string)
{
  gchar *string;

  g_return_if_fail (arg != NULL);
  g_return_if_fail (value_string != NULL);

  string = g_strdup (value_string);

  switch (arg->type)
    {
    case  GTK_TYPE_INVALID:
      g_warning ("gle_arg_set_from_string() used with invalid type");
      break;
    case  GTK_TYPE_CHAR:
      if (strlen (string) > 1)
	{
	  arg->d.int_data = 0;
	  sscanf (string, "%d", &arg->d.int_data);
	  arg->d.char_data = arg->d.int_data;
	}
      else
	arg->d.char_data = string[0];
      break;
    case  GTK_TYPE_BOOL:
      g_strdown (string);
      if ((string[0] > '0' && string[0] <= '9') ||
	  g_str_equal (string, "true") ||
	  g_str_equal (string, "yes"))
	arg->d.bool_data = TRUE;
      else
	arg->d.bool_data = FALSE;
      break;
    case  GTK_TYPE_INT:
      arg->d.int_data = 0;
      sscanf (string, "%d", &arg->d.int_data);
      break;
    case  GTK_TYPE_UINT:
      arg->d.uint_data = 0;
      sscanf (string, "%u", &arg->d.int_data);
      break;
    case  GTK_TYPE_LONG:
      arg->d.long_data = 0;
      sscanf (string, "%ld", &arg->d.long_data);
      break;
    case  GTK_TYPE_ULONG:
      arg->d.ulong_data = 0;
      sscanf (string, "%lu", &arg->d.ulong_data);
      break;
    case  GTK_TYPE_FLOAT:
      arg->d.float_data = 0;
      sscanf (string, "%f", &arg->d.float_data);
      break;
    case  GTK_TYPE_DOUBLE:
      arg->d.double_data = 0;
      sscanf (string, "%lf", &arg->d.double_data);
      break;
    case  GTK_TYPE_STRING:
      arg->d.string_data = g_strdup (value_string);
      break;
    default:
      g_warning ("gle_arg_set_from_string() used with type <%s>", gtk_type_name (arg->type));
      break;
    }
  g_free (string);
}
