/***************************************************************************

	TITLE:	ls_writeav.c
	
----------------------------------------------------------------------------

	FUNCTION:	Writes out time history data in Agile-VU flight format

----------------------------------------------------------------------------

	MODULE STATUS:	Developmental

----------------------------------------------------------------------------

	GENEALOGY:	Written 921230 by Bruce Jackson (see Mod History)

----------------------------------------------------------------------------

	DESIGNED BY:	EBJ
	
	CODED BY:	EBJ
	
	MAINTAINED BY:	EBJ

----------------------------------------------------------------------------

	MODIFICATION HISTORY:
	
	DATE	PURPOSE						BY
	
	920507	Converted from Trajectory Viewer		EBJ
	920806	Converted from ASC2_to_PO16; ported to IRIS	EBJ
	920810	Incorporated single file argument		EBJ
	921027  Modified and incorporated into LaRCSim code	EBJ
	930915	Modified to correct runway start time for long runs	EBJ
	931220	Cleaned up the time slice array access		EBJ
	940111	Fixed include files; was ls_eom.h; also changed
		DATA to SCALAR types.				EBJ
	940509	Changed so wrapped tapes are handled properly; also
		obey the output interpolation flag "write_spacing".	EBJ

	CURRENT RCS HEADER:

$Header: /aces/larcsim/dev/RCS/ls_writeav.c,v 1.10 1995/04/07 01:44:34 bjax Exp $
$Log: ls_writeav.c,v $
 * Revision 1.10  1995/04/07  01:44:34  bjax
 * Added logic to avoid endless loop if wrapped and Tape->Last == Tape->Length.
 *
 * Revision 1.9  1994/05/11  19:47:11  bjax
 * Added support for multiple runs; unfortunately, the runway has to
 * be drawn last, which means it covers up the aircraft on top of it.
 * Talk about priority problems!
 *
 * Revision 1.8  1994/05/09  21:20:26  bjax
 * Added support for wrapped tape buffer and output interpolation.
 *
 * Revision 1.7  1994/01/11  19:03:12  bjax
 * Fixed include files; changed DATA to SCALAR type.
 *
 * Revision 1.6  1993/12/20  16:48:32  bjax
 * Cleaned up the time slice array access method.  EBJ
 *
 * Revision 1.4  1993/09/15  20:42:37  bjax
 * Modified code to put correct start time on 'runway' vehicle time stamp.
 *
 * Revision 1.3  1993/08/03  19:59:28  bjax
 * Modified to use current state variables being stored in Tape.  EBJ
 *
 * Revision 1.2  1993/06/08  10:31:35  bjax
 * Changed runway heading from -30 to 0 deg.  EBJ
 *
 * Revision 1.1  92/12/30  13:20:35  bjax
 * Initial revision
 * 
 * Revision 1.3  93/12/31  10:34:11  bjax
 * Added $Log marker as well.
 * 

----------------------------------------------------------------------------

	REFERENCES:

----------------------------------------------------------------------------

	CALLED BY:

----------------------------------------------------------------------------

	CALLS TO:

----------------------------------------------------------------------------

	INPUTS:

----------------------------------------------------------------------------

	OUTPUTS:

--------------------------------------------------------------------------*/


/*	================
	 include files
	================	*/

#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include "ls_types.h"
#include "ls_tape.h"
#include "ls_sim_control.h"


/*	=============
	 definitions
	=============	*/
#define MAX_STRING 255
#define	NIL_POINTER 0L

/* temporary defs */ 
#define IALT 15
#define IPSI 8
#define IPHI 6
#define ITHETA 7
#define ISX 13
#define ISY 14


extern TAPE *Tape;
extern char *progname;

av_write_header( FILE *fp )
  {
	static int init = 0;
	static int header_num = 0;
	
	if (init == 0 )  /* should only have one source line */
	    {
		init = -1;
    		fprintf(fp, "Source:	'%s ACES simulation'\n", progname);		    
	    }
	
	header_num++;
	
/* Write header info */

	fprintf(fp, "Type:	%s\n", progname);
	fprintf(fp, "Role:	Blue\n");
	fprintf(fp, "Name:	Run%03d\n", header_num);
	fprintf(fp, "Parameters:\n");
	fprintf(fp, "XCG        'X Position'      ft          0  -50000.00   50000.00   25000.00\n");
	fprintf(fp, "YCG        'Y Position'      ft          0  -50000.00   50000.00   25000.00\n");
	fprintf(fp, "ZCG        'Z Position'      ft          0       0.00  -30000.00  -10000.00\n");
	fprintf(fp, "PHIB       'Roll Angle'      deg         1    -180.00     180.00      45.00\n");
	fprintf(fp, "THETAB     'Pitch Angle'     deg         1     -90.00      90.00      45.00\n");
	fprintf(fp, "PSIB       'Heading Angle'   deg         1    -180.00     180.00      45.00\n");
  }
  
  
av_write_rwy_object( FILE *fp, SCALAR t_min, SCALAR t_max )
  {
      	fprintf(fp, "Type:	Rwy\n");
	fprintf(fp, "Role:	Blue\n");
	fprintf(fp, "Name:	Runway\n");
	fprintf(fp, "Parameters:\n");
	fprintf(fp, "XCG        'X Position'      ft          0  -50000.00   50000.00   25000.00\n");
	fprintf(fp, "YCG        'Y Position'      ft          0  -50000.00   50000.00   25000.00\n");
	fprintf(fp, "ZCG        'Z Position'      ft          0       0.00  -30000.00  -10000.00\n");
	fprintf(fp, "PHIB       'Roll Angle'      deg         1    -180.00     180.00      45.00\n");
	fprintf(fp, "THETAB     'Pitch Angle'     deg         1     -90.00      90.00      45.00\n");
	fprintf(fp, "PSIB       'Heading Angle'   deg         1    -180.00     180.00      45.00\n");
	fprintf(fp, "Time:	%f\n", t_min);
	fprintf(fp, "0	0	0	0.000000	0	  0.000000\n");
	fprintf(fp, "Time:	%f\n", t_max);
	fprintf(fp, "0	0	0	0.000000	0	  0.000000\n");
  }

			    /*---------------+
===========================>|   ls_writeav   |<===============================
			    +---------------*/
							
void ls_writeav( char *file_name)

{
	int	wrapped = 1;
	int 	i, j;
	FILE	*fp;
	SCALAR	xic, yic, zic;
	SCALAR	t_min, t_max, t_last;
	
/* Protect against using bad Tape */

	if (Tape == NULL) return;

	fp = fopen(file_name, "w");
	
	av_write_header( fp );
		
	i = Tape->First;
	
	if (Tape->First < Tape->Last) wrapped = 0;
	
	t_min = Tape->T_Stamp[i];
	t_max = Tape->T_Stamp[i];
	t_last = Tape->T_Stamp[i];

	while( wrapped || (i <= Tape->Last) )
	{
	    if (Tape->T_Stamp[i] < t_last) av_write_header( fp );
	    if (Tape->T_Stamp[i] < t_min ) t_min = Tape->T_Stamp[i];
	    if (Tape->T_Stamp[i] > t_max ) t_max = Tape->T_Stamp[i];
	    t_last = Tape->T_Stamp[i];
	    fprintf(fp, "Time:\t%f\n", Tape->T_Stamp[i] );
	    fprintf(fp, "%f\t%f\t%f\t", 
				    Tape->Chan[ ISX     ]->Data[ i ],
				    Tape->Chan[ ISY     ]->Data[ i ],
				   -Tape->Chan[ IALT    ]->Data[ i ]);	
	    fprintf(fp, "%f\t%f\t%f\n",
			    57.3*Tape->Chan[ IPHI    ]->Data[ i ],
			    57.3*Tape->Chan[ ITHETA  ]->Data[ i ],
			    57.3*Tape->Chan[ IPSI    ]->Data[ i ]);	
	    
	    i = i + sim_control_.write_spacing;
	    if ( i >= Tape->Length ) 
		    if ( wrapped )
			{
			    i = i - Tape->Length;
			    wrapped = 0;
			}
		    else
			{
			    i = Tape->Last+1;
			}
	}
	
	av_write_rwy_object( fp, t_min, t_max );
	
	fclose( fp );
}
