/***************************************************************************

	TITLE:		ls_sym
	
----------------------------------------------------------------------------

	FUNCTION:	Symbol table support routines for LaRCsim

----------------------------------------------------------------------------

	MODULE STATUS:	developmental

----------------------------------------------------------------------------

	GENEALOGY:	Created 930629 by E. B. Jackson

----------------------------------------------------------------------------

	DESIGNED BY:	E. B. Jackson
	
	CODED BY:	ditto
	
	MAINTAINED BY:	

----------------------------------------------------------------------------

	MODIFICATION HISTORY:
	
	DATE	PURPOSE						BY

	940112	Restructured this routine to be more readable;
		added support for structure elements		EBJ

	940505  Modified to use ELF structures and functions;
		had to rewrite most symbol table routines as interim
		between COFF and ELF, since older "ldfcn" routines
		will not be supported after IRIX 5.2		EBJ

	950306	Added routines ls_get_sym_val() and ls_set_sym_val()
	
	CURRENT RCS HEADER:

$Header: /aces/larcsim/dev/RCS/ls_sym.c,v 2.7 1995/03/06 18:44:07 bjax Stab $
$Log: ls_sym.c,v $
 * Revision 2.7  1995/03/06  18:44:07  bjax
 * Added ls_get_sym_val and ls_set_sym_val() routines.
 *
 * Revision 2.6  1995/02/27  19:54:51  bjax
 * Added utility routines: ls_print_findsym_error(), ls_get_double(),
 * ls_set_double(). EBJ
 *
 * Revision 2.5  1994/05/17  15:07:40  bjax
 * Corrected so that full name to directory and file is used
 * to open symbol table, so that sims can be run from another
 * default directory.
 *
 * Revision 2.4  1994/05/11  16:25:29  bjax
 * Correct problem with bounds error checking on dimensioned variables
 * that were Typedefs.  Increased the allowable number of dimensions
 * to six from three. EBJ
 *
 * Revision 2.3  1994/05/06  20:19:30  bjax
 * More or less complete set of data types now supported.
 *
 * Revision 1.2  1993/07/30  17:37:42  bjax
 * Corrected logic to skip over unwanted procedures.  EBJ
 *
 * Revision 1.1  1993/07/27  23:43:21  bjax
 * Initial revision
 *

----------------------------------------------------------------------------

	REFERENCES:

----------------------------------------------------------------------------

	CALLED BY:

----------------------------------------------------------------------------

	CALLS TO:

----------------------------------------------------------------------------

	INPUTS:

----------------------------------------------------------------------------

	OUTPUTS:

--------------------------------------------------------------------------*/
#include <libelf.h>
#include <syms.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include "ls_sym.h"

	/* local definitions */

#define FAILURE 0
#define SUCCESS -1
#define FALSE 0
#define TRUE -1

	/* macro functions to make code a little easier to read */

#define ISTRUCT(x) ((x->ti.bt == btStruct) || (x->ti.bt == btUnion))
#define ISTYPEDEF(x) (AUX( x.itaux )->ti.bt == btTypedef)
#define SYMEXIT(x) {free(tokenbuf); if(!index_list) free(index_list); return x;}
#define LOCALSYM(x) ((SYMR *) (hdrr_base + hdrr->cbSymOffset) + x)
#define EXTRNSYM(x) ((EXTR *) (hdrr_base + hdrr->cbExtOffset) + x)
#define FD(x)       ((FDR *)  (hdrr_base + hdrr->cbFdOffset ) + x)
#define AUX(x)	    ((AUXU *) (hdrr_base + hdrr->cbAuxOffset) + x)
#define SS(x)  ((char *)(hdrr_base+hdrr->cbSsOffset)+FD(x.ifd)->issBase+x.psymr->iss)
#define ESS(x) ((char *)(hdrr_base+hdrr->cbSsExtOffset)+x.psymr->iss)

	/*
	 * special data structure typedef -- it combines
	 * most information about each symbol. 
	 */

typedef struct
{
    short ext_sym;	/* flag to indicate it is located in ext syms */
    short ifd;		/* file descriptor index */
    long  isym;		/* symbol index */
    long  idaux;	/* abs. index to aux with array dims; else 0 */
    long  itaux;	/* abs. index into aux space for type aux    */
    pSYMR psymr;	/* local ptr to symbol */
} lsSYM;

	/* the following variable has Global scope */

extern char *fullname;

	/* the following variables have File scope */

static long	symmax, symmaxlocal, symmaxextern;
static long	i, end_of_proc;
static int	module_found, symbol_found;
static unsigned long hdrr_base;	/* diff between section addr and offsets */
static HDRR	*hdrr;
static char 	*namep;
static lsSYM  	symbol;


/*	
 *		===== symtbread =====	
 * 
 * Given the absolute index of a symbol, this routine populates
 * the fields of the lsSYM data structure pointed to by the second
 * argument based on debugger symbol information
 */

static int symtbread( long j, lsSYM *mySym )

{
    pEXTR ext;
    FDR	  *fd;
    short i;

    /* return FAILURE if out of bounds */

    if (j < 0) return FAILURE;
    if (j > symmaxextern + symmaxlocal) return FAILURE;

    /* point to proper symbol */

    mySym->isym = j;
    if (j <= symmaxlocal)
	{						/* local symbol */
	    mySym->ext_sym = FALSE;
	    mySym->psymr = LOCALSYM(j);
	    for ( i = 0; i < hdrr->ifdMax; i++)
		{
		    fd = FD(i);
		    if (fd->isymBase > j) break;
		}
	    mySym->ifd = i-1;
	}
    else						/* global symbol */
	{
	    mySym->ext_sym = TRUE;
	    ext = EXTRNSYM(j - symmaxlocal);
	    mySym->ifd = ext->ifd;
	    mySym->psymr = &ext->asym;
	}

    mySym->itaux = FD(mySym->ifd)->iauxBase + mySym->psymr->index;
    mySym->idaux = 0;

    return SUCCESS;

}	/* end of symtbread */


/*		
 *		===== symgetname =====
 *
 *
 * Given an lsSYM symbol record, this routine returns a pointer
 * to a string containing the name of the symbol.
 */

static char *symgetname( lsSYM sym )
{
    if (sym.ext_sym == 0) 
	return SS(sym);
    else
	return ESS(sym);
}

/*
 *		===== loadSymbols =====
 *
 * This routine reads in the debugger symbol table section from the
 * executable file pointed to by global string progname. It returns
 * SYM_OPEN_ERR if the file can't be opened, or SYM_NO_SYMS if it
 * doesn't appear to have debugging information. It also sets the file
 * scope variables symmaxlocal and symmaxextern and calculates the
 * appropriate value of hdrr_base.
 */

static int loadSymbols( void )	/* open and read the symbol table */
{
    int 	fildes;
    Elf 	*elf;		/* ELF file pointer 	*/
    Elf_Scn 	*scn;		/* ELF section		*/
    Elf32_Shdr 	*shdr;		/* ELF section header	*/
    Elf_Data	*data;		/* ELF data member	*/

    if (elf_version(EV_CURRENT) == EV_NONE) return SYM_OPEN_ERR;
    fildes = open(fullname, O_RDONLY);
    if ((elf = elf_begin(fildes, ELF_C_READ, (Elf *)0)) == 0)
	return SYM_OPEN_ERR;
    scn = (Elf_Scn *)0;
    do
      {
	if ((scn = elf_nextscn( elf, scn )) == 0)
	    return SYM_NO_SYMS;
	if ((shdr = elf32_getshdr(scn)) == 0)
	    return SYM_NO_SYMS;
      } 
    while (shdr->sh_type != SHT_MIPS_DEBUG);	/* special MIPS section */

    data = (Elf_Data *)0;
    if (   ((data = elf_getdata( scn, data )) == 0 )
	|| ( data->d_size == 0 )
	|| ( data->d_buf == 0 ) )
	return SYM_NO_SYMS;

    hdrr = (pHDRR) data->d_buf;	/* save pointer to symbolic header */
    hdrr_base =  (off_t) hdrr - shdr->sh_offset;

/*    elf_end( elf );	/* free up elf descriptor -deleted 5/11 EBJ;
			   it appears to deallocate the sym table. */
    close( fildes );	/* close open file descriptor */

    symmaxlocal  = hdrr->isymMax;
    symmaxextern = hdrr->iextMax;

    return SYM_OK;
}


/*
 *		===== lookForModule =====
 *
 * This routine searches the debugger symbols table, starting at
 * the index location pointed to by file variable i, for a symbol
 * whose associated string matches the string pointed to by modname.
 * If successful, it returns the absolute symbol index of the found
 * module.
 */

static int lookForModule( const char *modname )

{

 AUXU *paux;

 while( i < symmax)
  {
    if (FAILURE == symtbread(i, &symbol)) return SYM_UNEXPECTED_ERR;
    namep = symgetname( symbol );
    if ( namep == NULL ) return SYM_UNEXPECTED_ERR;
    
    if ( symbol.psymr->st == stProc )     /* beginning of a procedure */
      {
	paux = AUX( symbol.itaux ); /* get isymMac */
	if ( paux == auxNil ) return SYM_UNEXPECTED_ERR;

	end_of_proc = paux->isym + FD(symbol.ifd)->isymBase - 1; 
	/* should point to one before next proc */
	
	module_found = !strcmp(namep, modname); /* returns 0 if equal */
	if( !module_found )
	  {
	    if ( i < symmaxlocal )
	      i = end_of_proc;  /* skip around procedure for speed */
	  }
	else
	  return SYM_OK;	/* module found successfully */
      }
/*    else */

	/* here if module not found and symbol is not stProc type
	   - do nothing; fall through and let counter increment */

    i++;	/* increment symbol index */
  }
  return SYM_MOD_NOT_FOUND;
}



/* 
 *		===== followRFD =====
 *
 *   This routine follows the rfd trail - that is, it takes file
 *   indirection information from the aux entries for a given symbol
 *   and returns a pointer to another symbol entry to which the original
 *   symbol refers - for example, a global symbol entry to a specific
 *   definition, or a typedef to its definition.
 *
 *   The argument i is an absolute index to a symbol contained
 *   in the symbol table; as is the return value (which represents the
 *   indirect referenced symbol).
 */


static long followRFD( long i )

{
  lsSYM	symbol;
  AUXU  *paux;
  long  iss, rfdbase;
  short irfd, newifd;
  RFDT  *pfd;

  if (FAILURE == symtbread(i, &symbol)) 
      return 0;
  
  paux = AUX( symbol.itaux+1 ); 		/* get iss */
  if ( paux->rndx.rfd != 4095 ) return 0; /* unexpected result */

  iss = paux->rndx.index;	   	/* and save it */

  paux = AUX( symbol.itaux+2 ); 		/* get rfd index */
  irfd = paux->isym;  /* this seems undocumented, but works */


  if (irfd < 0 || irfd > FD(symbol.ifd)->crfd)
      return SYM_UNEXPECTED_ERR;	/* bounds check */

  rfdbase = FD(symbol.ifd)->rfdBase;	/* get base of rfd entries */

  pfd = (RFDT *) (hdrr_base + hdrr->cbRfdOffset) + 
      rfdbase + irfd;  			/* point to new fd */

  newifd = *pfd;			/* dereference pointer */

  return iss + FD(newifd)->isymBase;	/* and get new isym */

}



/*
 *		===== lookForSym =====
 *
 *   This routine searches through symbol table, starting at present
 *   location pointed to by i, up to the end of the procedure (pointed
 *   to by end_of_proc), for a symbol whose string matches symname.
 *
 *   If not found, this routine returns SYM_VAR_NOT_FOUND. If the
 *   symbol is found, but isn't a static variable, this routine returns
 *   SYM_NOT_STATIC. If a static symbol is found that is NOT a structure,
 *   but expecting_struct is TRUE, this routine returns
 *   SYM_UNEXPECTED_ERR. If a static symbol is found that IS a structure, 
 *   but a scalar was expected, this routine returns SYM_NOT_SCALAR. If a 
 *   static symbol with the proper name is found within the procedure 
 *   symbol space that is not a structure (and expecting_struct is FALSE), 
 *   this routine returns SYM_OK with "i" pointing to the symbol entry and
 *   "addr" loaded with the value of that symbol's address.
 *   If a static symbol is found that is a structure, and expecting_struct
 *   is TRUE, this routine returns SYM_OK with "i" pointing to the 
 *   structure's stBlock symbol table entry and "addr" loaded with the value
 *   of the structure's beginning address. Any other result should return
 *   SYM_UNEXPECTED_ERR.
 */

static int lookForSym( char *symname, int lookingForMember,
		int expecting_struct, char **addr, int num_indices )
{
  long firstSym, lastSym;
  long idaux, itaux;
  AUXU *daux, *taux;

  symbol_found = FALSE;
  lastSym = end_of_proc;

  if ( FAILURE == symtbread(i, &symbol)) return SYM_UNEXPECTED_ERR;
  if ( symbol.psymr->st == stStruct ) /* if we're looking in a structure... */
      {
	  firstSym = i;			/* save start of structure */
	  i = symbol.psymr->index + 
	      FD(symbol.ifd)->isymBase - 1;	/* point to end of structure */

	  if ( FAILURE == symtbread(i, &symbol) ) return SYM_UNEXPECTED_ERR;
	  if ( symbol.psymr->st != stEnd ) return SYM_UNEXPECTED_ERR;

	  lastSym = i;

	  i = firstSym+1;
	  if( (i < firstSym) || (i > lastSym) ) return SYM_UNEXPECTED_ERR;
      }

  while( i < symmax ) /* loop, but make absolutely certain not to go off end */
    {
      if ( FAILURE == symtbread(i, &symbol)) return SYM_UNEXPECTED_ERR;
      namep = symgetname( symbol );
      if ( namep == NULL ) return SYM_UNEXPECTED_ERR;
      if ( symbol_found = !strcmp( namep, symname ) )
	{
	    /* symbol found  -- update address info */

	  if ( !lookingForMember )  /* looking for static symbol */
	    {
	      if ( !( ( symbol.psymr->st == stStatic ) ||
	              ( symbol.psymr->st == stGlobal ) ) ) 
		  	return SYM_NOT_STATIC;
	      *addr = (char *) symbol.psymr->value; 
	    }
	  else		/* looking for subsequent structure member */
	    {
	      if ( symbol.psymr->st != stMember ) return SYM_UNEXPECTED_ERR;
	      *addr = *addr + symbol.psymr->value/8;
	    }

	  idaux = 0;
	  daux = AUX( symbol.itaux );
	  if ( daux == NULL ) return SYM_UNEXPECTED_ERR;

	  if ( daux->ti.tq0 == tqArray ) /* array element found */
	      {
		  if (num_indices <= 0) return SYM_NOT_SCALAR;
		  idaux = symbol.itaux;	/* save pointer to dim. info */
	      }

	  while (ISTYPEDEF(symbol))	/* dereference to get base type */
	    {
		i = followRFD( i );
		if ( i == 0 ) return SYM_UNEXPECTED_ERR;
		if ( FAILURE == symtbread(i, &symbol)) 
		    return SYM_UNEXPECTED_ERR;
	  	/* check for proper array-ness */

		daux = AUX( symbol.itaux );
		if ( daux == NULL ) return SYM_UNEXPECTED_ERR;

		if ( daux->ti.tq0 == tqArray ) /* array element found */
		    {
			if (num_indices <= 0) return SYM_NOT_SCALAR;
			idaux = symbol.itaux;	/* save pointer to dim. info */
		    }
	    }

	  if ((num_indices > 0) && (idaux == 0)) return SYM_UNEXPECTED_ERR;
	  symbol.idaux = idaux;		/* restore pointer to array, if any */

	  taux = AUX( symbol.itaux );  /* get type aux entry */
	  if ( taux == NULL ) return SYM_UNEXPECTED_ERR;

	  if (expecting_struct && !ISTRUCT(taux)) return SYM_UNEXPECTED_ERR;
	  if (!expecting_struct && ISTRUCT(taux)) return SYM_NOT_SCALAR;

	  if (ISTRUCT(taux))  /* need to point to stStruct sym */
	    {
		if ( symbol.psymr->st == stGlobal ) /* need to find stStruct */
		    {
			i = followRFD( i );
			if ( i == 0 ) return SYM_UNEXPECTED_ERR;
			if ( FAILURE == symtbread(i, &symbol)) 
			    return SYM_UNEXPECTED_ERR;
			taux = AUX( symbol.itaux );
			if ( taux == NULL ) return SYM_UNEXPECTED_ERR;
		    }
		if ( symbol.psymr->st == stStatic ||
		     symbol.psymr->st == stMember ) /* need stStruct */
		    {
			i = i - 1; 		    /* back up to stEnd */
			if ( FAILURE == symtbread(i, &symbol)) 
			    return SYM_UNEXPECTED_ERR;
			i = symbol.psymr->index +
			    FD(symbol.ifd)->isymBase; /* pt to stStruct */
			if ( FAILURE == symtbread(i, &symbol)) 
			    return SYM_UNEXPECTED_ERR; /* get new sym */
			taux = AUX( symbol.itaux );	/* and itaux */
			if ( taux == NULL ) return SYM_UNEXPECTED_ERR;
		    }
		if ( !( symbol.psymr->st == stStruct ||
		        symbol.psymr->st == stUnion    ) )
		  	return SYM_UNEXPECTED_ERR;
	    }
	  return SYM_OK;
	}
      i++;		/* increment index to next symbol */
      if (( i >= lastSym ) && !symbol_found ) return SYM_VAR_NOT_FOUND;
    }
  return SYM_UNEXPECTED_ERR;
}



/*
 *		===== countChars =====
 *
 * This function counts the number of times a particular character
 * (given by "Char") is found in the provided string "strg" in the
 * argument "cnt".
 */

static int countChars( char *strg, int *cnt, char Char )
{
/* counts the number of Char in the string */
  char *ptr;

  ptr = strg;
  *cnt = -1;
  do
    {
      ptr++;
      ptr = strchr( ptr, Char );
      (*cnt)++;
    }
  while( ptr != NULL );
  return SYM_OK;
}



/*
 *		===== parseName =====
 *
 * This routine parses the provided variable name, and returns
 * indications of whether the name contains subelements
 * (expecting_struct) or indices (if num_indices > 0)
 */

int parseName(char **nextToken, char *myvarname, 
	      int *expecting_struct, int *numIndices, int **index_list )
{
  int	numchar;
  char	*lparenloc, *rparenloc, *lbrackloc, *rbrackloc, *seploc, *dotloc;
  char 	*sepstrg = "[%d";
  char	sepchar = '[';
  enum 	 { none, C, Fortran } array_type;
  int	result;
  int	*indexPtr, indexCtr;

  *numIndices = 0;
  dotloc = strchr( myvarname, '.' );
  if (dotloc == NULL)
    {
      *expecting_struct = FALSE;
      *nextToken = myvarname;
    }
  else
    {
      *expecting_struct = TRUE;
      *dotloc = '\0';		/* separate token from rest of symbol */
      *nextToken = dotloc+1;
    }

  lbrackloc = strchr( myvarname, '[' ); /* look for C array */
  rbrackloc = strchr( myvarname, ']' );
  lparenloc = strchr( myvarname, '(' ); /*  "    " FORTRAN array */
  rparenloc = strchr( myvarname, ')' );

  if (   (lbrackloc == NULL)
      && (rbrackloc != NULL) ) return SYM_UNMATCHED_PAREN;
  if (   (lparenloc == NULL)
      && (rparenloc != NULL) ) return SYM_UNMATCHED_PAREN;

  array_type = none;

  if (lbrackloc != NULL)
    {
      if (rbrackloc == NULL) return SYM_UNMATCHED_PAREN;
      if (lparenloc != NULL) return SYM_BAD_SYNTAX;

      array_type = C;
    }

  if (lparenloc != NULL)
    {
      if (rparenloc == NULL) return SYM_UNMATCHED_PAREN;
      if (lbrackloc == NULL) return SYM_BAD_SYNTAX;
      
      lbrackloc = lparenloc;
      rbrackloc = rparenloc;

      sepstrg[0] = '(';

      array_type = Fortran;
    }

  if (array_type != none)
    {
      /* allocate memory for indexes */

      if (array_type == C) 
	{
	  result = countChars( lbrackloc, numIndices, ']' );
	}
      if (array_type == Fortran)
	{
	  result = countChars( lbrackloc, numIndices, ',' );
	  (*numIndices)++;
	}
      if (result != SYM_OK) return result;

      (*index_list) = malloc( (*numIndices)*sizeof( int ) );
      if (*index_list == NULL) return SYM_MEMORY_ERR;

      /* read first index */

      indexPtr = *index_list;
      indexCtr = 0;

      numchar = sscanf( lbrackloc, sepstrg, indexPtr );
      if (numchar < 1) return SYM_BAD_SYNTAX;
      if (array_type == Fortran) (*indexPtr)--;
      if (*indexPtr < 0) return SYM_BAD_SYNTAX;
      indexPtr++;
      indexCtr++;

      /* read remaining indexes */

      seploc = lbrackloc;
      if (array_type == Fortran)
	{
	  sepchar = ',';
	  sepstrg[0] = sepchar;
	}
      while (indexCtr < *numIndices)
	{
	  seploc = strchr( seploc+1, sepchar );
	  if (seploc == NULL) return SYM_BAD_SYNTAX;

	  numchar = sscanf( seploc, sepstrg, indexPtr );
	  if (numchar < 1) return SYM_BAD_SYNTAX;
	  if (array_type == Fortran) (*indexPtr)--;
	  if (*indexPtr < 0) return SYM_BAD_SYNTAX;

	  indexPtr++;
	  indexCtr++;
	}
      *lbrackloc = '\0';
    }

  return SYM_OK;
}



/*
 *	===== calcOffset =====
 *
 * This function is given the number of dimensions of
 * an array, as well as a list of the indexes in each dimension, and
 * returns the offset from the initial entry. It is limited to three
 * dimensions.  On entry, both "symbol" and "paux" have to be
 * initialized to point to the array entry and its associated
 * auxiliary symbol entry.  If any index is outside the allowable
 * dimensions, the routine returns SYM_INDEX_BOUNDS_ERR (something dbx
 * doesn't do).  If all goes well, the offset is stored in the
 * location pointed to by argument "offset", and returns SYM_OK. Any
 * other result should return SYM_UNEXPECTED_ERR.
 */

static int	calcOffset( long *offset, int num_indices, int *index_list )
{
  long	size, dimLo, dimHi;
  pAUXU	dimpaux, paux;
  int j;


  *offset = 0;
  dimpaux = AUX( symbol.idaux );
  paux = dimpaux;
  if (paux[0].ti.bt == btTypedef) paux = paux+2; /* skip over extra RFD */
  switch( num_indices )
    {
	case 6: if (dimpaux->ti.tq5 != tqArray) return SYM_INDEX_BOUNDS_ERR;
	case 5: if (dimpaux->ti.tq4 != tqArray) return SYM_INDEX_BOUNDS_ERR;
	case 4: if (dimpaux->ti.tq3 != tqArray) return SYM_INDEX_BOUNDS_ERR;
	case 3: if (dimpaux->ti.tq2 != tqArray) return SYM_INDEX_BOUNDS_ERR;
	case 2: if (dimpaux->ti.tq1 != tqArray) return SYM_INDEX_BOUNDS_ERR;
	case 1: if (dimpaux->ti.tq0 != tqArray) return SYM_INDEX_BOUNDS_ERR;
	    break;
	default: return SYM_UNEXPECTED_ERR;
    }
  for(j = num_indices-1; j>=0; j--)
  {
      dimLo = paux[3].dnLow;
      dimHi = paux[4].dnHigh;
      if (index_list[j] > dimHi) return SYM_INDEX_BOUNDS_ERR;
      if (index_list[j] < dimLo) return SYM_INDEX_BOUNDS_ERR;
      size = paux[5].width/8;
      *offset = (*offset) + index_list[j]*size;
      paux = paux+5; /* fall through to next dimension */
  }
  return SYM_OK;

}	/* end of calcOffset */


/*
 *		===== ls_findsym ====
 *
 * The main routine. Given a module name and variable name, this
 * routine looks up the address and type of variable and returns them
 * to the calling program. If a variable is global, "modname" must
 * consist of a single asterisk "*". The variable name can be a
 * scalar, array, or structure, with fields separated with periods
 * (customary C usage). Arrays can have no more than three
 * dimensions. This routine has been tested with C modules; no FORTRAN
 * support for structures or arrays is guaranteed. An appropriate
 * success or error code is returned (see ls_sym.h for the complete
 * list). If the initial attempt to access the debugger symbol table
 * fails, subsequent calls to ls_findsym will return the load error
 * message (either SYM_NO_SYMS or SYM_OPEN_ERR) will be returned
 * without further attempts to access the table.
 */

int ls_findsym( const char *modname, const char *varname, 
	        char **addr, vartype *vtype )
{
  static int 	sym_load_status = SYM_NOT_LOADED;
  int		result, expecting_struct, elem_size;
  int		num_indices, *index_list = 0;
  long		offset;
  char 		*tokenbuf, *myvarname, *tokenptr;
  char		*nextSym;
  size_t	stringsize;
  AUXU *taux; /*temporary*/

/* Module initialization */

  if (sym_load_status == SYM_NOT_LOADED)
      sym_load_status = loadSymbols();
  
  if (sym_load_status != SYM_OK) return sym_load_status;

  /* start search for symbol from beginning of file, or global section
     if modname is '*'      */

/* Lookup initialization */

  *addr = (char *) NULL;
  symbol_found = FALSE;
  module_found = FALSE;
  i = 0;
  symmax = symmaxlocal;
  if ( modname[0] == '*' ) 	/* global symbol requested */
    {
      module_found = TRUE;
      i = symmaxlocal;
      symmax = symmaxextern + i;
      end_of_proc = symmax - 1;
    }

/* Ready to do lookup */

  if ( !module_found )
    {
      result = lookForModule( modname );
      if (result != SYM_OK) return result;
    }

/* make local copy of variable name */
  
  stringsize = strlen( varname );
  tokenbuf = malloc( stringsize+1 );
  if ( tokenbuf == NULL ) return SYM_MEMORY_ERR;
  myvarname = tokenbuf;
  strncpy( myvarname, varname, stringsize );
  myvarname[stringsize] = '\0';   /* make sure the string is terminated */
  tokenptr = myvarname;		/* initialize parser pointer */

  expecting_struct = FALSE;

/* loop until symbol found and not expecting a structure */

  while(!symbol_found || expecting_struct)	
	/* same as !(sym_fnd && !exp_strct) */
    {
      /* parse name into tokens */
      
      result = parseName( &nextSym, tokenptr, &expecting_struct, 
			  &num_indices, &index_list );
      if (result != SYM_OK) SYMEXIT( result );

      /* look for next required symbol */

      result = lookForSym( tokenptr, symbol_found, 
			   expecting_struct, addr, num_indices );
      if (result != SYM_OK) SYMEXIT( result );

      switch( AUX( symbol.itaux )->ti.bt ) 
	  {
	  case btChar:	 *vtype = Char;  elem_size = sizeof( char		); break;
	  case btUChar:  *vtype = UChar; elem_size = sizeof( unsigned char	); break;
	  case btShort:	 *vtype = SHint; elem_size = sizeof( short int		); break;
	  case btUShort: *vtype = USHint;elem_size = sizeof( unsigned short int	); break;
	  case btInt:    *vtype = Sint;  elem_size = sizeof( int		); break;
	  case btUInt:   *vtype = Uint;  elem_size = sizeof( int		); break;
	  case btLong:   *vtype = Slng;  elem_size = sizeof( long		); break;
	  case btULong:  *vtype = Ulng;  elem_size = sizeof( long		); break;
	  case btFloat:  *vtype = flt;   elem_size = sizeof( float		); break;
	  case btDouble: *vtype = dbl;   elem_size = sizeof( double		); break;
	  default: 
	      {
		*vtype = Unknown;
		if (!expecting_struct) SYMEXIT( SYM_NOT_SCALAR );
	     	elem_size = symbol.psymr->value/8;
	      }
	  }

      /* calculate address of indexed element, if any */

      if (num_indices > 0) 
	  {
	      result = calcOffset( &offset, num_indices, index_list );
	      if (result != SYM_OK) SYMEXIT( result );
	      *addr = (*addr) + offset;
	  }

      tokenptr = nextSym;
    }

  free( tokenbuf );
  free( index_list );

  if (*vtype == Unknown) return SYM_UNEXPECTED_ERR;

  return SYM_OK;
}


void ls_print_findsym_error(int result, char *mod_name, char *var_name)
/* Prints an appropriate error on stderr if result is non-zero */
{
    fprintf(stderr, "Error in routine ls_findsym: ");
    switch ( result )
	{
	case SYM_UNEXPECTED_ERR: 
	    fprintf(stderr, 
		    "Unexpected error encountered when\n\tlooking up variable '%s' in module '%s'.\n", 
		    var_name, mod_name);
	    fprintf(stderr,
		    "\tPossible indexing of scalar variable?\n");
	    break;
	case SYM_OPEN_ERR:	 
	    fprintf(stderr, 
		    "Error opening symbol table.\n");         
	    break;
	case SYM_NO_SYMS:        
	    fprintf(stderr, 
		    "Symbol table not found.\n");
	    break;
	case SYM_MOD_NOT_FOUND:	 
	    fprintf(stderr, 
		    "Module '%s' not found.\n", mod_name);    
	    break;
	case SYM_VAR_NOT_FOUND:	 
	    fprintf(stderr, 
		    "Variable '%s' not found in module '%s'.\n", var_name, mod_name);    
	    break;
	case SYM_NOT_SCALAR:	 
	    fprintf(stderr, 
		    "Variable '%s' in module '%s' is non-scalar. Facility variables must be scalar.\n",
		    var_name, mod_name );       
	    break;
	case SYM_NOT_STATIC:	 
	    fprintf(stderr, 
		    "Variable '%s' in module '%s' must be declared static to be used in facilities.\n",
		    var_name, mod_name );
	    break;
	case SYM_MEMORY_ERR:	 
	    fprintf(stderr, 
		    "Memory error in ls_findsym routine; couldn't allocate something.\n");       
	    break;
	case SYM_UNMATCHED_PAREN:
	    fprintf(stderr, 
		    "Unmatched parenthesis found when\n\tlooking for variable '%s' in module '%s'.\n",
		    var_name, mod_name);  
	    break;
	case SYM_BAD_SYNTAX:	 
	    fprintf(stderr, 
		    "Bad syntax when looking for variable '%s' in module '%s'.\n",
		    var_name, mod_name);       
	    break;
	case SYM_INDEX_BOUNDS_ERR: 
	    fprintf(stderr, 
		    "Symbol indexing bounds error detected when\n\tlooking for '%s' in module '%s'.\n",
		    var_name, mod_name);
	    break;
	default: 
	    fprintf(stderr, 
		    "Unrecognized error code %d returned while looking for '%s\%s'.\n",
		    result, var_name, mod_name);
	} /* end of switch (result) statement */
}


double	ls_get_double(vartype sym_type, void *addr )
    /* obtains data at addr and returns a double value, based on type given in sym_type */
{
    double value = 1./0.;	/* Generate Inf */

    if (addr)
	{
	    switch( sym_type )
		{
		case Char:  value = *(  signed       char   *) addr; break;
		case UChar: value = *(unsigned       char   *) addr; break;
		case SHint: value = *(  signed short int    *) addr; break;
		case USHint:value = *(unsigned short int    *) addr; break;
		case Sint:  value = *(  signed       int    *) addr; break;
		case Uint:  value = *(unsigned       int    *) addr; break;
		case Slng:  value = *(  signed long  int    *) addr; break;
		case Ulng:  value = *(unsigned long         *) addr; break;
		case  flt:  value = *(               float  *) addr; break;
		case  dbl:  value = *(               double *) addr; break;
		} /* end of switch( sym_type ) statement */
	}
    return value;
}


void ls_set_double(vartype sym_type, void *addr, double value )
    /* Sets variable at addr to value of double provided */
{
    switch( sym_type )
	{
	case Char:  *(  signed       char   *) addr = value; break;
	case UChar: *(unsigned       char   *) addr = value; break;
	case SHint: *(  signed short int    *) addr = value; break;
	case USHint:*(unsigned short int    *) addr = value; break;
	case Sint:  *(  signed       int    *) addr = value; break;
	case Uint:  *(unsigned       int    *) addr = value; break;
	case Slng:  *(  signed long  int    *) addr = value; break;
	case Ulng:  *(unsigned long	    *) addr = value; break;
	case  flt:  *(               float  *) addr = value; break;
	case  dbl:  *(               double *) addr = value; break;
	} /* end of switch( sym_type ) statement */
}



double	ls_get_sym_val( symbol_rec *symrec, int *error )

	/* This routine attempts to return the present value of the symbol
	   described in symbol_rec. If Addr is non-zero, the value of that
	   location, interpreted as type double, will be returned. If Addr
	   is zero, and Mod_Name and Par_Name are both not null strings, 
	   the ls_findsym() routine is used to try to obtain the address
	   by looking at debugger symbol tables in the executable image, and
	   the value of the double contained at that address is returned, 
	   and the symbol record is updated to contain the address of that
	   symbol. If an error is discovered, 'error' will be non-zero and
	   and error message is printed on stderr.			*/
{
    *error = 0;

    if (!symrec->Addr)
	{
	    /* Here on null address; look up symbol in tables */

	    *error = ls_findsym( symrec->Mod_Name, symrec->Par_Name, 
				 &symrec->Addr, &symrec->Par_Type );

	    if ((!*error) && (!symrec->Addr))	/* still null addr */
		*error = SYM_UNEXPECTED_ERR;

	    if (*error) /* report any problems and give up */
		{
		    ls_print_findsym_error( *error, 
					    symrec->Mod_Name, 
					    symrec->Par_Name );
		    return *error;
		}
	}

    /* here on non-NULL address */

    return ls_get_double(symrec->Par_Type, symrec->Addr);
	
}


void 	ls_set_sym_val( symbol_rec *symrec, double value )

	/* This routine sets the value of a double at the location pointed
	   to by the symbol_rec's Addr field, if Addr is non-zero. If Addr
	   is zero, and Mod_Name and Par_Name are both not null strings, 
	   the ls_findsym() routine is used to try to obtain the address
	   by looking at debugger symbol tables in the executable image, and
	   the value of the double contained at that address is returned, 
	   and the symbol record is updated to contain the address of that
	   symbol. If an error is discovered, 'error' will be non-zero and
	   and error message is printed on stderr.			*/
{
    int error;

    if (!symrec->Addr)
	{
	    /* Here on null address; look symbol in tables */

	    error = ls_findsym( symrec->Mod_Name, symrec->Par_Name,
				&symrec->Addr, &symrec->Par_Type );
	    
	    if ((!error) && (!symrec->Addr))	/* still null addr */
			    error = SYM_UNEXPECTED_ERR;
	    if (error)
		{
		    ls_print_findsym_error( error,
					    symrec->Mod_Name,
					    symrec->Par_Name );
		    return;
		}
	}

    /* here on non-NULL address */

    ls_set_double( symrec->Par_Type, symrec->Addr, value );
    
    return;
}


