/***************************************************************************

	TITLE:		ls_settings.c
	
----------------------------------------------------------------------------

	FUNCTION:	Performs settings file utilities for LaRCsim

		Two major routines are provided in this module: 
	ls_get_settings() and ls_put_settings(). These routines read and
	write the .set file used to record various LaRCsim user defined
	settings (run time length, trim variables, initial conditions,
	and variables to record, for example).

		The ls_get_settings() routine locates and opens (using 
	ls_fopen()), also provided in this module) the appropriate settings
	file, and parses the information contained therein.  It has a list
	of facilities (such as IC, record, trim, etc.) that utilize variables
	in the settings file and calls their "get_set" routine as their 
	keywords are encountered.

		The ls_fopen() routine searches for an appropriately named 
	file somewhere along the LARCSIMPATH or within the default directory
	if no path is defined.

		The ls_put_settings() routine creates a new settings file
	and calls each facility's "put_set" routine to plop in the appropriate
	settings.

----------------------------------------------------------------------------

	MODULE STATUS:	incomplete

----------------------------------------------------------------------------

	GENEALOGY:	Created 950301 by E. B. Jackson

----------------------------------------------------------------------------

	DESIGNED BY:	Bruce Jackson
	
	CODED BY:	Bruce Jackson
	
	MAINTAINED BY:	

----------------------------------------------------------------------------

	MODIFICATION HISTORY:
	
	DATE	PURPOSE						BY

	950308	Added comment line with time stamp to ls_put_settings().
								EBJ
	950314	Added init facility.				EBJ
	950406	Added #includes of ls_types.h & ls_tape.h (to gain access
		to Tape->Length parameter); added new facility:
		ls_sim_get_set() and ls_sim_put_set(), to save and
		restore simulation option settings.		EBJ
	
	CURRENT RCS HEADER:

$Header: /aces/larcsim/dev/RCS/ls_settings.c,v 1.6 1995/04/07 01:35:58 bjax Exp $
$Log: ls_settings.c,v $
 * Revision 1.6  1995/04/07  01:35:58  bjax
 * Added ls_sim_get_set() and ls_sim_put_set() routines to save & restore
 * simulation options.
 *
 * Revision 1.5  1995/03/15  12:22:31  bjax
 * Added init facility; reworked logic of ls_get_settings() so that
 * a file name can be passed; if no file name is supplied, the default
 * settings file is opened.
 * EBJ
 *
 * Revision 1.4  1995/03/08  12:30:42  bjax
 * Added time, date, and user stamp to comment line in settings file output.
 *
 * Revision 1.3  1995/03/07  22:34:26  bjax
 * Added guts to ls_put_settings(); now have two facilities online: trim & record.
 *
 * Revision 1.2  1995/03/06  18:47:15  bjax
 * Reworked the facility list so that "set" facilities are passed the
 * pointer to the end of the buffer, so they can detect overruns, and
 * return pointer to next token in buffer. EBJ
 *
 * Revision 1.1  1995/03/03  02:17:34  bjax
 * Initial revision
 *

----------------------------------------------------------------------------

	REFERENCES:

----------------------------------------------------------------------------

	CALLED BY:

----------------------------------------------------------------------------

	CALLS TO:

----------------------------------------------------------------------------

	INPUTS:

----------------------------------------------------------------------------

	OUTPUTS:

--------------------------------------------------------------------------*/

#include <limits.h>	/* defines PATH_MAX */
#include <sys/types.h>	/* needed for stat(3C) */
#include <sys/stat.h> 
#include <sys/param.h>	/* needed for realpath(3C) */
#include <stdlib.h>	/* needed for realpath(3C), getenv(3C) */
#include <stdio.h>
#include <string.h>
#include "ls_types.h"
#include "ls_constants.h"	/* for NIL_POINTER and PATHNAME */
#include "ls_sim_control.h"	/* for simname */
#include "ls_tape.h"		/* for TAPE def'n */

extern TAPE *Tape;

static char rcsid[] = "$Id: ls_settings.c,v 1.6 1995/04/07 01:35:58 bjax Exp $";

#define DEFAULT_PATH "./"
#define PATH_SEP  ":"
#define MAX_PATHNAME_LENGTH PATH_MAX
#define MAX_LINE_SIZE 255
#define INDIRECT_FLAG '@'
#define COMMENT_FLAG '#'

FILE *ls_fopen(const char *filename, const char *type, struct stat *statbuf )
{
    static int init = 0;
    static char *pathname;

    FILE *fp = 0L;
    char *tempname, *homedir;
    char qual_name_buf[ MAX_PATHNAME_LENGTH ];
    int	status;
    int len;

    if (init == 0)

	/* Initializes the static string pathname to contain either
	   the current directory path "./" or the pathname specified
	   in an environment pathname (defined in ls_constants.h) */
	{
	    init = -1;
	    
	    tempname = getenv( PATHNAME );
	    if (tempname == NIL_POINTER)
		{
		    pathname = (char *) malloc(strlen(DEFAULT_PATH));
		    strcpy(pathname, DEFAULT_PATH);
		}
	    else
		{
		    pathname = (char *) malloc(strlen(tempname));
		    strcpy(pathname, tempname);
		}
	}

    /* Now try to construct the fully qualified file name and open it.
       If the filename starts with a slash, ignore the pathname and
       try to open the absolute filename provided */

    if ((filename[0] == '/') || (*type == 'w'))  /* no path allowed */
	{
	    return fopen( filename, type );
	}
    else
	/* Here if filename isn't absolute. Cycle through each of the
	   directories listed in pathname, expanding a leading ~ into
	   the user's home directory, and any ./ or ../ into default
	   or previous directory, until either a file is located or the
	   directory list is exhausted. */
	{
	    tempname = strtok(pathname, PATH_SEP);  /* tempname has first dir */
	    do
		{
		    len = strlen(tempname);
		    if (len > MAX_PATHNAME_LENGTH) break;
		    if (len <= 0) break;	/* shouldn't happen */
		    if (tempname[0] == '~')	
				/* expand home directory name */
			{
			    homedir = getenv("HOME");
			    if (homedir == NIL_POINTER) break; /* give up */
			    len = strlen( homedir );
			    if (len > MAX_PATHNAME_LENGTH) break;
			    if (len <= 0) break;
			    strcpy( qual_name_buf, homedir );
			    strncat( qual_name_buf, tempname+1,
				     MAX_PATHNAME_LENGTH-strlen(qual_name_buf));
			    tempname = qual_name_buf; /* point to new str */
			}
		    tempname = realpath( tempname, qual_name_buf ); 
				/* expand ./ or ../ directory name */
		    if (tempname == NIL_POINTER) break;
		    if (tempname[len-1] != '/')
			strcat(qual_name_buf, "/" );	/* add final slash */
		    tempname = 			/* append file name */
			strncat(qual_name_buf, filename, 
				MAX_PATHNAME_LENGTH-strlen(qual_name_buf));
		    if (tempname)
			{
			    /* if still a valid string, check to see if
			       'tempname' describes an existing file */
			    status = stat( qual_name_buf, statbuf );
			    if (!status)
				{
				    return fopen( tempname, type );
				}
			    tempname = strtok(NIL_POINTER, PATH_SEP);
			}
		    else break;
		}
	    while (tempname);
	}

    return fp;

}


#define NUM_FACILITIES 4
struct fac
{
    char *keyword;
    char *(*get_set_func)(); /* pointers to functions returning ptr to char */
    void  (*put_set_func)(); /* pointers to functions returning void */
};

static struct fac facilities[NUM_FACILITIES];

        char *ls_sim_get_set(    char *buffer, char *eob );
extern	char *ls_record_get_set( char *buffer, char *eob );
extern	char *ls_trim_get_set(   char *buffer, char *eob );
extern	char *ls_init_get_set(   char *buffer, char *eob );

       void ls_sim_put_set(   FILE *fp );
extern void ls_record_put_set(FILE *fp );
extern void ls_trim_put_set(  FILE *fp );
extern void ls_init_put_set(  FILE *fp );


void init_fac_list()
/* Initialize the above facility list */
{
    static char keyword0[] = "sim";
    static char keyword1[] = "record";
    static char keyword2[] = "trim";
    static char keyword3[] = "init";

    facilities[0].keyword = keyword0;
    facilities[1].keyword = keyword1;
    facilities[2].keyword = keyword2;
    facilities[3].keyword = keyword3;

    facilities[0].get_set_func = &ls_sim_get_set;
    facilities[1].get_set_func = &ls_record_get_set;
    facilities[2].get_set_func = &ls_trim_get_set;
    facilities[3].get_set_func = &ls_init_get_set;

    facilities[0].put_set_func = &ls_sim_put_set;
    facilities[1].put_set_func = &ls_record_put_set;
    facilities[2].put_set_func = &ls_trim_put_set;
    facilities[3].put_set_func = &ls_init_put_set;

}


int ls_parse_settings(FILE *fp, struct stat *statbuf )
    /* this routine reads the settings file into a buffer,
       resolves comment lines & includes, and calls the
       appropriate get_set routines.	*/
{
    char *buffer, *bufptr, *endptr, **lasts, *nullptr, null = '\0';
    char line[MAX_LINE_SIZE];
    off_t filesize;
    int status, n, i, skiptok;
    FILE *new_settings_file;
    struct stat *new_statbuf;

    nullptr = &null;
    lasts = &nullptr;

    /* read file into buffer */

    filesize = statbuf->st_size;
    buffer = (char *) malloc(filesize);
    bufptr = buffer;
    endptr = buffer+filesize;	/* points to one char past end of buffer */

    if (buffer == NIL_POINTER) return -1;
    status = fread(buffer, filesize, 1, fp);
    if (status != 1) { free( buffer ); return -1; }

    bufptr = strtok_r( buffer, "\n", lasts);
    while ((bufptr < endptr) && (bufptr >= buffer))
	{
	    skiptok = 0;
	    n = sscanf(bufptr, "%s", line);
	    switch( line[0] )	    /* examine first char */
		{
		case INDIRECT_FLAG:
		    /* if the first character of a line starts with redirection
		       symbol, open that file and parse it */

		    new_statbuf = (struct stat *)malloc(sizeof( struct stat ));
		    if (new_statbuf == NIL_POINTER) {free(buffer); return -1;}
		    new_settings_file = ls_fopen( bufptr+1, "r", new_statbuf);
		    if (new_settings_file)
			{
			    status = ls_parse_settings( new_settings_file,
							new_statbuf );
			    if (status) fprintf(stderr, 
				"Error parsing subordinate settings file.\n");
			    fclose(new_settings_file);
			}
		    free( new_statbuf );
		    break;
		case COMMENT_FLAG:
		    /* skip over comments */
		    break;
		default:	/* not a comment or redirection; look at word */
		    for(i=0;i<NUM_FACILITIES;i++)
			if (strcasecmp( line, facilities[i].keyword ) == 0) 
			    {
/*				printf("%s\n", facilities[i].keyword); */
				bufptr = 
				   (*facilities[i].get_set_func)(bufptr,endptr);
					/* call the command that matches */
				skiptok = 1;
				break;
			    }
		}
	    if (skiptok)
		{	
		    lasts = &nullptr; /* reset strtok search */
		    bufptr = strtok_r(bufptr, "\n", lasts);
		}
	    else
		bufptr = strtok_r( NIL_POINTER, "\n", lasts );
	}
    
    free( buffer );
    return 0;
}

void ls_get_settings( char *desired_file_name )
{
    FILE *settings_file;
    struct stat statbuf;
    int status, namelen;

    char *settings_file_name;

    if (desired_file_name[0] == '\0') 
	{
	    settings_file_name = (char *) malloc( strlen(sim_control_.simname) + 1 );
	    if (settings_file_name)
		{
		    settings_file_name[0] = '.';
		    settings_file_name[1] = '\0';
		    strcat(settings_file_name, sim_control_.simname);
		}
	}
    else
	{  /* make local copy */
	    namelen = strlen( desired_file_name )+1;
	    settings_file_name = (char *) malloc( namelen );
	    if (settings_file_name) strncpy( settings_file_name, desired_file_name, namelen );
	}
	    
		
    if (settings_file_name)
	{
	    settings_file = ls_fopen( settings_file_name, "r", &statbuf );
	    if (settings_file)
		{
		    init_fac_list();
		    status = ls_parse_settings( settings_file, &statbuf );
		    if (status) fprintf(stderr, "Error parsing settings file.\n");
		    fclose( settings_file );
		}
	}
    free( settings_file_name );
}

void ls_put_settings()
{
    FILE *settings_file;
    char *settings_file_name;
    struct stat statbuf;
    int i;

    settings_file_name = (char *) malloc( strlen(sim_control_.simname) + 3 );
    if (settings_file_name)
	{
	    settings_file_name[0] = '.';
	    settings_file_name[1] = '/';
	    settings_file_name[2] = '.';
	    settings_file_name[3] = '\0';
	    strcat(settings_file_name, sim_control_.simname);

	    settings_file = ls_fopen( settings_file_name, "w", &statbuf );
	    if (settings_file)
		{
		    init_fac_list();
		    fprintf(settings_file, "# .%s created at %s %s by %s\n",
			    sim_control_.simname,
			    sim_control_.date_string,
			    sim_control_.time_stamp,
			    sim_control_.userid );
		    for(i=0;i<NUM_FACILITIES;i++)
			    (*facilities[i].put_set_func)(settings_file);
		    (void) fclose( settings_file );
		}
	}
    free( settings_file_name );
}

#define FACILITY_NAME_STRING "sim"
#define CURRENT_VERSION 10

void ls_sim_put_set( FILE *fp )
{
    int i;

    if (fp==0) return;
    fprintf(fp, "\n");
    fprintf(fp, "#==============================  %s\n", FACILITY_NAME_STRING);
    fprintf(fp, "\n");
    fprintf(fp, FACILITY_NAME_STRING);
    fprintf(fp, "\n");
    fprintf(fp, "%04d\n", CURRENT_VERSION);
    fprintf(fp, "    write_av       %d\n", sim_control_.write_av);
    fprintf(fp, "    write_mat      %d\n", sim_control_.write_mat);
    fprintf(fp, "    write_tab      %d\n", sim_control_.write_tab);
    fprintf(fp, "    write_asc1     %d\n", sim_control_.write_asc1);
    fprintf(fp, "    write_spacing  %d\n", sim_control_.write_spacing);
    fprintf(fp, "    end_time       %f\n", sim_control_.end_time);
    fprintf(fp, "    model_hz       %f\n", sim_control_.model_hz); 
    fprintf(fp, "    term_update_hz %f\n", sim_control_.term_update_hz);
    fprintf(fp, "    data_rate      %f\n", sim_control_.model_hz / sim_control_.save_spacing);
    fprintf(fp, "    buffer_time    %f\n", Tape->Length*sim_control_.save_spacing/sim_control_.model_hz);
    fprintf(fp, "end\n");
    return;
}


char *ls_sim_get_set(char *buffer, char *eob)
/* This routine parses the settings file for "sim" entries. */
{

    static char *fac_name = FACILITY_NAME_STRING;
    char *bufptr, **lasts, *nullptr, null = '\0';
    int n, ver, i, error;
    float buffer_time, data_rate;
    char line[MAX_LINE_SIZE];
    char word[MAX_LINE_SIZE];
    char value[MAX_LINE_SIZE];

    /* set default values */

    buffer_time = sim_control_.time_slices * sim_control_.save_spacing / sim_control_.model_hz;
    data_rate   = sim_control_.model_hz / sim_control_.save_spacing;

    nullptr = &null;
    lasts   = &nullptr;

    n = sscanf(buffer, "%s", line);
    if (n == 0) return 0L;
    if (strncasecmp( fac_name, line, strlen(fac_name) )) return 0L;

    bufptr = strtok_r( buffer+strlen(fac_name)+1, "\n", lasts);
    if (bufptr == 0) return 0L;

    sscanf( bufptr, "%d", &ver );
    if (ver != CURRENT_VERSION) return 0L;

    while( eob > bufptr )
      {
	bufptr = strtok_r( 0L, "\n", lasts );
	if (bufptr == 0) return 0L;
	if (strncasecmp( bufptr, "end", 3) == 0) break;

	sscanf( bufptr, "%s", line );
	if (line[0] != '#') /* ignore comments */
	  {
	    n = sscanf( bufptr, "%s %s", word, value);
	    if (n == 2)
		{
		    if (strncmp(word, "write_av"      ,  8) == 0) sim_control_.write_av = atoi(value);
		    if (strncmp(word, "write_mat"     ,  9) == 0) sim_control_.write_mat = atoi(value);
		    if (strncmp(word, "write_tab"     ,  9) == 0) sim_control_.write_tab = atoi(value);
		    if (strncmp(word, "write_asc1"    , 10) == 0) sim_control_.write_asc1 = atoi(value);
		    if (strncmp(word, "write_spacing" , 13) == 0) sim_control_.write_spacing = atoi(value);
		    if (strncmp(word, "end_time"      ,  8) == 0) sim_control_.end_time = atof(value);
		    if (strncmp(word, "model_hz"      ,  8) == 0) sim_control_.model_hz = atof(value);
		    if (strncmp(word, "term_update_hz", 14) == 0) sim_control_.term_update_hz = atof(value);
		    if (strncmp(word, "data_rate"     ,  9) == 0) data_rate = atof(value);
		    if (strncmp(word, "buffer_time"   , 11) == 0) buffer_time = atof(value);
		}
		
	  }
      }


    sim_control_.save_spacing = (int) (0.5 + sim_control_.model_hz / data_rate);
    if (sim_control_.save_spacing < 1) sim_control_.save_spacing = 1;

    sim_control_.time_slices = buffer_time * sim_control_.model_hz / sim_control_.save_spacing;
    if (sim_control_.time_slices < 2) sim_control_.time_slices = 2;

    bufptr = *lasts;
    return bufptr;
}
