/***************************************************************************

	TITLE:		ls_record   $Id: ls_record.c,v 1.11 1995/04/07 01:46:43 bjax Exp $
	
----------------------------------------------------------------------------

	FUNCTION:	Store time history data from sim runs

----------------------------------------------------------------------------

	MODULE STATUS:	developmental

----------------------------------------------------------------------------

	GENEALOGY:	Created October 26, 1992 by Bruce Jackson

----------------------------------------------------------------------------

	DESIGNED BY:	Bruce Jackson
	
	CODED BY:	Bruce Jackson
	
	MAINTAINED BY:	maintainer

----------------------------------------------------------------------------

	MODIFICATION HISTORY:
	
	DATE	PURPOSE						BY

	930727	Original version (1.1) was hardwired to record certain variables;
		this version uses the ls_sym routines to look up scalar addresses
		prior to run. (Note: this doesn't work right yet for many of the
		variables defined in ls_eom.h, since they are only definitions
		of three-element vectors... shucks).
						EBJ
		
	931008	Added "interp"olation of frames, so only every INTERP'd frame
		is retained. Note that, by design, data is recorded every frame
		so that execution times don't vary; but only every INTERP'th frame
		is saved (by advancing the Next pointer).  This automatically saves
		the very first frame and the very last frame,  although the last
		frame won't necessarily be spaced equally with regard to preceeding
		frames.						EBJ
	
	931008	Also added min and max value info for each channel. EBJ
	
	940111	Changed include files from ls_eom.h to ls_types,  ls_generic,  and
		ls_sim_control.h

	940121	Modified to use new, improved ls_findsym, which now understands and
		locates structures and arrays (both of which are needed to access 
		the new global variables).				EBJ

	940506	Corrected logic that was setting value of each parm to zero. EBJ
		Also gave fixed variable useful names.
		
	940509	Fixed bug in circular buffer handling; now buffer wraps to
		location 0 correctly.

	950228	Incorporated utility routines provided now by ls_sym, such as
		ls_print_findsym_error() and ls_get_double().			EBJ

	950302	Modified CHANNEL definition in ls_tape.h to use symbol_rec provided
		by ls_sym.h to get consistent use of symbol record in this facility.

	950307	Now supports ls_record_get_set() and ls_record_put_set() calls. EBJ

	950405	Made length of data channels a sim_control_ parameter.		EBJ

	CURRENT RCS HEADER INFO:

$Header: /aces/larcsim/dev/RCS/ls_record.c,v 1.11 1995/04/07 01:46:43 bjax Exp $
$Log: ls_record.c,v $
 * Revision 1.11  1995/04/07  01:46:43  bjax
 * Modified to use Tape->Length instead of MAX_SLICES as tape length
 * reference; changed ls_record_tape_init() into two parts: ls_record_
 * channels_init() and ls_record_alloc_storage(), so individual Channels
 * can be allocated at the last moment, when Tape->Length has been determined;
 * modified initialization of Tape structure to reflect change of Channel
 * .Data from array of SCALAR to point to array of scalars.
 *
 * Revision 1.10  1995/03/15  12:16:23  bjax
 * Added flag marker line to ls_record_put_set() routine.
 *
 * Revision 1.9  1995/03/07  22:36:06  bjax
 * Moved short names of hardwired variables to alias field; added ls_record_put_set()
 * function. EBJ
 *
 * Revision 1.8  1995/03/06  18:42:34  bjax
 * Major structural changes: making use of ls_get_sym_val; separated
 * ls_record_tape_init() from ls_record() body; added ls_record_get_set();
 * minor cleanups.
 *
 * Revision 1.7  1995/03/03  02:00:50  bjax
 * Modified to use new def'n of Tape->Chan structure (includes symbol rec
 * defined in ls_sym.h). EBJ
 *
 * Revision 1.6  1995/02/28  12:58:16  bjax
 * Modified to use new ls_sym routines ls_print_findsym_error
 * and ls_get_double().  EBJ
 *
 * Revision 1.5  1994/05/17  12:25:05  bjax
 * For unknown reasons, the "interp" initialization was being
 * incorrectly initialized by sim_control_.save_settings. Changed
 * the declaration from static int to static short fixed the problem,
 * it appears.
 *
 * Revision 1.4  1994/05/09  21:20:52  bjax
 * Fixed problem with tape wrapping to second time slice.
 *
 * Revision 1.3  1994/05/06  20:18:27  bjax
 * More or less complete set of data types now converted properly.
 * Added comment line (first column '#') in .set settings file.
 *
 * Revision 1.2.1.13  1994/05/06  18:22:46  bjax
 * Gave useful short names to fixed data channels 0-18; corrected
 * interpretation and conversion of most data types.
 *
 * Revision 1.2.1.12  1994/05/06  16:35:48  bjax
 * Added close of settings file to end of ls_record_get_set routine.
 *
 * Revision 1.2.1.11  1994/05/06  16:32:02  bjax
 * Minor mods to record_get_set routine.
 *
 * Revision 1.2.1.10  1994/05/06  15:32:24  bjax
 * Fixed bug with all data values set to zero.
 *
 * Revision 1.2.1.9  1994/03/28  19:43:38  bjax
 * Added support for local "settings" file (e.g. navion.set) in cwd.
 * There appears to be a problem in ls_findsym, however.
 *
 * Revision 1.2.1.8  1994/01/11  18:56:41  bjax
 * Changed header includes from ls_eom to ls_types, ls_generic, and ls_sim_control
 *
 * Revision 1.2.1.7  1993/12/20  16:50:48  bjax
 * Cleaned up the time slice acess method. EBJ
 *
 * Revision 1.2.1.6  1993/10/08  22:04:02  bjax
 * Added Min value, max value calculations at record time. EBJ
 *
 * Revision 1.2.1.5  1993/10/08  19:34:36  bjax
 * Added interpolation logic, so every frame 6th frame is saved (10 Hz
 * at present execution speeds). Need to make it a control variable,
 * tough.  -- EBJ
 *
 * Revision 1.2.1.4  1993/08/03  20:00:09  bjax
 * Fixed to make Tape.Chan[].Addr pointer type compatible with ls_findsym call.
 *
 * Revision 1.2.1.3  1993/07/30  18:33:57  bjax
 * Corrected index on rudder initialization.
 *
 * Revision 1.2.1.1  1993/07/28  16:22:26  bjax
 * Further development of using symbol table lookups. EBJ
 *
 * Revision 1.1  1992/12/30  13:19:27  bjax
 * Initial revision
 *

----------------------------------------------------------------------------

	REFERENCES:

----------------------------------------------------------------------------

	CALLED BY:

----------------------------------------------------------------------------

	CALLS TO:

----------------------------------------------------------------------------

	INPUTS:

----------------------------------------------------------------------------

	OUTPUTS:

--------------------------------------------------------------------------*/


#include "ls_types.h"
#include "ls_generic.h"
#include "ls_sim_control.h"
#include "ls_tape.h" /* includes ls_sym.h */
#include <math.h>
#include <string.h> /* for strtok, strncmp, etc. */
#include <stdio.h>

#define NIL_POINTER 0L
#define INTERP sim_control_.save_spacing
#define LINE_LENGTH 256
#define HARDWIRED 19

#define FACILITY_NAME_STRING "record"
#define CURRENT_VERSION 10

extern TAPE *Tape;	/* declared in ls_main.c */
extern SCALAR Simtime;	/* declared in ls_main.c */

static int noTape = 0;
char line[LINE_LENGTH];
int num, abort;
SYMBOL_NAME mod_name, par_name;


ls_record_channels_init()
{
    int i, result;
    static int channels_allocated = 0;

    if (!channels_allocated)
      {
	channels_allocated = -1;
	noTape = 0;

	Tape = (TAPE *) malloc(sizeof(TAPE));
	if (Tape == NIL_POINTER) 
	{
	    fprintf(stderr, "ls_tape: memory allocation error\n");
	    noTape = -1;
	    return;
	} /* end of "if (Tape == NIL_POINTER)" path when 
	     allocating Tape structure */
	else
	{
	    Tape->Num_Chan = HARDWIRED;  /* presets */
	    Tape->First = -1;
	    Tape->Last = 0;
	    Tape->Current = 0;
	    Tape->Next = 0;
	    for(i=0;i<Tape->Num_Chan;i++)
	    {
		Tape->Chan[i] = (CHANNEL *) malloc( sizeof( CHANNEL ));
		if (Tape->Chan[i] == NIL_POINTER)
		{
		    fprintf(stderr, "ls_tape: memory allocation error\n");
		    noTape = -1;
		    for(;i>0;--i) free(Tape->Chan[i]);
		    free(Tape);
		    return;
		} /* end of "if (Tape->Chan[i] == NIL_POINTER)" path */
		else
		{
		    Tape->Chan[i]->Max_value = (SCALAR) sqrt(-3.); /* NaN */
		    Tape->Chan[i]->Min_value = (SCALAR) sqrt(-3.);
		} /* end of "if (Tape->Chan[i] <> NIL_POINTER)" path */
	    } /* end of "for(i=0;i<Tape->Num_Chan;++)" loop */
	    for(i=0;i<16;i++)
	      {
		strcpy( Tape->Chan[i]->Symbol.Mod_Name, "*" ); 
			/* first few are global */
	      }
	    strcpy( Tape->Chan[ 0]->Symbol.Par_Name, 
		    "generic_.geocentric_position_v[0]");
	    strcpy( Tape->Chan[ 1]->Symbol.Par_Name, 
		    "generic_.geocentric_position_v[1]");
	    strcpy( Tape->Chan[ 2]->Symbol.Par_Name, 
		    "generic_.geocentric_position_v[2]");
	    strcpy( Tape->Chan[ 3]->Symbol.Par_Name, 
		    "generic_.v_local_v[0]");
	    strcpy( Tape->Chan[ 4]->Symbol.Par_Name, 
		    "generic_.v_local_v[1]");
	    strcpy( Tape->Chan[ 5]->Symbol.Par_Name, 
		    "generic_.v_local_v[2]");
	    strcpy( Tape->Chan[ 6]->Symbol.Par_Name, 
		    "generic_.euler_angles_v[0]");
	    strcpy( Tape->Chan[ 7]->Symbol.Par_Name, 
		    "generic_.euler_angles_v[1]");
	    strcpy( Tape->Chan[ 8]->Symbol.Par_Name, 
		    "generic_.euler_angles_v[2]");
	    strcpy( Tape->Chan[ 9]->Symbol.Par_Name, 
		    "generic_.omega_body_v[0]");
	    strcpy( Tape->Chan[10]->Symbol.Par_Name, 
		    "generic_.omega_body_v[1]");
	    strcpy( Tape->Chan[11]->Symbol.Par_Name, 
		    "generic_.omega_body_v[2]");
	    strcpy( Tape->Chan[12]->Symbol.Par_Name, 
		    "generic_.earth_position_angle");
	    strcpy( Tape->Chan[13]->Symbol.Par_Name, 
		    "generic_.d_cg_rwy_local_v[0]");
	    strcpy( Tape->Chan[14]->Symbol.Par_Name, 
		    "generic_.d_cg_rwy_local_v[1]");
	    strcpy( Tape->Chan[15]->Symbol.Par_Name, 
		    "generic_.d_cg_rwy_local_v[2]");
    
    /* control positions hardwired as well */
    
	    strcpy( Tape->Chan[16]->Symbol.Mod_Name, "*");
	    strcpy( Tape->Chan[16]->Symbol.Par_Name, "cockpit_.long_stick");
	    strcpy( Tape->Chan[17]->Symbol.Mod_Name, "*");
	    strcpy( Tape->Chan[17]->Symbol.Par_Name, "cockpit_.lat_stick");
	    strcpy( Tape->Chan[18]->Symbol.Mod_Name, "*");
	    strcpy( Tape->Chan[18]->Symbol.Par_Name, "cockpit_.rudder_pedal");
	
	    for(i=0;i<Tape->Num_Chan;i++)
	    {
		result = ls_findsym( Tape->Chan[i]->Symbol.Mod_Name, 
				     Tape->Chan[i]->Symbol.Par_Name,    
			  (char **) &Tape->Chan[i]->Symbol.Addr, 
				    &Tape->Chan[i]->Symbol.Par_Type  );
		if (result)
		  {
		      ls_print_findsym_error( result, 
					      Tape->Chan[i]->Symbol.Mod_Name,
					      Tape->Chan[i]->Symbol.Par_Name );
		      Tape->Chan[i]->Symbol.Addr = NIL_POINTER;
		  } /* end of result non-zero path */
	      } /* end of "for(i=0;i<Tape->Num_Chan;i++)" loop */
	      
	      /* now rename fixed channels to something useful */
	      strcpy( Tape->Chan[ 0]->Symbol.Alias,  "geocLat");
	      strcpy( Tape->Chan[ 1]->Symbol.Alias,  "geocLon");
	      strcpy( Tape->Chan[ 2]->Symbol.Alias,  "radiusV");
	      strcpy( Tape->Chan[ 3]->Symbol.Alias,  "v_north");
	      strcpy( Tape->Chan[ 4]->Symbol.Alias,  "v_east");
	      strcpy( Tape->Chan[ 5]->Symbol.Alias,  "v_down");
	      strcpy( Tape->Chan[ 6]->Symbol.Alias,  "phi_r");
	      strcpy( Tape->Chan[ 7]->Symbol.Alias,  "theta_r");
	      strcpy( Tape->Chan[ 8]->Symbol.Alias,  "psi_r");
	      strcpy( Tape->Chan[ 9]->Symbol.Alias,  "pb");
	      strcpy( Tape->Chan[10]->Symbol.Alias,  "qb");
	      strcpy( Tape->Chan[11]->Symbol.Alias,  "rb");
	      strcpy( Tape->Chan[12]->Symbol.Alias,  "epa");
	      strcpy( Tape->Chan[13]->Symbol.Alias,  "X");
	      strcpy( Tape->Chan[14]->Symbol.Alias,  "Y");
	      strcpy( Tape->Chan[15]->Symbol.Alias,  "Z");
	      strcpy( Tape->Chan[16]->Symbol.Alias,  "long_stk");
	      strcpy( Tape->Chan[17]->Symbol.Alias,  "lat_stk");
	      strcpy( Tape->Chan[18]->Symbol.Alias,  "rud_ped");
	} /* end of "(Tape != NIL_POINTER)" path */
      } /* end of "(!channels_allocated)" path */

} /* end of ls_record_channels_init() */


void ls_record_alloc_storage()
{
    int i;
    static int storage_allocated = 0;

    if (!storage_allocated && !noTape)
	{
	    Tape->Length = sim_control_.time_slices;
	    Tape->T_Stamp = (SCALAR *) malloc( Tape->Length*sizeof( SCALAR ) );
	    if (Tape->T_Stamp == NIL_POINTER)
		{
		    fprintf(stderr, "ls_tape: memory allocation error\n");
		    for(i=0;i<Tape->Num_Chan;i++) free(Tape->Chan[i]);
		    free(Tape);
		    noTape = -1;
		    return;
		} /* end of "if (Tape->T_Stamp == NIL_POINTER)" path */

	    for(i=0;i<Tape->Num_Chan;i++)
		{
		    Tape->Chan[i]->Data = (SCALAR *) malloc( Tape->Length*sizeof( SCALAR ) );
		    if (Tape->Chan[i]->Data == NIL_POINTER)
			{
			    fprintf(stderr, "ls_tape: memory allocation error\n");
			    noTape = -1;
			    free(Tape->Chan[i]);
			    for(;i>0;--i) {free(Tape->Chan[i]->Data); free(Tape->Chan[i]);}
			    free(Tape);
			    return;
			} /* end of "if (Tape->Chan[i]->Data == NIL_POINTER)" path */
		} /* end of for(i=0;i<Tape->Num_Chan;i++) loop */
	}
}

char *ls_record_get_set(char *buffer, char *eob)
/* This routine parses the settings file for "tape" entries. */
{

    static char *fac_name = FACILITY_NAME_STRING;
    char *bufptr, **lasts, *nullptr, null = '\0';
    int n, ver, i, error;

    nullptr = &null;
    lasts = &nullptr;
    abort = 0;


    n = sscanf(buffer, "%s", line);
    if (n == 0) return 0L;
    if (strncasecmp( fac_name, line, strlen(fac_name) )) return 0L;

    bufptr = strtok_r( buffer+strlen(fac_name)+1, "\n", lasts);
    if (bufptr == 0) return 0L;

    sscanf( bufptr, "%d", &ver );
    if (ver != CURRENT_VERSION) return 0L;

    ls_record_channels_init();

    while( !abort && (eob > bufptr))
      {
	bufptr = strtok_r( 0L, "\n", lasts );
	if (bufptr == 0) return 0L;
	if (strncasecmp( bufptr, "end", 3) == 0) break;

	sscanf( bufptr, "%s", line );
	if (line[0] != '#') /* ignore comments */
	  {
	    num = sscanf( bufptr, "%s %s", mod_name, par_name );
	    if ((num == 2) && (Tape->Num_Chan < MAX_TAPE_CHANNELS))
		{	/* add channel */
		    i = Tape->Num_Chan;
		    Tape->Chan[i] = (CHANNEL *) malloc( sizeof( CHANNEL ) );
		    if (Tape->Chan[Tape->Num_Chan] == NIL_POINTER)
			{
			    fprintf(stderr, "ls_tape: memory allocation error\n");
			    abort = -1;
			} 
		    else
			{

			    /* initialize to NaN */
			    Tape->Chan[i]->Max_value = (SCALAR) sqrt(-3.);
			    Tape->Chan[i]->Min_value = (SCALAR) sqrt(-3.);
			    strcpy( Tape->Chan[i]->Symbol.Mod_Name, 
				    mod_name);
			    strcpy( Tape->Chan[i]->Symbol.Par_Name, 
				    par_name);
			    (void) ls_get_sym_val( &Tape->Chan[i]->Symbol, &error );
			    if (error) Tape->Chan[i]->Symbol.Addr = 0L;
			    Tape->Num_Chan++;

			}
		}
	  }
      }
    bufptr = *lasts;
    return bufptr;
}


void ls_record_put_set( FILE *fp )
{
    int i;

    if (fp==0) return;
    fprintf(fp, "\n");
    fprintf(fp, "#==============================  %s\n", FACILITY_NAME_STRING);
    fprintf(fp, "\n");
    fprintf(fp, FACILITY_NAME_STRING);
    fprintf(fp, "\n");
    fprintf(fp, "%04d\n", CURRENT_VERSION);
    for (i=HARDWIRED; i<Tape->Num_Chan; i++)
	fprintf(fp, "    %s	%s\n", 
		Tape->Chan[i]->Symbol.Mod_Name,
		Tape->Chan[i]->Symbol.Par_Name );
    fprintf(fp, "end\n");
    return;
}

ls_record()
{
    static int 	inited = 0;
    SCALAR	value;
    static short interp;
    int i, result;
    
    if (inited == 0)
    {
	inited = -1;
	
	interp = INTERP;
	ls_record_channels_init();	/* make sure we've set up the Tape header & channels */
	ls_record_alloc_storage();	/* then allocate data storage for each channel */
    
    } /* end of initialization section */
 
 
    
    if( !noTape )		/* Run time section */
      {				/* Advance pointers */
	Tape->Current = Tape->Next;
	Tape->Last = Tape->Current;
	if (Tape->Current == Tape->First) Tape->First++;
	if (Tape->First >= Tape->Length) Tape->First = 0;
	if (Tape->First < 0) Tape->First = 0; /* To handle startup */

	Tape->T_Stamp[ Tape->Current ] = Simtime;   /* save time stamp */
	    
	for (i=0;i<Tape->Num_Chan;i++)
	    if (Tape->Chan[i]->Symbol.Addr)
		{
		    value = ls_get_double( Tape->Chan[i]->Symbol.Par_Type, Tape->Chan[i]->Symbol.Addr );
		    if ( !( value <= Tape->Chan[i]->Max_value ) ) Tape->Chan[i]->Max_value = value;
		    if ( !( value >= Tape->Chan[i]->Min_value ) ) Tape->Chan[i]->Min_value = value;
		    Tape->Chan[i]->Data[ Tape->Current ] = value;
		} /* end of for(i=0;i<Tape->Num_Chan;i++) loop */

	/* Only advance Next pointer if this is a 'keeper' frame */
	    
	if (interp == INTERP) Tape->Next++;
	if (Tape->Next >= Tape->Length) Tape->Next = 0;
	if (--interp <= 0) interp = INTERP;
      } /* end of run time section */
  }
