/***************************************************************************

	TITLE:	ls_ifgl.c
	
----------------------------------------------------------------------------

	FUNCTION:	Human interface for real-time runs of LaRCSIM
	                models, using Silicon Graphics IRIS workstation.

----------------------------------------------------------------------------

	MODULE STATUS:	Developmental

----------------------------------------------------------------------------

	GENEALOGY:	Created 921230 by Bruce Jackson.

----------------------------------------------------------------------------

	DESIGNED BY:	EBJ
	
	CODED BY:	EBJ
	
	MAINTAINED BY:	EBJ

----------------------------------------------------------------------------

	MODIFICATION HISTORY:
	
	DATE	PURPOSE						BY

	930105  Added help menu capability.                     EBJ
	930111  Added support for passing program name;
                changed "N" to "G" on HUD display.              EBJ
	930315  Added dummy routine ls_cockpit_exit() for compatibility
		with alternate "curses" cockpit interface.
		Also added dummy routine ls_pause().		EBJ
	930701	Added bullet model.				EBJ
	930802	Added dummy routines for synchronization ls_sync, ls_resync,
		and ls_unsync(), since GL does sync with graphics calls.  EBJ
	930826	Added interface to VMIC 3114 board to read blue
		cockpit stick... sort of kludgey for now.	EBJ
	930921	Benchmarking graphics reveals the following truths for
		the current IRIS ONYX/VTX hardware:
		
	    -	subpixel(TRUE) has little effect
	    -	calling swapbuffers() as the last thing in this routine is good
	    -	RGBMode is faster than CMAP mode!
	    -	v3f() is faster than v3i() !
		
	931215	Added logic for discrete inputs			EBJ
	931217	It was inevitable; now we've got a building to blow up.
	931220	Added cockpit structure for passing switch positions
								EBJ
	931221	Added call to swapinterval() to slow graphics to 20 Hz.
		This was necessary to maintain a steady sim update rate, 
		since graphics appear to be raster-fill rate limited, 
		and slowdowns are evident when the runway environment fills
		even a WINDOWMARGIN of 80 is specified.		    EBJ
	940105	Renamed this module "ls_ifgl.c" from "ls_glcockpit.c"
		for consistency with other interface routines.	EBJ
	940106  Removed all the old REALSTICK logic for compatibility with
		new sim_control structure; also moved the getwscrn() call to 
		after the first winopen() call to try to fix an error when
		running over the network.				EBJ
	940111	Changed include file from ls_eom.h to ls_types and ls_generic
	940204	Removed ls_sync() and associated dummy routines; real ones
		will be used instead and disabled by control flags if necessary.
	940216	Added hud color variation to signal frame overrun.	EBJ
	940506	Added support for interp. of sim_control_.debug flag	EBJ

<<<<<<< ls_ifgl.c
        940824  Added heading to HUD and centerline to runway.  MLB
        940825  Vsi and navigation information added to HUD.  MLB

	950314	Made VSI use vertical velocity & offsets; HUD now shows global
		variable cockpit_.throttle_pct.				EBJ
	950316	Increased world size to 400x400 nm; grid spacing set at
		5 nm.							EBJ
	950321	Changed 'A' and 'S' key to drive throttle_pct.		EBJ

$Header: /aces/larcsim/dev/RCS/ls_ifgl.c,v 1.15 1995/03/29 16:11:10 bjax Exp $
=======
$Header: /aces/larcsim/dev/RCS/ls_ifgl.c,v 1.15 1995/03/29 16:11:10 bjax Exp $
>>>>>>> 1.9
$Log: ls_ifgl.c,v $
 * Revision 1.15  1995/03/29  16:11:10  bjax
 * Added calculation to darken sky as we go higher. EBJ
 *
 * Revision 1.14  1995/03/21  13:44:33  bjax
 * Changed use of 'A' and 'S' keys to drive Throttle_pct.
 *
 * Revision 1.13  1995/03/16  13:33:56  bjax
 * Fixed N-S/E-W readouts to nm. EBJ
 *
 * Revision 1.12  1995/03/15  12:18:28  bjax
 * Moved 'paused' variable to sim_control_ common block; reworked
 * pause logic so HUD indicates paused condition more accurately;
 * changed VSI & lat/long readouts to use correct simulation generic
 * variables V_down, D_cg_north_rwy, etc.; changed throttle readout
 * from Throttle[3] to new global variable Throttle_pct; added call
 * to ls_save_current_as_ic() if trim is successful; removed calls
 * to sleep() for GLmouse mode (since sim now starts in paused state).
 * EBJ
 *
 * Revision 1.11  1995/02/27  20:59:46  bjax
 * Added 'T' key that fires off a trim request.
 *
 * Revision 1.9  1994/05/13  17:23:57  bjax
 * Moved around some graphics calls after using gldebug;
 * add'l checks for debug mode prior to calls to swapbuffers();
 *
 * Revision 1.8  1994/05/13  13:16:15  bjax
 * Uncommented-in call to ls_ACES to read stick; this is needed
 * to read the buttons so that a second click of the 'pause' button
 * is read (so that the 'pause' can be cancelled).
 *
 * Revision 1.7  1994/05/10  20:11:48  bjax
 * Commented out call to ls_ACES; this call was moved to main routine
 * to speed up stick reading.  Graphics really should be moved
 * to async process.
 *
 * Revision 1.6  1994/05/06  15:37:30  bjax
 * Removed local "db" flag, and substituted sim_control_.debug flag.
 *
 * Revision 1.5  1994/02/16  12:59:25  bjax
 * Added logic to allow aborts while paused; use HUD to signal overrun.
 *
 * Revision 1.4  1994/02/04  12:59:34  bjax
 * Removed ls_sync() and associated dummy routines.
 *
 * Revision 1.3  1994/01/11  19:07:55  bjax
 * Fixed include files.
 *
 * Revision 1.2  1994/01/11  18:30:05  bjax
 * Removed REALSTICK macro definition; this duty now performed by
 * sim_control_.sim_type; changed logic for invoking mouse stuff;
 * added explicit return result to getgdesc(GD_TIMERHZ) call.
 *
 * Revision 1.1  1994/01/05  19:57:24  bjax
 * Initial revision
 *
 * Revision 1.9.2.7  1993/12/21  17:40:15  bjax
 * Added call to swapinterval to slow screen update rate to 20 Hz for
 * consistent performance.  This gave time to run with antialiasing and
 * probably z-buffer, although z-buffer causes some problems and isn't worth
 * the effort.
 *
 * Revision 1.9.2.6  1993/12/21  14:34:15  bjax
 * Modified to use new cockpit interface; added speedbrake & gear switches.
 *
 * Revision 1.9.2.4  1993/12/20  16:56:44  bjax
 * Matched the sign convention for target & weapons. EBJ
 *
 * Revision 1.9.2.3  1993/12/17  23:19:24  bjax
 * Building added.
 *
 * Revision 1.9.2.2  1993/12/17  19:15:01  bjax
 * Same version as 1.9.1.13; started new branch.  EBJ
 *
 * Revision 1.9.1.13  1993/12/15  13:52:51  bjax
 * Added support for discrete inputs. EBJ
 *
 * Revision 1.9.1.12  1993/09/23  16:58:48  bjax
 * This version uses multiple polygons to represent the ground, in an
 * (unsuccessful) attempt to allow full-screen 60Hz operation. EBJ
 *
 * Revision 1.9.1.11  1993/09/21  17:01:33  bjax
 * More tuning for graphics: added call to subpixel(), changed almost all
 * calls from v3i() to v3f()'s (bullets are still v3i's); slightly
 * shrunk the window margin size from 100 to 80; increased the gridlines
 * from 9 to 24; moved call to swapbuffers() to the end of the routine (this
 * had a significant improvement in performance). -- EBJ
 *
 * Revision 1.9.1.10  1993/09/17  17:58:11  bjax
 * Okay, lots of changes:
 *    -- Converted most of the ground elements from floating to long int
 *       vertices, in an attempt to avoid loss of 60 Hz sync time
 *    -- Now drawing the ground as a polygon, since circf() call seemed
 *       to take a long time (this might be revisited later, but square ground
 *       looks fine at low altitude)
 *    -- Turned of Gourand shading; switched to flat shading algorithm to
 *       speed things up
 *    -- Went back to non-full screen window, since performance (and 60 Hz
 *       operation) seems to depend EXTREMELY strongly on the number of pixels
 *       in the window.  A margin all around of 100 pixels wide lets the
 *       whole thing run at 60 Hz quite nicely, apparently.
 *    -- This version has the cursor turned off.
 *    -- Added calls to gflush() after each frame is drawn, just to 'make sure'
 *    -- Added call to gexit() when ls_cockpit_exit() is called, just
 *       to 'make sure' we do things right.
 *
 * Revision 1.9.1.9  1993/09/15  18:27:21  bjax
 * This version has stick & throttle read from routine ls_readstick. EBJ
 *
 * Revision 1.9.1.8  1993/09/01  19:27:36  bjax
 * Includes stick interface protocode.
 *
 * Revision 1.9.1.7  1993/08/03  19:04:24  bjax
 * Remember: compile first, then archive!  bjax
 *
 * Revision 1.9.1.6  1993/08/03  19:03:27  bjax
 * Okay, okay, I think I got all the merges.  EBJ
 *
 * Revision 1.9.1.5  1993/08/03  19:02:37  bjax
 * Oops another bug.  EBJ
 *
 * Revision 1.9.1.4  1993/08/03  19:00:34  bjax
 * Finally got the bullets back in , I think.  EBJ
 *
 * Revision 1.9.1.3  1993/08/03  16:33:26  bjax
 * Oops. Fixed problem, I think.  EBJ
 *
 * Revision 1.9.1.2  1993/08/03  16:31:37  bjax
 * Added dummy sync routines.  EBJ
 *
 * Revision 1.9  1993/07/16  19:29:38  bjax
 * Changed RWYLENGTH and RWYWIDTH to floats (added .).  Relocated calls to
 * subpixel(), blendfunction(), and linesmooth() prior to gconfig(), and
 * commented them out after observing performance hit.      EBJ
 *
 * Revision 1.8.1.2  1993/07/02  17:02:06  bjax
 * Got 'em to work.  Bullets now flying! ... and impacting ground.  EBJ
 *
 * Revision 1.8.1.1  93/07/02  13:49:49  bjax
 * This version is intended to have bullets!  Not yet complete.
 * 
 * Revision 1.8  93/03/15  08:57:44  bjax
 * Added dummy ls_cockpit_exit() and ls_pause() for compatibility with
 * sun version of ls_main().   EBJ
 * 
 * Revision 1.7  93/01/12  08:00:41  bjax
 * Added program name string for window display; changed "N" to "G" on hud.
 * 
 * Revision 1.6  93/01/06  09:56:09  bjax
 * Corrected Revision string.
 * 
 * Revision 1.5  93/01/06  09:50:24  bjax
 * Added help menu.
 * 
 * Revision 1.4  92/12/30  14:15:16  bjax
 * Reversed calls to rotate when DELKEY or PAGEDOWNKEY are depressed.
 * 
 * Revision 1.3  92/12/30  14:12:30  bjax
 * Changed call from "init" to "ls_init".
 * 
 * Revision 1.2  92/12/30  13:52:21  bjax
 * Changed to point to navion.h in navion directory.
 * 
 * Revision 1.1  92/12/30  13:18:18  bjax
 * Initial revision
 * 
 * Revision 1.2  93/12/31  10:43:10  bjax
 * Added End, Delete, Page Up/Down keys for view selection.
 * 	

----------------------------------------------------------------------------

	REFERENCES:
	
	The GL world view is oriented +X forward, +Y to left,  and +Z up.

----------------------------------------------------------------------------

	CALLED BY:

----------------------------------------------------------------------------

	CALLS TO:

----------------------------------------------------------------------------

	INPUTS:

----------------------------------------------------------------------------

	OUTPUTS:

--------------------------------------------------------------------------*/
/* cockpit.c - performs simple user interface on GL */
#include <gl/gl.h>
#include <gl/device.h>
#include <stdio.h>
#include <math.h>
#include "ls_types.h"
#include "ls_generic.h"
#include "ls_constants.h"
#include "ls_sim_control.h"
#include "ls_cockpit.h"

#define X	0
#define Y	1

#define RGBMODE (comment this line out for colormap mode)

#define WINDOWMARGIN 0

#ifdef GLPROF
#define OBJ(x) glprof_object(x);
#else
#define OBJ(x)
#endif

/* ground grid. MAXGRID is +/- ft from rwy threshold */
#define MAXGRID 200.*6076.
#define GRIDLINES 40	/* 5 nm grids */

#define ATMOS_MAX 420000.	/* max altitude of blue sky */

/* stick (mouse) and discrete gearing */
#define LON_SCALE 0.3
#define LAT_SCALE 0.3
#define DELTAARROWAMT 20
#define DELTARUDDER 0.01

/* top-to-bottom viewing angle */
#define YWINDOWANGLE 30.

/* Head-up Display geometry: X is +left, Y is + up */
#define HUDDIST 200
#define HUDBORESIGHTSIZE 1
#define HUDLADDERWIDTH 20
#define HUDLADDERWINGLETLENGTH 2
#define HUDVELX 12
#define HUDVELY  0
#define HUDALTX -4
#define HUDALTY  0
#define HUDDATAX 30
#define HUDMACHY   -4
#define HUDALPHAY  -6
#define HUDNZY     -8
#define HUDTY     -10

/* runway geometry */
#define RWYLENGTH 15000.
#define RWYWIDTH 300.

/* Name of program that invoked this application */
extern char *progname;

extern SCALAR Simtime;

/* variables with FILE visibility */
static Matrix mhome;
static int trigger = 0;

typedef struct
    {
	double x, y, z, killradius2;
	int hit;
    } target,  *ptarget;

static int targets_alive = 0;

#define MAXTARGETS 10

static ptarget targetlist[MAXTARGETS];

/* cockpit interface data block  - defn's in ls_cockpit.h */

COCKPIT cockpit_;



void drawsky()
{

#define SKYCOLOR 3*64-13
  static short skycolor0[3]    = {  63,  63, 255 };
  short skycolor[3];
  int i;
  float factor;
  

  OBJ("drawsky"); /* marker for GLProf */

  /* to provide sky darkening - goes to black at ATMOS_MAX */

  factor = Altitude/ATMOS_MAX;
  if (factor < 0.) factor = 0.0;
  if (factor > 1.) factor = 1.0;
  factor = sqrt( 1.0 - pow(factor,2) );
  for(i=0;i<3;i++) skycolor[i] = factor*skycolor0[i];
  

#ifdef RGBMODE
  c3s(skycolor);
#else
  color(SKYCOLOR);
#endif
  clear();
}



void drawground()
{

#define GROUNDCOLOR 59
  static short groundcolor[3] = {   8, 127,   8 };
#define GROUNDSIZE MAXGRID
  int i, j;
  float vert0[3], vert1[3];
  float grdspace;
  float d;

  OBJ("drawground");  /* marker for GLProf */

#ifdef RGBMODE
  c3s(groundcolor);
#else
  color(GROUNDCOLOR);
#endif
  grdspace = MAXGRID/GRIDLINES;

  vert0[2] = Runway_altitude;
  vert1[2] = Runway_altitude;

  /* outer loop - south to north progression of quadrilateral strips */
  for(i=0; i<2*GRIDLINES; i++)
  {
    bgnqstrip();
    
      vert0[1] = -MAXGRID+(i*grdspace);  /* southern edge coordinate */
      vert1[1] = vert0[1] + grdspace;    /* northern edge coordinate */
      /* inner loop - east to west quadstrips */
      for(j=2*GRIDLINES; j>=0; j--) 
      {
	  vert0[0] = -MAXGRID+(j*grdspace);
	  vert1[0] = vert0[0];
	  v3f(vert0);
	  v3f(vert1);
      }
      
    endqstrip();
  }
}



void drawgrid()
{
#define GRIDCOLOR BLACK
  static short gridcolor[3]   = {   0,   0,   0 };
  static float grid[5][3] = {
    { -MAXGRID, -MAXGRID, 0. },
    {  MAXGRID, -MAXGRID, 0. },
    {  MAXGRID,  MAXGRID, 0. },
    { -MAXGRID,  MAXGRID, 0. },
    { -MAXGRID, -MAXGRID, 0. }
  };
  static inited = 0;
  static float grdspace;

  int i;
  float d;
  float vert[3];

  OBJ("drawgrid");  /* marker for GLProf */

  if (!inited)
  {
    inited = -1;
    grdspace = MAXGRID/GRIDLINES;
    grid[0][2] = Runway_altitude+0.25;
    grid[1][2] = Runway_altitude+0.25;
    grid[2][2] = Runway_altitude+0.25;
    grid[3][2] = Runway_altitude+0.25;
    grid[4][2] = Runway_altitude+0.25;
  }

  /* add gridlines */

#ifdef RGBMODE
  c3s(gridcolor);
#else
  color(GRIDCOLOR);
#endif

    
  /* outline playing field */
  bgnline();
    for(i=0;i<5;i++) v3f(grid[i]);
  endline();

  /* draw gridlines */
  vert[2] = Runway_altitude;
  for(i=1; i<2*GRIDLINES; i++)
  {
    d = -MAXGRID+(i*grdspace);

    /* draw N-S gridlines */
    vert[0] = d;
    bgnline();
      vert[1] = -MAXGRID;  v3f( vert );
      vert[1] =  MAXGRID;  v3f( vert );
    endline();

    /* draw E-W gridlines */
    vert[1] = d;
    bgnline();
      vert[0] = -MAXGRID;  v3f( vert );
      vert[0] =  MAXGRID;  v3f( vert );
    endline();
  }
}



void drawrwy()
{
#define RWYCOLOR 8
  static short rwycolor[3]    = { 127, 127, 127 };
  static float vert1_track[3],vert2_track[3];
  static short linecolor[3] = { 0, 0, 0 };
  static float rwy[5][3] = {
    { 0,          RWYWIDTH/2, 0. },
    { RWYLENGTH,  RWYWIDTH/2, 0. },
    { RWYLENGTH, -RWYWIDTH/2, 0. },
    { 0,         -RWYWIDTH/2, 0. },
    { 0,          RWYWIDTH/2, 0. }
  };
  static inited = 0;

  int i;

  OBJ("drawrwy");  /* marker for GLProf */

    if(!inited)
    {
	inited = -1;
	rwy[0][2] = Runway_altitude+0.5;
	rwy[1][2] = Runway_altitude+0.5;
	rwy[2][2] = Runway_altitude+0.5;
	rwy[3][2] = Runway_altitude+0.5;
	rwy[4][2] = Runway_altitude+0.5;
    }

#ifdef RGBMODE
  c3s(rwycolor);
#else
  color(RWYCOLOR);
#endif

  /* add a runway */
  bgnpolygon();
  for(i=0;i<5;i++)
    v3f(rwy[i]);
  endpolygon();

  /* outline rwy */
#ifdef RGBMODE
  cpack(0x00000000);/* black */
#else
  color(BLACK);
#endif
  bgnline();
  for(i=0;i<5;i++)
    v3f(rwy[i]);
  endline();

/* rwy centerline */
vert1_track[0] = 1000.0;
vert1_track[1] = 0.0;
vert1_track[2] = 0.0;
vert2_track[0] = 1500.0;
vert2_track[1] = 0.0;
vert2_track[2] = 0.0;

/* draw rwy centerline */
/* deflinestyle(1,0xFFFD);*/
/* setlinestyle(1);*/
linewidth(3);
if(Altitude > 50.) linewidth(2);
if(Altitude > 500.)linewidth(1);
c3s(linecolor);
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 2000.0;
vert2_track[0] = 2500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();


vert1_track[0] = 3000.0;
vert2_track[0] = 3500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 4000.0;
vert2_track[0] = 4500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 5000.0;
vert2_track[0] = 5500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 6000.0;
vert2_track[0] = 6500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 7000.0;
vert2_track[0] = 7500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 8000.0;
vert2_track[0] = 8500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 9000.0;
vert2_track[0] = 9500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 10000.0;
vert2_track[0] = 10500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 11000.0;
vert2_track[0] = 11500.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 12000.0;
vert2_track[0] = 12750.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();

vert1_track[0] = 13250.0;
vert2_track[0] = 14000.0;
bgnline();
v3f(vert1_track);
v3f(vert2_track);
endline();


linewidth(1);
/*setlinestyle(0);*/
}

void drawtargets()
{
#define TARGCOLOR 3
#define TARGSIZE 20.
#define TARG_X_IC 7500.
#define TARG_Y_IC 500.
#define TARG_Z_IC 0.
  static short targcolor[3]    = { 30, 30, 20 };
  static struct {
    int exists;
    double x, y, z;
    double xdot, ydot, zdot;
    double p, q, r;
    double phi, theta, psi;
    } piece[5];

  static float targ_geom[5][5][3] =
  { 
      {	/* east face */
	{  TARGSIZE, -TARGSIZE,  0.}, 
	{  TARGSIZE, -TARGSIZE,  2.0*TARGSIZE }, 
	{ -TARGSIZE, -TARGSIZE,  2.0*TARGSIZE }, 
	{ -TARGSIZE, -TARGSIZE,  0.}, 
	{  TARGSIZE, -TARGSIZE,  0.}, 	  
      }, 
      {	/* north face */
	{  TARGSIZE,  TARGSIZE,  0.}, 
	{  TARGSIZE,  TARGSIZE,  2.0*TARGSIZE }, 
	{  TARGSIZE, -TARGSIZE,  2.0*TARGSIZE }, 
	{  TARGSIZE, -TARGSIZE,  0.}, 
	{  TARGSIZE,  TARGSIZE,  0.}, 	  
      }, 
      {	/* west face */
	{ -TARGSIZE,  TARGSIZE,  0.}, 
	{ -TARGSIZE,  TARGSIZE,  2.0*TARGSIZE }, 
	{  TARGSIZE,  TARGSIZE,  2.0*TARGSIZE }, 
	{  TARGSIZE,  TARGSIZE,  0.}, 
	{ -TARGSIZE,  TARGSIZE,  0.}, 	  
      }, 
      {	/* south face */
	{ -TARGSIZE, -TARGSIZE,  0.}, 
	{ -TARGSIZE, -TARGSIZE,  2.0*TARGSIZE }, 
	{ -TARGSIZE,  TARGSIZE,  2.0*TARGSIZE }, 
	{ -TARGSIZE,  TARGSIZE,  0.}, 
	{ -TARGSIZE, -TARGSIZE,  0.}, 	  
      }, 
      {	/* top face */
	{  TARGSIZE, -TARGSIZE,  2.0*TARGSIZE }, 
	{  TARGSIZE,  TARGSIZE,  2.0*TARGSIZE }, 
	{ -TARGSIZE,  TARGSIZE,  2.0*TARGSIZE }, 
	{ -TARGSIZE, -TARGSIZE,  2.0*TARGSIZE }, 
	{  TARGSIZE, -TARGSIZE,  2.0*TARGSIZE }, 	  
      }
  };
  static inited = 0;
  static hit_init = 0;
  static double oldSimtime = 5.;
  static double targ_x, targ_y, targ_z;
  double dt;
  int i, j;

  OBJ("drawtargets");  /* marker for GLProf */

    if(!inited)
    {
	inited = -1;
	targ_x = TARG_X_IC;
	targ_y = TARG_Y_IC;
	targ_z = TARG_Z_IC;
	targetlist[0] = (ptarget) malloc(sizeof(target));
	if (targetlist[0] == 0L) return;
	targetlist[0]->x = targ_x;
	targetlist[0]->y = targ_y;
	targetlist[0]->z = targ_z+TARGSIZE;
	targetlist[0]->killradius2=4*TARGSIZE*TARGSIZE;
	targetlist[0]->hit = 0;
    }
    if (oldSimtime > Simtime)
	{
	    targets_alive = 1;
	    targetlist[0]->hit = 0;
	    oldSimtime = Simtime;
	    hit_init = 0;
	    for(i = 0; i<5; i++)
	    {
		piece[i].exists = 0;
		piece[i].x = targ_x;
		piece[i].y = targ_y;
		piece[i].z = targ_z;
		piece[i].xdot = 0.;
		piece[i].ydot = 0.;
		piece[i].zdot = 0.;
		piece[i].p = 0.;
		piece[i].q = 0.;
		piece[i].r = 0.;
		piece[i].phi = 0.0;
		piece[i].theta = 0.0;
		piece[i].psi = 0.0;
	    }
	}
    
    dt = Simtime - oldSimtime;
    oldSimtime = Simtime;


#ifdef RGBMODE
  c3s(targcolor);
#else
  color(TARGCOLOR);
#endif

  /* draw the target */
  if (!targetlist[0]->hit)
  {
    pushmatrix();
    translate( targ_x, -targ_y, targ_z );
    for(i = 0; i<5; i++)
    {
	bgnpolygon();
	for(j = 0; j<5; j++)
	{
	    v3f(targ_geom[i][j]);
	}
	endpolygon();
    }
    popmatrix();
  }
  else
  {
      if(!hit_init)
      {
	  hit_init=-1;

	  targets_alive--;
	  
	  piece[0].exists = 1;
	  piece[0].zdot = 100.;
	  piece[0].xdot = 0.;
	  piece[0].ydot = -300.;
	  piece[0].p = 20.;
	  piece[0].q = -10.;
	  piece[0].r = 300.;

	  piece[1].exists = 1;
	  piece[1].zdot = 50.;
	  piece[1].xdot = 300.;
	  piece[1].ydot = 10.;
	  piece[1].p = 200.;
	  piece[1].q = -20.;
	  piece[1].r = 2000.;

	  piece[2].exists = 1;
	  piece[2].zdot = 200.;
	  piece[2].xdot = 0.;
	  piece[2].ydot = 300.;
	  piece[2].p = -50.;
	  piece[2].q = 20.;
	  piece[2].r = 5000.;

	  piece[3].exists = 1;
	  piece[3].zdot = 100.;
	  piece[3].xdot = -300.;
	  piece[3].ydot = 10.;
	  piece[3].p = -200.;
	  piece[3].q = 20.;
	  piece[3].r = 200.;

	  piece[4].exists = 1;
	  piece[4].zdot = 100.;
	  piece[4].xdot = -3.;
	  piece[4].ydot = 10.;
	  piece[4].p = 200.;
	  piece[4].q = 20.;
	  piece[4].r = 2000.;
      }
      for(i = 0; i<5; i++)
      {
	  if (piece[i].exists)
	    {
		piece[i].zdot  = piece[i].zdot  - dt*Gravity;
		piece[i].x     = piece[i].x     + dt*piece[i].xdot;
		piece[i].y     = piece[i].y     + dt*piece[i].ydot;
		piece[i].z     = piece[i].z     + dt*piece[i].zdot;
		piece[i].phi   = piece[i].phi   + dt*piece[i].p;
		piece[i].theta = piece[i].theta + dt*piece[i].q;
		piece[i].psi   = piece[i].psi   + dt*piece[i].r;
		if(piece[i].z < Runway_altitude) piece[i].exists = 0;
		pushmatrix();
		translate( piece[i].x, -piece[i].y, piece[i].z );
		rotate( (int) piece[i].phi, 'x');
		rotate( (int) piece[i].theta, 'y');
		rotate( (int) piece[i].psi, 'z');
		bgnpolygon();
		for(j = 0; j<5; j++)
		{
		    v3f(targ_geom[i][j]);
		}
		endpolygon();
		popmatrix();
	  }
      }
  }
}



void drawhud( merect )

  Matrix merect;     /* erect matrix centered at eyepoint */

{
#define HUDCOLOR WHITE
  static short hudcolor[3]    = { 255, 255, 255 };
  static short slocolor[3]    = { 255, 0, 0 };
  static float hudboresight[4][3] = {
    { HUDDIST, -HUDBORESIGHTSIZE,  0 },
    { HUDDIST,  HUDBORESIGHTSIZE,  0 },
    { HUDDIST,  0, -HUDBORESIGHTSIZE },
    { HUDDIST,  0,  HUDBORESIGHTSIZE }
  };
  static float vvect[6][3] = {      /* velocity vector wing & tail */
    {   -HUDBORESIGHTSIZE,  0,  0 },
    { -2*HUDBORESIGHTSIZE,  0,  0 },
    {  0,    HUDBORESIGHTSIZE,  0 },
    {  0,  2*HUDBORESIGHTSIZE,  0 },
    {  0,   -HUDBORESIGHTSIZE,  0 },
    {  0, -2*HUDBORESIGHTSIZE,  0 }
  };
  static float pvect[6][3] = {      /* pitch ladder */
    { HUDDIST,  0.5*HUDLADDERWIDTH, 0}, 
    { HUDDIST,  0.5*HUDLADDERWIDTH, 0}, 
    { HUDDIST,  3*HUDBORESIGHTSIZE, 0},
    { HUDDIST, -0.5*HUDLADDERWIDTH, 0},
    { HUDDIST, -0.5*HUDLADDERWIDTH, 0},
    { HUDDIST, -3*HUDBORESIGHTSIZE, 0}
  };
  static float nvect[4][3] = {      /* nadir marker */
    {  HUDBORESIGHTSIZE,  HUDBORESIGHTSIZE, -HUDDIST },
    { -HUDBORESIGHTSIZE, -HUDBORESIGHTSIZE, -HUDDIST },
    {  HUDBORESIGHTSIZE, -HUDBORESIGHTSIZE, -HUDDIST },
    { -HUDBORESIGHTSIZE,  HUDBORESIGHTSIZE, -HUDDIST }
  };
  char *altstr="999999.9";
  char *velstr="999999.9";
  char *machstr= "M 9.99";
  char *alphstr= "A 99.9";
  char *nzstr  = "G 99.9";
  char *thrstr = "T 20%%";
  char *Psistr = "999.9";
  char *Vsistr="999999.9";
  char *Latstr="999999.9";
  char *Lonstr="999999.9";
  float ang, angrad;
  static float tyme;
  static float tymep;
  static float vsi;
  static float vsip;
  static float altp;

  
  OBJ("drawhud");  /* marker for GLProf */

  pushmatrix();   /* save hud centered matrix */

  /* note: at this point, screen coordinates are
     +X away from eye, +Y to left, and +Z up     */
  
#ifdef RGBMODE
  c3s(hudcolor);
#else
  color(HUDCOLOR);
#endif

  if( sim_control_.overrun || sim_control_.paused ) c3s(slocolor); /* signal less than real-time */
  /* draw bore sight */
  bgnline();
  {
    v3f(hudboresight[0]);
    v3f(hudboresight[1]);
  }
  endline();
  bgnline();
  {
    v3f(hudboresight[2]);
    v3f(hudboresight[3]);
  }
  endline();

  /* write alphanumeric data */
  cmovi(HUDDIST, HUDVELX, HUDVELY);
  sprintf(velstr, "%5.0f", V_equiv_kts);
  charstr(velstr);
  
  cmovi(HUDDIST, HUDALTX, HUDALTY);
  sprintf(altstr, "%5.0f", Altitude);
  charstr(altstr);

  cmovi(HUDDIST, HUDDATAX, HUDMACHY);
  sprintf(machstr, "M %3.2f", Mach_number);
  charstr(machstr);

  cmovi(HUDDIST, HUDDATAX, HUDALPHAY);
  sprintf(alphstr, "A %3.1f", Alpha*57.3);
  charstr(alphstr);

  cmovi(HUDDIST, HUDDATAX, HUDNZY);
  sprintf(nzstr, "G %3.2f", -N_Z_cg);
  charstr(nzstr);

  cmovi(HUDDIST, HUDDATAX, HUDTY);
  sprintf(thrstr, "T %3.0f%%", Throttle_pct*100.);
  charstr(thrstr);

  cmovi(HUDDIST, 1.5, 5.); /* add heading info to HUD -jbd*/
  sprintf(Psistr, "%2.0f", Psi*57.3);
  charstr(Psistr);
  
  cmovi(HUDDIST, -6., -4.); /* add vertical speed to HUD -mlb*/
  sprintf(Vsistr, "%5.0f", -V_down*60.);
  charstr(Vsistr);

  cmovi(HUDDIST, -20., -8.); /* add navigation to HUD -mlb*/
  sprintf(Latstr, "N-S %5.2f", D_pilot_north_of_rwy/6076.);
  charstr(Latstr);

  cmovi(HUDDIST, -20., -10.); /* add navigation to HUD -mlb*/
  sprintf(Lonstr, "E-W %5.2f", D_pilot_east_of_rwy/6076.);
  charstr(Lonstr);

  /* draw velocity vector */

  rot(-Beta*RAD_TO_DEG, 'z');
  rot(Alpha*RAD_TO_DEG, 'y');
  translate( HUDDIST, 0, 0);
  rotate(900, 'y');

  arc(0, 0, HUDBORESIGHTSIZE, 1, 3600);
  bgnline();
     v3f( vvect[0] );
     v3f( vvect[1] );
  endline();
  bgnline();
     v3f( vvect[2] );
     v3f( vvect[3] );
  endline();
  bgnline();
     v3f( vvect[4] );
     v3f( vvect[5] );
  endline();


  /* draw pitch ladder */

  loadmatrix(merect);    /* set up for eyepoint centered erect drawing */
  rot(-Psi*RAD_TO_DEG, 'z');
 
  /* draw horizon line */
  pvect[0][1] = HUDLADDERWIDTH;
  pvect[0][2] = 0;
  pvect[3][1] = -pvect[0][1];
  pvect[3][2] = 0;
  bgnline();
    v3f( pvect[0] );
    v3f( pvect[2] );
  endline();
  bgnline();
    v3f( pvect[3] );
    v3f( pvect[5] );
  endline();
  
  /* draw apex marker */
  pushmatrix();        /* save pitch ladder coordinates */  
  translate(0, 0, +HUDDIST);
  arc(0, 0, HUDBORESIGHTSIZE, 1, 3600);
  popmatrix();

  /* draw nadir marker */
  bgnline();
    v3f(nvect[0]);
    v3f(nvect[1]);
  endline();
  bgnline();
    v3f(nvect[2]);
    v3f(nvect[3]);
  endline();

  /* draw upper pitch ladder */
  pushmatrix();        /* save pitch ladder coordinates */
  for(ang=5.; ang < 85.1; ang=ang+5)
    {
      rotate(-50, 'y');
      angrad = ang*DEG_TO_RAD;
      pvect[0][1] = 0.5*HUDLADDERWIDTH + HUDLADDERWINGLETLENGTH*cos(angrad);
      pvect[0][2] = -HUDLADDERWINGLETLENGTH*sin(angrad);
      pvect[3][1] = -pvect[0][1];
      pvect[3][2] = pvect[0][2];
      bgnline();
        v3f( pvect[0] );
        v3f( pvect[1] );
        v3f( pvect[2] );
      endline();
      bgnline();
        v3f( pvect[3] );
        v3f( pvect[4] );
        v3f( pvect[5] );
      endline();
    }
  popmatrix();  /* restore pitch ladder coordinates */

  /* draw "backside" 15 degrees beyond apex */
  pushmatrix();        /* save pitch ladder coordinates */
  rotate(-900, 'y');
  for(ang=5.; ang < 15.1; ang=ang+5)
    {
      rotate(-50, 'y');
      angrad = (90 - ang)*DEG_TO_RAD;
      pvect[0][1] = 0.5*HUDLADDERWIDTH + HUDLADDERWINGLETLENGTH*cos(angrad);
      pvect[0][2] = HUDLADDERWINGLETLENGTH*sin(angrad);
      pvect[3][1] = -pvect[0][1];
      pvect[3][2] = pvect[0][2];
      bgnline();
        v3f( pvect[0] );
        v3f( pvect[1] );
        v3f( pvect[2] );
      endline();
      bgnline();
        v3f( pvect[3] );
        v3f( pvect[4] );
        v3f( pvect[5] );
      endline();
    }
  popmatrix();  /* restore pitch ladder coordinates */

  /* draw lower pitch ladder */
  pushmatrix();   /* save pitch ladder coordinates */
  for(ang=-5.; ang > -85.1; ang=ang-5)
    {
      rotate( 50, 'y');
      angrad = ang*DEG_TO_RAD;
      pvect[0][1] = 0.5*HUDLADDERWIDTH + HUDLADDERWINGLETLENGTH*cos(angrad);
      pvect[0][2] = -HUDLADDERWINGLETLENGTH*sin(angrad);
      pvect[3][1] = -pvect[0][1];
      pvect[3][2] = pvect[0][2];
      bgnline();
        v3f( pvect[0] );
        v3f( pvect[1] );
        v3f( pvect[2] );
      endline();
      bgnline();
        v3f( pvect[3] );
        v3f( pvect[4] );
        v3f( pvect[5] );
      endline();
    }
  popmatrix();  /* restore pitch ladder coordinates matrix */


  /* draw "backside" 15 degrees beyond nadir */
  pushmatrix();        /* save pitch ladder coordinates */
  rotate(900, 'y');
  for(ang=-5.; ang > -15.1; ang=ang-5)
    {
      rotate( 50, 'y');
      angrad = (-90 - ang)*DEG_TO_RAD;
      pvect[0][1] = 0.5*HUDLADDERWIDTH + HUDLADDERWINGLETLENGTH*cos(angrad);
      pvect[0][2] = HUDLADDERWINGLETLENGTH*sin(angrad);
      pvect[3][1] = -pvect[0][1];
      pvect[3][2] = pvect[0][2];
      bgnline();
        v3f( pvect[0] );
        v3f( pvect[1] );
        v3f( pvect[2] );
      endline();
      bgnline();
        v3f( pvect[3] );
        v3f( pvect[4] );
        v3f( pvect[5] );
      endline();
    }
  popmatrix();  /* restore pitch ladder coordinates */

  popmatrix();  /* restore HUD centered coordinates */
  
}



void drawweapons()
{
#define MAXBULLETS 50
#define FIREINTERVAL 0.10
#define LIFETIME 20.
#define MUZZLEVEL 2000.
#define BULLETSIZE 1
#define BULLETCOLOR 1
#define EXPLOSIONCOLOR 7
    static short bulletcolor[3] = {  255, 0,  0 };
    static short explosioncolor[3] = {  255, 255,  255 };
    static float bvect[4][3] = { /* bullet image */
	{  BULLETSIZE,  0,  0 }, 
	{  0, 0, BULLETSIZE   }, 
	{ -BULLETSIZE,  0,  0 }, 
	{  BULLETSIZE,  0,  0 }
    };
    static float expl[4][3] = { /* explosion image */
	{  10*BULLETSIZE,  0,  0 }, 
	{  0, 0, 10*BULLETSIZE   }, 
	{ -BULLETSIZE*10,  0,  0 }, 
	{  BULLETSIZE*10,  0,  0 }
    };

	
    typedef struct
	{
	    double age, xdot, ydot, zdot, x, y, z;
	} bullet,  *pbullet;

    static  int	bullets_away = 0;
    
    static pbullet bulletlist[MAXBULLETS];
    
    static double oldSimtime = 5.;
    static double lastFiredtime = 0.;
    double x_miss, y_miss, z_miss, miss2;
    double dt;
    int i, j;
    
    OBJ("drawweapons");  /* marker for GLProf */

    if (oldSimtime > Simtime)
	{
	    for (i = 0; i < bullets_away; i++)
		free( bulletlist[i] );
	    bullets_away = 0;
	    oldSimtime = Simtime;
	    lastFiredtime = 0.;
	}
    
    dt = Simtime - oldSimtime;
    oldSimtime = Simtime;
    
    if (bullets_away || trigger)
	{
	  if (trigger && (bullets_away < MAXBULLETS) 
	    && (Simtime > lastFiredtime + FIREINTERVAL))
	    {
		lastFiredtime = Simtime;
		bulletlist[bullets_away] = (pbullet) malloc(sizeof(bullet));
		if (bulletlist[bullets_away] == 0L) return;
		bulletlist[bullets_away]->age = 0.;
		bulletlist[bullets_away]->xdot = 
		    V_north_rel_ground + MUZZLEVEL*T_local_to_body_11;
		bulletlist[bullets_away]->ydot = 
		    V_east_rel_ground  + MUZZLEVEL*T_local_to_body_12;
		bulletlist[bullets_away]->zdot = 
		    -V_down_rel_ground  - MUZZLEVEL*T_local_to_body_13;
		bulletlist[bullets_away]->x = X_cg_rwy;
		bulletlist[bullets_away]->y = Y_cg_rwy;
		bulletlist[bullets_away]->z = H_cg_rwy;
		
		bullets_away++;
		
	    }
	  trigger = 0; /* to clear the trigger even if we were out of bullets */
	  for (i = 0; i < bullets_away; i++)
	    {
		bulletlist[i]->age = bulletlist[i]->age + dt;
		if (bulletlist[i]->age > LIFETIME)
		    {
			free (bulletlist[i]);
			for (j = i; j < (bullets_away-1); j++)
			    bulletlist[j] = bulletlist[j+1];
			bullets_away--;
			bulletlist[bullets_away] = 0L;
		    }
		else
		    {
			bulletlist[i]->zdot = bulletlist[i]->zdot - Gravity*dt;
			bulletlist[i]->x = bulletlist[i]->x + bulletlist[i]->xdot*dt;
			bulletlist[i]->y = bulletlist[i]->y + bulletlist[i]->ydot*dt;
			bulletlist[i]->z = bulletlist[i]->z + bulletlist[i]->zdot*dt;			
		    }
	    }
	  for (i = 0; i < bullets_away; i++)
	    {
	      pushmatrix();
	      translate(bulletlist[i]->x, -bulletlist[i]->y, bulletlist[i]->z);
	      rotate( (int) (-(Psi*57.3+90.)*10.), 'Z');
	      rotate( (int) (Theta*573.),  'X');
	      #ifdef RGBMODE
	        c3s(bulletcolor);
	      #else
	        color(BULLETCOLOR);
	      #endif
	      bgnpolygon();
		for(j = 0; j < 4; j++) v3f(bvect[j]);
	      endpolygon();
	      if(bulletlist[i]->z < 0)
		{
		  #ifdef RGBMODE
		    c3s(explosioncolor);
		  #else
		    color(EXPLOSIONCOLOR);
		  #endif
		  bgnpolygon();
		    for(j = 0; j < 4; j++) v3f(expl[j]);
		  endpolygon();
		  bulletlist[i]->age = Simtime + LIFETIME;	
		}
	      popmatrix();

	      /* check for hits on each target */

	      for(j = 0; j < targets_alive; j++)
		{
		  x_miss = bulletlist[i]->x - targetlist[j]->x;
		  y_miss = bulletlist[i]->y - targetlist[j]->y;
		  z_miss = bulletlist[i]->z - targetlist[j]->z;
		  miss2 = x_miss*x_miss + y_miss*y_miss + z_miss*z_miss;
		  if (miss2 < targetlist[j]->killradius2)
		    targetlist[j]->hit = -1;
		}
	    }
	}
    return;	
}



void drawworld(phi, theta, psi, xrwy, yrwy, alt, hudon)

float phi, theta, psi, xrwy, yrwy, alt;
int hudon;
{
  Matrix merect;

  OBJ("drawworld");  /* marker for GLProf */

  if (hudon) pushmatrix();     /* save current eyepoint for HUD */

  rot(-phi,   'x');
  rot( theta, 'y');
  rot( psi,   'z');
  
  getmatrix(merect);	/* save rotated matrix for HUD */
  
  translate(-xrwy, yrwy, -alt);

  drawsky(); 
  drawground(); 
  drawgrid(); 
  drawrwy();
  drawtargets();
  drawweapons(); 

  if (hudon) 
    {
      popmatrix();
      drawhud(merect);
    };


}

void ls_help()
{

#define HELPBKGND 8
#define HELPCOLOR WHITE
  static short helpbkgnd[3]  = { 127, 127, 127 };
  static short helpcolor[3]  = { 255, 255, 255 };

  int row;
  Device dev;
  short val;

  ls_unsync();  /* disable timer interrupts */

#ifdef RGBMODE
  c3s(helpbkgnd);
#else
  color(HELPBKGND);
#endif

  pushmatrix(); /* save the current drawing frame matrix */
  clear();     /* write the background color */
  
  loadmatrix( mhome );
 
#ifdef RGBMODE
    c3s(helpcolor);
#else
    color(HELPCOLOR);
#endif
  
/* Note: in this frame, +X is left, +Y is up */
#define HELPDIST HUDDIST
#define HELPXLEFT +20
#define HELPXRIGHT 0
#define HELPROWINC -4
#define HELPSTARTROW +40

  cmovi( HELPDIST, HELPXLEFT, HELPSTARTROW );
  charstr("LaRCSIM HELP MENU");
  cmovi( HELPDIST, HELPXLEFT, HELPSTARTROW + HELPROWINC/2);
  charstr("EBJ/AGCB/GCD/FSD/LaRC/NASA");
  cmovi( HELPDIST, HELPXLEFT, HELPSTARTROW + HELPROWINC );
  charstr("$Date: 1995/03/29 16:11:10 $");

  row = HELPSTARTROW + 3*HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("ESC");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Quit");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("'?'");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Help");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("'a'");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Retard throttle");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("'s'");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Advance throttle");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("Mouse buttons");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Left, center, right rudder");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("'r'");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Reset sim");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("'p'");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Pause sim (second 'p' to restart)");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("Arrow keys");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Look around");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("Home");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Forward view");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("End");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Rear view");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("Insert");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Left view");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("Page Up");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Right view");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("Delete");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Left downward view");
  row = row + HELPROWINC;

  cmovi( HELPDIST, HELPXLEFT, row  );charstr("Page Down");
  cmovi( HELPDIST, HELPXRIGHT, row );charstr("Right downward view");
  row = row + 2*HELPROWINC;


  cmovi( HELPDIST, HELPXLEFT, row );charstr("Press any key to return to flight...");
  

  if(!sim_control_.debug) swapbuffers(); 

  qdevice(KEYBD);
  qreset();
  dev = qread(&val);    /* wait for user input */
  unqdevice(KEYBD);
  qreset();

  popmatrix();     /* reload original drawing matrix */

  ls_resync();

}



int ls_cockpit( )

{
  int abort,  db;
  
  static int old_left_but, old_right_but, old_first_trig, old_second_trig;

  static short val, mval[2], mbias[2];
  static long org[2], size[2], win;
  long xWindSize,  yWindSize;
  static Device dev, mdev[2];
  static double fscale[2];
  
  static int inited = FALSE;
  static int hudon = TRUE;
  
  static unsigned short cros[16] = {
    0x0100, 0x0100, 0x0100, 0x0100,
    0x0100, 0x0100, 0x0100, 0xfffe,
    0x0100, 0x0100, 0x0100, 0x0100,
    0x0100, 0x0100, 0x0100, 0x0000
    };
  
  float ar;

  long savescrn, gd_timerhz;
  static short blk[3] = {0, 0, 0};

  glprof_object("ls_cockpit");  /* marker for GLProf */

  if (!inited)
    {
      gd_timerhz = getgdesc(GD_TIMERHZ);
      /* create window */
      xWindSize = getgdesc(GD_XPMAX) - 2*WINDOWMARGIN;
      yWindSize = getgdesc(GD_YPMAX) - 2*WINDOWMARGIN;
      prefposition(WINDOWMARGIN, xWindSize + WINDOWMARGIN, 
		   WINDOWMARGIN, yWindSize + WINDOWMARGIN);
      foreground();
      winopen(progname);
      mmode(MVIEWING);
      savescrn = scrnselect(getwscrn());
      scrnselect(savescrn);
      blanktime( 1800*gd_timerhz ); /* delay timeout for half hour */
      if(!sim_control_.debug)
        {
	  doublebuffer();  /* so it can be overridden in debug mode */
	}
      shademodel(FLAT);
#ifdef RGBMODE
      RGBmode();
#endif
      subpixel(TRUE); 

      gconfig();
   
      if(!sim_control_.debug)
        {
	  swapinterval( ( short ) 3 );    /* should force sim to run at 20 Hz */
	}

      blendfunction(BF_SA, BF_MSA); 
      linesmooth(SML_SMOOTHER); 

      ar = (float)xWindSize/(float)yWindSize;
      
      perspective(YWINDOWANGLE*10, ar, 10., 1000000.);
      polarview(0., -900, 900, 0);
#ifdef RGBMODE
      c3s(blk);
#else
      color(BLACK);
#endif
      if(!sim_control_.debug) swapbuffers(); 

      getmatrix( mhome );
      
      pushmatrix();
      drawworld(Phi*RAD_TO_DEG, Theta*RAD_TO_DEG, Psi*RAD_TO_DEG, 
		X_pilot_rwy, Y_pilot_rwy, H_pilot_rwy, hudon );
      if(!sim_control_.debug) swapbuffers(); 
      popmatrix();
      
      
      /* set up to read keys */
      qdevice(SPACEKEY);
      qdevice(ESCKEY);
      qdevice(AKEY);
      qdevice(PKEY);
      qdevice(RKEY);
      qdevice(SKEY);
      qdevice(TKEY);
      qdevice(LEFTMOUSE);
      qdevice(RIGHTMOUSE);
      qdevice(MIDDLEMOUSE);
      qdevice(LEFTARROWKEY);
      qdevice(RIGHTARROWKEY);
      qdevice(UPARROWKEY);
      qdevice(DOWNARROWKEY);
      qdevice(HOMEKEY);
      qdevice(ENDKEY);
      qdevice(INSERTKEY);
      qdevice(PAGEUPKEY);
      qdevice(DELKEY);
      qdevice(PAGEDOWNKEY);
      qdevice(BUT52);  /* actually slash/question mark key */

      if (sim_control_.sim_type == GLmouse)
	{
	  /* define the cursor (for mouse flying only) */
	  curstype(C16X1);
	  defcursor(1, cros);
	  curorigin(1, 7, 7);
	  setcursor(1, 0, 0); 
      
	  /* sleep(3);    /* to give us time to center cursor */

	  /* set up to read mouse */
	  getorigin(&org[X], &org[Y]);
	  getsize(&size[X], &size[Y]);
	  mdev[X] = MOUSEX;
	  mdev[Y] = MOUSEY;
      
	  mbias[X] = org[X]+size[X]/2;
	  mbias[Y] = org[Y]+size[Y]/2;

	  fscale[X] = LAT_SCALE/(double)(size[X]/2);
	  fscale[Y] = LON_SCALE/(double)(size[Y]/2);
	} 
      else
	{
	  cursoff();
	}

      inited = TRUE;
    }
  
  do
    {
      abort = FALSE;
      if(qtest())
	{
	  dev = qread(&val);
	  if (val==0)
	    {
	      switch (dev)
		{
		case SPACEKEY:	/* trigger */
		  trigger = -1;
		  break;
		case ESCKEY:    /* abort */
		  abort = TRUE;
		  if (sim_control_.paused)
		    {
		      ls_resync(); /* turn timer back on */
		      sim_control_.paused = FALSE;
		    }
		  break;
		case BUT52:  /* display help menu */
		  ls_help();
		  break;
		case AKEY:      /* retard throttle */
		  Throttle_pct = Throttle_pct - 0.01;
		  if(Throttle_pct <-0.2) Throttle_pct = -0.2;
		  break;
		case SKEY:      /* advance throttle */
		  Throttle_pct = Throttle_pct + 0.01;
		  if(Throttle_pct >1.) Throttle_pct = 1;
		  break;
		case LEFTMOUSE: /* left rudder */
		  Rudder_pedal = Rudder_pedal + DELTARUDDER;
		  break;
		case RIGHTMOUSE: /* right rudder */
		  Rudder_pedal = Rudder_pedal - DELTARUDDER;
		  break;
		case MIDDLEMOUSE: /* center rudder */
		  Rudder_pedal = 0;
		  break;
		case RKEY:      /* reset sim */
		  ls_init();
		  pushmatrix();
		  drawworld(Phi*57.3, Theta*57.3, Psi*57.3, 
			    X_pilot_rwy, Y_pilot_rwy, H_pilot_rwy, hudon );
		  if(!sim_control_.debug) swapbuffers(); 
		  popmatrix();
		  break;
		case TKEY:	/* request trim */
		  Q_body = 0.0;	/* force to zero pitch rate */ 
		  if (ls_trim()) ls_save_current_as_ic();
		  break;
		case PKEY:      /* temporarily pause */
		  if (sim_control_.paused)
		    {
		      ls_resync(); /* turn timer back on */
		      sim_control_.paused = FALSE;
		    }
		  else
		    {
		      ls_unsync(); /* turn timer off to disable interrupts */
		      sim_control_.overrun = TRUE; /* will turn hud to red */
		      sim_control_.paused = TRUE;
		    }
		  break;
		case LEFTARROWKEY:
		  rotate(-DELTAARROWAMT, 'z');
		  break;
		case RIGHTARROWKEY:
		  rotate( DELTAARROWAMT, 'z');
		  break;
		case UPARROWKEY:
		  rotate( DELTAARROWAMT, 'y');
		  break;
		case DOWNARROWKEY:
		  rotate( -DELTAARROWAMT, 'y');
		  break;
		case HOMEKEY:
		  loadmatrix( mhome );
		  break;
		case ENDKEY:
		  loadmatrix( mhome );
		  rotate( 1800, 'z');
		  break;
		case INSERTKEY:
		  loadmatrix( mhome );
		  rotate( -900, 'z');
		  break;
		case PAGEUPKEY:
		  loadmatrix( mhome );
		  rotate(  900, 'z');
		  break;
		case DELKEY:
		  loadmatrix( mhome );
		  rotate( -450, 'y');
		  rotate( -900, 'z');
		  break;
		case PAGEDOWNKEY:
		  loadmatrix( mhome );
		  rotate( -450, 'y');
		  rotate(  900, 'z');
		  break;
		}
	    }
	  qreset();
	}

	if (sim_control_.sim_type == cockpit)
	    {
	      ls_ACES();    /* read stick, throttle, and buttons/switches */
	
	      if( Left_button    > old_left_but    ) qenter(RKEY, 0);
	      if( Right_button   > old_right_but   ) qenter(PKEY, 0);
	      if( First_trigger  > old_first_trig  ) {};
	      if( Second_trigger > old_second_trig ) {};
	      old_left_but       = Left_button;
	      old_right_but      = Right_button;
	      old_first_trig     = First_trigger;
	      old_second_trig    = Second_trigger;
	      
	      trigger = First_trigger;
	    }
	else  
	    {   
	      getdev(2, mdev, mval );
	      Long_control = fscale[Y]*(double)(mval[Y] - mbias[Y]);
	      Lat_control  = fscale[X]*(double)(mval[X] - mbias[X]);
	    }
	      
	pushmatrix();
	drawworld(Phi*57.3, Theta*57.3, Psi*57.3, 
		X_pilot_rwy, Y_pilot_rwy, H_pilot_rwy, hudon );
	gflush();
	popmatrix();
	if(!sim_control_.debug) swapbuffers(); 
	
    }
  while (sim_control_.paused);
  return abort;
  
}

void ls_cockpit_exit()
{
    gexit();
}

