/***************************************************************************

	TITLE:		ls_funcgen.c
	
----------------------------------------------------------------------------

	FUNCTION:	Function generation routines for LaRCsim models

----------------------------------------------------------------------------

	MODULE STATUS:	developmental

----------------------------------------------------------------------------

	GENEALOGY:	

	Function table interpolation routines written 911220 E. B. Jackson		
	to support MATLAB/SIMULAB non-linear models.

	THEORY:

	Breakpoint data sets and function tables are stored separately in
	BREAKPOINTS and DATA structures. They are associated together in
	an individual FUNC_DATA structure; the FUNC_DATA structure is an
	abstraction of a multi-dimensional curve or surface.

	The NONLINEAR_FUNCTION structure associates this function data with
	the interpolation information (index and weights as well as the last
	value returned on the previous lookup call). This structure is an
	abstraction of the process of interpolating a FUNC_DATA curve; it
	includes a pointer to the function data as well as state
	information about where the function was most recently found,
	which speeds up subsequent searches, since a crawl through the
	breakpoint vector is used instead of a binary search.

	The tables are effectively unlimited in size and number of dimensions;
	the maximum length in any dimension is set by MAX_LENGTH, and the
	number of dimensions is set by MAX_DIMENSION; both are declared in
	ls_funcgen.h header file.

	Another data structure, ARG_LIST, is used to pass
	interpolation information to the lookup function. It contains
	the current index value and interpolation ratio for each
	dimension of the nonlinear function.

	USE:

	===== Initialization process =====

	* declare parameters 

	%define N_ALPHA1	10
	%define N_XMACH1 	 4
	%define N_DWF		 5

	* declare breakpoints

	static BREAKPOINTS ALPHA1_PTS   = { "ALPHA1", N_ALPHA1,
		{ 0.0, 2.0, 4.0, 6.0, 8.0, 10.0, 12.0, 14.0, 16.0, 18.0 }};
	static BREAKPOINTS XMACH1_PTS 	= { "XMACH1", N_XMACH1,
		{ 0.3, 0.6, 0.8, 0.9 }};
	static BREAKPOINTS DWF_PTS      = { "DWF",    N_DWF, 
		{  -30., -15., 0., 15., 30.}};

	* declare data set (first variable changes most rapidly)

	static	SCALAR	CLWF_DATA[ N_ALPHA1*N_XMACH1*N_DWF ] = {
	*	CLWF	POINTS	*
	* WF =  -30.0	*
	* XMACH = 0.3   *
	-0.52853E-01 ,-0.55962E-01 ,-0.65289E-01 ,-0.68398E-01 ,-0.54407E-01 ,
	-0.47770E-01 ,-0.43343E-01 ,-0.25638E-01 ,-0.13147E-01 ,-0.86206E-02 ,
	* XMACH = 0.6   *
	-0.34528E-01 ,-0.34432E-01 ,-0.35534E-01 ,-0.38899E-01 ,-0.40023E-01 ,
	-0.42346E-01 ,-0.45678E-01 ,-0.57322E-01 ,-0.60051E-01 ,-0.47768E-01 ,
	* XMACH = 0.8   *
	-0.52853E-01 ,-0.55962E-01 ,-0.65289E-01 ,-0.68398E-01 ,-0.54407E-01 ,
	-0.47770E-01 ,-0.43343E-01 ,-0.25638E-01 ,-0.13147E-01 ,-0.86206E-02 ,
	* XMACH = 0.9   *
	-0.34528E-01 ,-0.34432E-01 ,-0.35534E-01 ,-0.38899E-01 ,-0.40023E-01 ,
	-0.42346E-01 ,-0.45678E-01 ,-0.57322E-01 ,-0.60051E-01 ,-0.47768E-01 ,

	* WF =  -15.0	*
	* XMACH = 0.3   *
	-0.46403E-01 ,-0.49133E-01 ,-0.57322E-01 ,-0.60051E-01 ,-0.47768E-01 ,
	-0.39377E-01 ,-0.33779E-01 ,-0.11394E-01 ,-0.46116E-02 ,-0.33766E-02 ,
		.
		.
		.
	* WF =   30.0	*
		.
		.
		.
	* XMACH = 0.9   *
	0.59318E-01 , 0.87301E-01 , 0.77601E-01 , 0.48501E-01 , 0.77601E-01 ,
	0.62989E-01 , 0.53241E-01 , 0.14262E-01 , 0.18469E-01 , 0.11304E-01 };


	* associate the break points with function data

	static FUNC_DATA CLWF_PTS = {
		"CLWF POINTS", 				* name of point set
		3, 					* number of breakpoints
		{N_ALPHA_1, N_XMACH1, N_DWF},   	* size of each breakpoint set
	};

	
	* a typical function will then look like this  (creation of these
	* should be automated in the near future )


	float	clwfl_fn( float alpha, float mach, float dwfl )

	{
	  static NONLINEAR_FUNCTION	CLWFL_NLF = 
	    { "WFL Lift", NULL, { NULL, NULL, NULL }, 
	    { 0.0, 0.0, 0.0 }, { 4.0, 0.3, -30. } };

	  ARG_LIST   arg_list;

	  static int init=0;
	
	  if (!init) 
	    {
		init = -1;
		CLWFL_NLF.ptr_to_data = &CLWF_PTS; 
		CLWFL_NLF.bkPtList[0] = &ALPHA1_PTS;
		CLWFL_NLF.bkPtList[1] = &XMACH1_PTS;
		CLWFL_NLF.bkPtList[2] = &DWF_PTS;
	    }
	
	* Normalize breakpoints	*

		arg_list.index_and_weight[0] = normalize_bkpt( &CLWFL_NLF, 0, alpha );
		arg_list.index_and_weight[1] = normalize_bkpt( &CLWFL_NLF, 1, mach );
		arg_list.index_and_weight[2] = normalize_bkpt( &CLWFL_NLF, 2, dwfl );
	
	* Perform lookup and return	*

	return funcgen( &CLWFL0_NLF, &arg_list, 2 );

      } * End of CLwfl *


	===== Operation =====

	clwfl = clwfl_fn( Alpha, Mach, Dwfl );

----------------------------------------------------------------------------

	DESIGNED BY:	Bruce Jackson
	
	CODED BY:	Bruce Jackson
	
	MAINTAINED BY:	

----------------------------------------------------------------------------

	MODIFICATION HISTORY:
	
	DATE	PURPOSE						BY
	
	940216	Moved rcsid variable inside the function to get rid of
		archive and linker warnings.			EBJ
	
	CURRENT RCS HEADER:

$Header: /aces/larcsim/dev/RCS/ls_funcgen.c,v 1.6 1994/05/20 21:49:03 bjax Stab $
$Log: ls_funcgen.c,v $
 * Revision 1.6  1994/05/20  21:49:03  bjax
 * Added end-of-line character to emsg1 in routine getdata.
 *
 * Revision 1.5  1994/02/16  17:34:33  bjax
 * Moved rcsid to inside function to get rid of linker warnings.
 *
 * Revision 1.4  1994/01/11  18:25:52  bjax
 * Added large amounts of comments to header record to show how to
 * use funcgen stuff.
 *

----------------------------------------------------------------------------

	REFERENCES:

----------------------------------------------------------------------------

	CALLED BY:

----------------------------------------------------------------------------

	CALLS TO:

----------------------------------------------------------------------------

	INPUTS:

----------------------------------------------------------------------------

	OUTPUTS:

--------------------------------------------------------------------------*/

#include "ls_funcgen.h"
#include "ls_err.h"

extern ERROR error;

float normalize_bkpt( NONLINEAR_FUNCTION *nlfunct, int dim, DATA value )
     
{
#define NLFbpl	nlfunct->bkPtList[ dim ]

    char rcsid[] = "$Id: ls_funcgen.c,v 1.6 1994/05/20 21:49:03 bjax Stab $";

  int	    index, prev_index;
  DATA	    weight;
  char	    *stptr;
  static char *emsg1 = "Normalization value of &f1 less than \n\
lowest breakpoint value &f2 in set &s2.\n";
  static char *emsg2 = "Normalization value of &f1 greater than \n\
largest breakpoint value &f2 in set &s2.\n";
  static char *emsg3 = "Ran off lower end of breakpoint vector &s2 \n\
with normalization value of &f1.\n";
  static char *emsg4 = "Ran off upper end of breakpoint vector &s2 \n\
with normalization value of &f1.\n";
  
  if (value == nlfunct->latest_bkpt_value[ dim ] ) 
    return nlfunct->latest_index_and_weights[ dim ];
  if (value < NLFbpl->bkPts[ 0 ] )
    {
      error.severity = warning;
      error.code = E_DATA_INVALID;
      error.strg1 = emsg1;
      error.strg2 = &NLFbpl->name[0];
      error.fp1 = value;
      error.fp2 = NLFbpl->bkPts[ 0 ];
      return 0;
    }
  if (value > NLFbpl->bkPts[ (NLFbpl->length-1) ])
    {
      error.severity = warning;
      error.code = E_DATA_INVALID;
      error.strg1 = emsg2;
      error.strg2 = &NLFbpl->name[0];
      error.fp1 = value;
      error.fp2 = NLFbpl->bkPts[ (NLFbpl->length-1) ];
      return NLFbpl->bkPts[ (NLFbpl->length-1) ];
    }
  
  /* start looking from last position */
  
  index = nlfunct->latest_index_and_weights[ dim ];
  if (value < nlfunct->latest_bkpt_value[ dim ] )
    {
      /* search downward */
      prev_index = index + 1;
      while ( NLFbpl->bkPts[ index ] > value )
	{
	  prev_index = index;
	  index--;
	  if ( index < 0 ) 
	    {
	      error.severity = fatal;
	      error.code = E_FUNCGEN_INDEX_ERROR;
	      error.strg1 = emsg3;
	      error.strg2 = &NLFbpl->name[0];
	      error.fp1 = value;
	      return 0;
	    }			
	}
      /* found value below - figure weight */
      
      weight = ( value - NLFbpl->bkPts[ index ] ) /
	(NLFbpl->bkPts[ prev_index ] - 
	 NLFbpl->bkPts[ index ]);
    }
  else
    {
      /* search upward */
      prev_index = index - 1;
      while ( NLFbpl->bkPts[ index ] < value )
	{
	  prev_index = index;
	  index++;
	  if ( index >= NLFbpl->length )
	    {
	      error.severity = fatal;
	      error.code = E_FUNCGEN_INDEX_ERROR;
	      error.strg1 = emsg4;
	      error.strg2 = &NLFbpl->name[0];
	      error.fp1 = value;
	      return 0;
	    }			
	}
      /* found value below - figure weight */
      
      weight = ( value - NLFbpl->bkPts[ prev_index ] ) /
	(NLFbpl->bkPts[ index ] - NLFbpl->bkPts[ prev_index ]);
      index = prev_index;
    }
  nlfunct->latest_bkpt_value[ dim ] = value;
  nlfunct->latest_index_and_weights[ dim ] = (DATA)index + weight;
  return nlfunct->latest_index_and_weights[ dim ];
}


DATA getpt( NONLINEAR_FUNCTION *func_ptr, ARG_LIST *arg_list )
     
{
  int	i, offset, mult;
  DATA	*data_ptr;
  static char *emsg1 = "Function index &i1 value of &i2 exceeds index length \
&i3 in function &s1.\n";
  
  offset = 0;
  mult = 1;
  for ( i = 0; i < func_ptr->ptr_to_data->dim ; i++ )
    {
      if (arg_list->index[i] > func_ptr->ptr_to_data->length[i]) 
	{
	  error.severity = fatal;
	  error.code = E_FUNCGEN_INDEX_ERROR;
	  error.strg1 = emsg1;
	  strcpy( error.strg2, func_ptr->ptr_to_data->name );
	  error.ip1 = i+1;
	  error.ip2 = arg_list->index[i]+1;
	  error.ip3 = func_ptr->ptr_to_data->length[i]+1;
	  return 0;
	}
      offset = offset + (arg_list->index[i])*mult;
      mult = mult * func_ptr->ptr_to_data->length[i];
    }
  data_ptr = (DATA *) func_ptr->ptr_to_data->pts;
  return *(data_ptr+offset);
}

DATA funcgen( NONLINEAR_FUNCTION *func_ptr, ARG_LIST *arg_list, int dim	)
     
{
  DATA				a, b;
  float				weight;
  int					index;
  
  if (dim < 0) /* err; */ return 0;
  if (dim > func_ptr->ptr_to_data->dim) /* err; */ return 0;
  arg_list->index[dim] = arg_list->index_and_weight[dim];
  weight = arg_list->index_and_weight[dim] - arg_list->index[dim];
  
  if (dim == 0)		/* all but first index have been interpreted */
    {
      a = getpt( func_ptr, arg_list );
      arg_list->index[dim]++;
      b = getpt( func_ptr, arg_list );
      arg_list->index[dim]--;
    }
  else	/* more than one dimension to interpret - recurse */
    {
      a = funcgen( func_ptr, arg_list, dim-1);
      arg_list->index[dim]++;
      b = funcgen( func_ptr, arg_list, dim-1);
      arg_list->index[dim]--;
    }
  return (a + weight*(b-a));
}
