/***************************************************************************

	TITLE:		ls_ACES.c
	
----------------------------------------------------------------------------

	FUNCTION:	ACES cockpit interface routines for LaRCsim

----------------------------------------------------------------------------

	MODULE STATUS:	developmental

----------------------------------------------------------------------------

	GENEALOGY:	Created 20 DEC 93 by E. B. Jackson for ver 1.3

----------------------------------------------------------------------------

	DESIGNED BY:	Bruce Jackson
	
	CODED BY:	Bruce Jackson
	
	MAINTAINED BY:	Bruce Jackson

----------------------------------------------------------------------------

	MODIFICATION HISTORY:
	
	DATE	PURPOSE						BY
	
	940215	Added calibration values for stick positions; added logic to
		use calibration values; added call to ls_ACES_cal()
		routine; moved rcsid inside function body.		EBJ

	950228	Changed gear_sel_down to gear_sel_up.			EBJ
	
	CURRENT RCS HEADER:

$Header: /aces/larcsim/dev/RCS/ls_ACES.c,v 1.8 1995/02/28 20:36:15 bjax Stab $
$Log: ls_ACES.c,v $
 * Revision 1.8  1995/02/28  20:36:15  bjax
 * Changed name of gear_sel_down to gear_sel_up to reflect correct
 * sense of switch. EBJ
 *
 * Revision 1.7  1994/09/01  14:55:18  bailey
 * Extended time for speedbrake deployment
 *
 *
 * Revision 1.7  1994/08/25  14:14:21  mlb
 * Throttle set to zero and speedbrake implemented for
 * reverse thrust.
 *
 * Revision 1.6  1994/05/06  21:08:10  bjax
 * Fixed glitch in channel 0, due to improper setting of CSR reg.
 *
 * Revision 1.5  1994/05/06  20:22:52  bjax
 * Changed raw[] buffer from signed short to unsigned short.
 * Added CAL_FILE macro for calibration file path and name.
 *
 * Revision 1.4  1994/04/11  20:42:07  bjax
 * Added support for rudder pedals by THRUSTMASTER!
 *
 * Revision 1.3  1994/02/16  17:36:28  bjax
 * Moved rcsid to inside function body to get rid of linker warnings.
 *
 * Revision 1.2  1994/02/16  13:08:36  bjax
 * Added calibration for stick; moved some definitions to ls_ACES.h file.
 *
 * Revision 1.1  1993/12/21  14:34:49  bjax
 * Initial revision
 *

----------------------------------------------------------------------------

	REFERENCES:

----------------------------------------------------------------------------

	CALLED BY:

----------------------------------------------------------------------------

	CALLS TO:

----------------------------------------------------------------------------

	INPUTS:

----------------------------------------------------------------------------

	OUTPUTS:

--------------------------------------------------------------------------*/

#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <sys/mman.h>
#include <sys/resource.h>
#include <math.h>

#include "ls_types.h"
#include "ls_cockpit.h"
#include "ls_ACES.h"

#define DT 0.01666667
#define THRTAU 0.5

#define CAL_FILE "/aces/data/ACES.cal"
  
#define aroundup(x) ((char *)(((int) (x) + page_size - 1) & ~(page_size - 1)))
#define v_int		unsigned int *
#ifndef TRUE
#define TRUE -1
#endif
#ifndef FALSE
#define FALSE 0
#endif

struct vmic3114_control {
   unsigned short int ident_1;               /* 0x00 contains 0x000A */
   unsigned short int ident_2;               /* 0x02 contains 0x0000 */
   unsigned short int csr;                   /* 0x04 Control Status Reg */
   unsigned short int reserved_1;            /* 0x06 */
   unsigned short int ssr;                   /* 0x08 Scan Status Reg */
   unsigned short int io_ports;              /* 0x0A */
   unsigned short int icr;                   /* 0x0C Input Control Reg */
   unsigned short int ocr;                   /* 0x0E Output Control Reg */
   unsigned short int intr_control_input;    /* 0x10 interrupt control */
   unsigned short int intr_control_output;   /* 0x12 interrupt control */
   unsigned short int reserved_2;            /* 0x14 */
   unsigned short int reserved_3;            /* 0x16 */
   unsigned short int intr_vector_input;     /* 0x18 interrupt vector */
   unsigned short int intr_vector_output;    /* 0x1A interrupt vector */
   unsigned short int reserved_4;            /* 0x1C */
   unsigned short int reserved_5;            /* 0x1E */
};

struct vmic3114_data {
   short int input_buffer[65536];         /* 64K words of input buffer */
   short int output_buffer[65536];         /* 64K words of output buffer */
};

#define LONGAD	0
#define LATAD	1
#define PEDAD	6

#define	BOARD_ID	    0x000A	    /* Board ID code (word) */

#define FAIL_LED_OFF	    0x8000	    /* CSR Control bits */
#define ANALOG_OUTPUTS_ON   0x4000
#define PORT1_DIR_OUT	    0x2000
#define PORT0_DIR_OUT	    0x1000
#define ADVANCE_OUTPUT_ADDR 0x0800
#define OUTPUT_SINGLE_SCAN  0x0400
#define OUTPUT_SYNCHRONOUS  0x0200
#define SEL_OUTPUT_BUFF_1   0x0100
#define ADVANCE_INPUT_ADDR  0x0080
#define INPUT_SINGLE_SCAN   0x0040
#define INPUT_SYNCHRONOUS   0x0020
#define SEL_INPUT_BUFF_B    0x0010
#define SOFTWARE_RESET	    0x0008
#define MASTER_MODE	    0x0004
#define DIS_OUTPUT_SCAN	    0x0002
#define ENABLE_BUFFERS	    0x0001

#define INPUTS_ARE_DIFF	    0x0080	    /* SSR Status bits */
#define BINARY_CODING	    0x0040
#define DAC_SCAN_COMPLETE   0X0020
#define DAC_SCAN_BUFF_A	    0x0010
#define DAC_SYNC_ENABLED    0x0008
#define ADC_SCAN_COMPLETE   0x0004
#define ADC_SCAN_BUFF_A	    0x0002
#define ADC_SYNC_ENABLED    0x0001

#define TEST_MODE_3	    0x1800	    /* ICR & OCR bit codes */
#define TEST_MODE_2	    0x1000	    /* not all bits apply  */
#define	TEST_MODE_1	    0x0800	    /* to both ocr & icr   */
#define TEST_MODE_OFF	    0x0000
#define	GAIN_16		    0x0700
#define GAIN_8		    0x0300
#define GAIN_4		    0x0200
#define GAIN_2		    0x0100
#define GAIN_1		    0x0000
#define SCAN_3		    0x00F0	    /* OCR ONLY!!!      */
#define SCAN_7		    0x00E0	    /* OCR ONLY!!!      */
#define SCAN_15		    0x00D0	    /* OCR ONLY!!!      */
#define SCAN_30		    0x00C0	    /* OCR ONLY!!!      */
#define SCAN_61		    0x00B0	    /* OCR ONLY!!!      */
#define SCAN_122	    0x00A0	    /* OCR ONLY!!!      */
#define SCAN_244	    0x0090	    /* OCR ONLY!!!      */
#define DIS_INPUT_SCAN	    0x0080	    /* <- ICR only	*/
#define SCAN_488	    0x0080	    /* OCR ONLY!!!      */
#define SCAN_977	    0x0070	    /* These faster scans work for both out and in      */
#define SCAN_1953	    0x0060
#define SCAN_3906	    0x0050
#define SCAN_7813	    0x0040
#define SCAN_15625	    0x0030
#define SCAN_31250	    0x0020
#define SCAN_62500	    0x0010
#define SCAN_125000	    0x0000
#define BUFF_64K	    0x000F
#define BUFF_32K	    0x000E
#define BUFF_16K	    0x000D
#define BUFF_8K		    0x000C
#define BUFF_4K		    0x000B
#define BUFF_2K		    0x000A
#define BUFF_1K		    0x0009
#define BUFF_512	    0x0008
#define BUFF_256	    0x0007
#define BUFF_128	    0x0006
#define BUFF_64		    0x0005
#define BUFF_32		    0x0004
#define BUFF_16		    0x0003
#define BUFF_8		    0x0002
#define BUFF_4		    0x0001
#define BUFF_2		    0x0000

#define FLAG_BIT	    0x80	    /* BIM control bits */
#define FLAG_AUTO_CLEAR	    0x40
#define EXTERNAL_VECTOR	    0x00	    /* always 0 (see manual */
#define INTERRUPT_ENABLE    0x10
#define INTR_AUTO_CLEAR	    0x08
#define REQUEST_LEVEL_7	    0x07
#define REQUEST_LEVEL_6	    0x06
#define REQUEST_LEVEL_5	    0x05
#define REQUEST_LEVEL_4	    0x04
#define REQUEST_LEVEL_3	    0x03
#define REQUEST_LEVEL_2	    0x02
#define REQUEST_LEVEL_1	    0x01
#define INTERRUPTS_OFF	    0x00


/*=================  VMIC INTERFACE ROUTINES ==========================*/

void adc_reset(struct vmic3114_control *adc_control)
{
   adc_control->csr = 0x0008;
   sginap(50);
}

void adc_aquire(struct vmic3114_control *adc_control)
{
   adc_control->icr = 0x0012;
   adc_control->csr = 0x8005;
}

char *attmemdev(unsigned long int addr, unsigned long int size,
		char *devname)
{
   unsigned char *vme_adr;
   char *taddr;
   off_t dev_base;
   caddr_t tstart_adr;
   int fdvme32, page_size;
   v_int status;
   int rcode;
   struct rlimit *rlp;

   page_size = getpagesize();
   
   rlp = (struct rlimit *) malloc( sizeof( struct rlimit ));

   rcode = getrlimit( RLIMIT_DATA, rlp );
   if (rcode < 0) {
	perror("attmemdev: call to getrlimit(RLIMIT_DATA) failed ");
   }
   rcode = getrlimit( RLIMIT_STACK, rlp );
   if (rcode < 0) {
	perror("attmemdev: call to getrlimit(RLIMIT_STACK) failed ");
   }
   rcode = getrlimit( RLIMIT_VMEM, rlp );
   if (rcode < 0) {
	perror("attmemdev: call to getrlimit(RLIMIT_VMEM) failed ");
   }
   rcode = getrlimit( RLIMIT_RSS, rlp );
   if (rcode < 0) {
	perror("attmemdev: call to getrlimit(RLIMIT_RSS) failed ");
   }

   vme_adr = (unsigned char *)addr;
   dev_base = (off_t)vme_adr;
   
   tstart_adr = (caddr_t)sbrk(size);
   if ((int)tstart_adr == -1) {
      perror("attmem: sbrk allocation failed ");
      return ((char *)0);
   }
/*   taddr = (char *)aroundup((int)(tstart_adr)); */
    taddr = 0;  /* changed since mmap likes to put whereever if taddr = 0 */
    
   fdvme32 = open(devname, O_RDWR);
   if (fdvme32 < 0) {
      perror("attmem: vme i/o OPEN failed ");
      printf("device: %s\n", devname);
      return ((char *)0);
   }
   status = (v_int)mmap(taddr, size, PROT_READ | PROT_WRITE, MAP_SHARED,
                        fdvme32, dev_base);
   if ((int)status == -1) {
      perror("attmem: mmap failed ");
      return ((char *)0);
   }
   return ((char *)((unsigned long)status));
}

int adc_attach(struct vmic3114_control **adc_control,
               struct vmic3114_data **adc_data)
{
   int succ;

   succ = TRUE;

   *adc_control = (struct vmic3114_control *)attmemdev(VMIC3114_CONTROL_ADDR,
						       VMIC3114_CONTROL_LENGTH,
						       VMIC3114_CONTROL_BUS_DEVICE);
   if ((int)(*adc_control == 0)) {
      printf("adc_attach: Unable to attach to vmic3114 control store.\n");
      succ = FALSE;
   }

   if (succ) {
      *adc_data = (struct vmic3114_data *)attmemdev(VMIC3114_DATA_ADDR,
						    VMIC3114_DATA_LENGTH,
						    VMIC3114_DATA_BUS_DEVICE);
      if ((int)(*adc_data == 0)) {
         printf("adc_attach: Unable to attach to vmic3114 data block\n");
         succ = FALSE;
      }
   }

   return succ;
}

/*=================  ACES COCKPIT INTERFACE ROUTINES ==========================*/


ls_ACES( )
{
char rcsid[] = "$Id: ls_ACES.c,v 1.8 1995/02/28 20:36:15 bjax Stab $";

    /* calibration data */
    
    const static CALIBRATION_DATA cal;

  /* output limits */
  
    const static float tfwdstop   =  1.0;
    const static float taftstop   = -0.01;
    const static float stkfwdstop =  1.0;
    const static float stkaftstop = -1.0;
    const static float stklftstop = -1.0;
    const static float stkrgtstop =  1.0;
    const static float pedlftstop = -1.0;
    const static float pedrgtstop =  1.0;
    
    static  float tthrow;
  
    static  float trange_inv[4];
    
    static  float fwdscale, aftscale, lftscale, rgtscale, pedlscale, pedrscale;
    
    static  struct vmic3114_control *adc_control;
    static  struct vmic3114_data *adc_data;
    
    short   int soft_csr, soft_ssr, soft_icr;
    static  int inited = 0;
    
    static  float arg;
    static  float c1, c2;
    float   thrin[4];
    static  float throttle_past[4] = {0.2, 0.2, 0.2, 0.2};

    static float speedbrake;    
    static float   buf[8];
    float   long_AD, lat_AD, ped_AD, long_sig, lat_sig,  ped_sig;
    static unsigned short int raw[8];
    static long wait;
    short  unsigned int di, dio;
    int i, j;

    if (!inited)
      {
      
	inited = -1;
	
	ls_ACES_cal( &cal ); /* call the calibration routine */
	
	/* calculate throttle filter coefficients assuming dt */
	
	arg =  -DT/THRTAU ;
	c2 = exp(arg);
	c1 = 1-c2;
	
	/* calculate scaling values */
	
	tthrow = tfwdstop - taftstop;
	
	for( i = 0; i<4; i++ )
	    trange_inv[i] = 1.0/(cal.thrfwd[i] - cal.thraft[i]);
	
	fwdscale = stkfwdstop/(cal.stkfwd - cal.stklg0);
	aftscale = stkaftstop/(cal.stkaft - cal.stklg0);
	
	lftscale = stklftstop/(cal.stklft - cal.stklt0);
	rgtscale = stkrgtstop/(cal.stkrgt - cal.stklt0);
	
	pedlscale = pedlftstop/(cal.pedlft - cal.pedctr);
	pedrscale = pedrgtstop/(cal.pedrgt - cal.pedctr);
	
	if (!adc_attach(&adc_control,&adc_data)) {
	  printf("Unable to connect to adc card.\n");
	  exit(1);
	}
	
	/* reset card */
	
	adc_control->csr = SOFTWARE_RESET;
	do
	  {
	    soft_csr = adc_control->csr;  
	  } 
	while ( soft_csr & SOFTWARE_RESET );
	
	/* set up for synchronous read */
	
	adc_control->icr = (SCAN_125000 | BUFF_8); 
	adc_control->csr = (FAIL_LED_OFF | INPUT_SINGLE_SCAN
			    | MASTER_MODE | ENABLE_BUFFERS | DIS_OUTPUT_SCAN); 
	
	soft_icr = adc_control->icr;
	adc_control->icr = soft_icr | DIS_INPUT_SCAN;  /* Put DISABLE IN hi */
	sginap(1);
	adc_control->icr = soft_icr & ~DIS_INPUT_SCAN;  /* Put DISABLE IN lo */

	/* set up to read all 16 inputs */

	adc_control->csr &= ((~PORT1_DIR_OUT) & (~PORT0_DIR_OUT)); 

      } 

    /* Normal run loop cycles through here */

    wait = 0;
    do 
      { 
	soft_ssr = adc_control->ssr;
	wait++;
      } 
    while ((soft_ssr & ADC_SCAN_BUFF_A) ); /* wait for AD SCAN A */

    soft_icr = adc_control->icr;           /* Start of single-scan toggle */
    adc_control->icr = soft_icr | DIS_INPUT_SCAN;  /* Put DISABLE IN hi */

    for( i=0; i<8; i++)
    {
	raw[i] = adc_data->input_buffer[i];
	buf[i] = ((float)(raw[i])/204.8) - 10.0;
    }

    di = ~(adc_control->io_ports);

/* debugging code to display DI's
 * 
 *
	j=0;
	dio = di;
	for (i=0;i<16;i++)
	  {
	    if( dio & 0x8000 ) printf("1"); else printf("0");
	    dio = dio << 1;
	    j++;
	    if (j >= 4)
	      {
	        j = 0;
	        printf(" ");
	      }
	  }
	printf("\n"); 
 */

    long_AD = buf[LONGAD];
    lat_AD  = buf[LATAD]; 
    ped_AD  = buf[PEDAD];
    
    long_sig = long_AD - cal.stklg0;
    lat_sig  = lat_AD  - cal.stklt0;
    ped_sig  = ped_AD  - cal.pedctr;
    
    SB_extend = di & SPEEDBRAKE_EXTEND;
    SB_retract = di & SPEEDBRAKE_RETRACT;

    speedbrake=speedbrake-(.00002*SB_extend)+(.00002*SB_retract);
    if(speedbrake > 0.)   speedbrake = 0.;
    if(speedbrake < -0.2) speedbrake = -0.2;
    
    if (long_sig > 0.)
	Long_control = long_sig*fwdscale;
    else
	Long_control = long_sig*aftscale;
	
    if (lat_sig > 0.)
	Lat_control = lat_sig*rgtscale;
    else
	Lat_control = lat_sig*lftscale;
    
    if (ped_sig > 0.)
	Rudder_pedal = ped_sig*pedrscale;
    else
	Rudder_pedal = ped_sig*pedlscale;

    for( i=0; i<4; i++)
    {
	thrin[i] = (float) tthrow*((float) buf[i+2] 
	    - cal.thraft[i])*trange_inv[i] + taftstop +speedbrake;
    
        if (thrin[i] > 1.) thrin[i] = 1.;
        if (thrin[i] < -0.2) thrin[i] = -0.2; 
    
	Throttle[i] = thrin[i]*c1 + throttle_past[i]*c2;
	throttle_past[i] = Throttle[i];
    }

    Fwd_trim = di & FWD_TRIM;
    Aft_trim = di & AFT_TRIM;
    Right_trim  = di & RIGHT_TRIM;
    Left_trim  = di & LEFT_TRIM;
    
    Left_button = di & LEFT_PUSH_BUTTON;
    Right_button = di & RIGHT_PUSH_BUTTON;
    First_trigger = di & FIRST_TRIG_BUTTON;
    Second_trigger = di & SECOND_TRIG_BUTTON;
    
    Gear_sel_up = di & LANDING_GEAR_UP;
    
    adc_control->icr = soft_icr & ~DIS_INPUT_SCAN;  /* Put DISABLE IN lo to start scan */
}


ls_ACES_cal( CALIBRATION_DATA *cal )	/* performs cockpit calibration routine */

{
    FILE *fp;
    int num;
    
    fp = fopen(CAL_FILE, "r");
    if (fp != 0)
    /* if ACES.cal exists, use it */
	{
	    num = fscanf( fp, "%f %f %f %f\n", 
		&cal->thraft[0],  &cal->thraft[1], &cal->thraft[2], &cal->thraft[3] );
	    if (num != 4) fprintf(stderr, "ACES.cal improper format, line 1\n");
	    num = fscanf( fp, "%f %f %f %f\n", 
		&cal->thrfwd[0],  &cal->thrfwd[1], &cal->thrfwd[2], &cal->thrfwd[3] );
	    if (num != 4) fprintf(stderr, "ACES.cal improper format, line 2\n");
	    num = fscanf( fp, "%f %f %f\n", 
		&cal->stkaft, &cal->stklg0, &cal->stkfwd );
	    if (num != 3) fprintf(stderr, "ACES.cal improper format, line 3\n");
	    num = fscanf( fp, "%f %f %f\n", 
		&cal->stklft, &cal->stklt0, &cal->stkrgt );
	    if (num != 3) fprintf(stderr, "ACES.cal improper format, line 4\n");
	    num = fscanf( fp, "%f %f %f\n", 
		&cal->pedlft, &cal->pedctr, &cal->pedrgt );
	    if (num != 3) fprintf(stderr, "ACES.cal improper format, line 5\n");
	    fclose( fp );
	}
    else
    /* use default calibration numbers */   /* 931215  EBJ */
	{
	    fprintf(stderr,  CAL_FILE);
	    fprintf(stderr, " not found; using default values\n");
	    cal->thraft[0] =  -5.752;
	    cal->thraft[1] =  -6.709;
	    cal->thraft[2] =  -6.548;
	    cal->thraft[3] =  -7.314;
	    cal->thrfwd[0] =  -3.325;
	    cal->thrfwd[1] =  -4.307;
	    cal->thrfwd[2] =  -4.097;
	    cal->thrfwd[3] =  -4.858;
	    cal->stkaft    = -10.000;
	    cal->stklg0    =  -0.405;
	    cal->stkfwd    =   9.116;
	    cal->stklft    =  -9.996;
	    cal->stklt0    =  -0.366;
	    cal->stkrgt    =   9.297;
	    cal->pedlft    =  -5.900;
	    cal->pedctr    =  -0.100;
	    cal->pedrgt    =   5.600;
	}
 
}
