package zed.rainxch.githubstore.feature.auth.data.repository

import co.touchlab.kermit.Logger
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.withContext
import zed.rainxch.githubstore.core.data.data_source.TokenDataSource
import zed.rainxch.githubstore.core.domain.model.DeviceStart
import zed.rainxch.githubstore.core.domain.model.DeviceTokenSuccess
import zed.rainxch.githubstore.feature.auth.data.network.GitHubAuthApi
import zed.rainxch.githubstore.feature.auth.data.getGithubClientId
import zed.rainxch.githubstore.feature.auth.domain.repository.AuthRepository

class AuthRepositoryImpl(
    private val tokenDataSource: TokenDataSource,
) : AuthRepository {

    override val accessTokenFlow: Flow<String?>
        get() = tokenDataSource.tokenFlow.map { it?.accessToken }

    override val isAuthenticatedFlow: Flow<Boolean>
        get() = tokenDataSource.tokenFlow.map { it != null }

    override suspend fun isAuthenticated(): Boolean =
        tokenDataSource.current() != null

    override suspend fun startDeviceFlow(): DeviceStart =
        withContext(Dispatchers.Default) {
            val clientId = getGithubClientId()
            require(clientId.isNotBlank()) {
                "Missing GitHub CLIENT_ID. Add GITHUB_CLIENT_ID to local.properties."
            }

            try {
                val result = GitHubAuthApi.startDeviceFlow(clientId)
                Logger.d { "✅ Device flow started. User code: ${result.userCode}" }
                result
            } catch (e: Exception) {
                Logger.d { "❌ Failed to start device flow: ${e.message}" }
                throw Exception(
                    "Failed to start GitHub authentication. " +
                            "Please check your internet connection and try again.",
                    e
                )
            }
        }

    override suspend fun awaitDeviceToken(start: DeviceStart): DeviceTokenSuccess =
        withContext(Dispatchers.Default) {
            val clientId = getGithubClientId()
            val timeoutMs = start.expiresInSec * 1000L
            var remainingMs = timeoutMs
            var intervalMs = (start.intervalSec.coerceAtLeast(5)) * 1000L
            var consecutiveErrors = 0
            val maxConsecutiveErrors = 5

            Logger.d { "⏱️ Starting token polling. Expires in: ${start.expiresInSec}s, Interval: ${start.intervalSec}s" }

            while (remainingMs > 0) {
                try {
                    val res = GitHubAuthApi.pollDeviceToken(clientId, start.deviceCode)
                    val success = res.getOrNull()

                    if (success != null) {
                        Logger.d { "✅ Token received successfully!" }
                        withRetry(maxAttempts = 3) {
                            tokenDataSource.save(success)
                        }
                        return@withContext success
                    }

                    val error = res.exceptionOrNull()
                    val msg = (error?.message ?: "").lowercase()

                    Logger.d { "📡 Poll response: $msg (errors: $consecutiveErrors/$maxConsecutiveErrors)" }

                    when {
                        "authorization_pending" in msg -> {
                            consecutiveErrors = 0
                            delay(intervalMs)
                            remainingMs -= intervalMs
                        }

                        "slow_down" in msg -> {
                            consecutiveErrors = 0
                            intervalMs += 5000
                            Logger.d { "⚠️ Slowing down polling to ${intervalMs}ms" }
                            delay(intervalMs)
                            remainingMs -= intervalMs
                        }

                        "access_denied" in msg -> {
                            throw CancellationException("You denied access to the app")
                        }

                        "expired_token" in msg || "expired_device_code" in msg -> {
                            throw CancellationException(
                                "Authorization timed out. Please try again."
                            )
                        }

                        "unable to resolve" in msg ||
                                "no address" in msg ||
                                "failed to connect" in msg ||
                                "connection refused" in msg ||
                                "network is unreachable" in msg -> {
                            consecutiveErrors++
                            Logger.d { "⚠️ Network error, retrying... ($consecutiveErrors/$maxConsecutiveErrors)" }

                            val backoffDelay = intervalMs * (1 + consecutiveErrors)

                            if (consecutiveErrors >= maxConsecutiveErrors) {
                                throw Exception(
                                    "Network connection unstable during authentication. " +
                                            "Please check your connection and try again."
                                )
                            }
                            delay(backoffDelay)
                            remainingMs -= backoffDelay
                        }

                        else -> {
                            consecutiveErrors++
                            Logger.d { "⚠️ Error: $msg (attempt $consecutiveErrors/$maxConsecutiveErrors)" }

                            if (consecutiveErrors >= maxConsecutiveErrors) {
                                throw Exception("Authentication failed: $msg")
                            }

                            val backoffDelay = intervalMs * 2
                            delay(backoffDelay)
                            remainingMs -= backoffDelay
                        }
                    }

                } catch (e: CancellationException) {
                    throw e
                } catch (e: Exception) {
                    Logger.d { "❌ Poll error: ${e.message}" }
                    Logger.d { "❌ Error type: ${e::class.simpleName}" }
                    consecutiveErrors++

                    if (consecutiveErrors >= maxConsecutiveErrors) {
                        throw Exception(
                            "Authentication failed after multiple attempts. " +
                                    "Error: ${e.message}",
                            e
                        )
                    }

                    val backoffDelay = intervalMs * (1 + consecutiveErrors)
                    delay(backoffDelay)
                    remainingMs -= backoffDelay
                }
            }

            throw CancellationException(
                "Authentication timed out. Please try again and complete the process faster."
            )
        }

    private suspend fun <T> withRetry(
        maxAttempts: Int = 3,
        initialDelay: Long = 1000,
        block: suspend () -> T
    ): T {
        repeat(maxAttempts - 1) { attempt ->
            try {
                return block()
            } catch (e: Exception) {
                Logger.d { "⚠️ Retry attempt ${attempt + 1} failed: ${e.message}" }
                delay(initialDelay * (attempt + 1))
            }
        }
        return block()
    }
}