package zed.rainxch.githubstore.feature.home.presentation

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import co.touchlab.kermit.Logger
import githubstore.composeapp.generated.resources.Res
import githubstore.composeapp.generated.resources.home_failed_to_load_repositories
import githubstore.composeapp.generated.resources.no_repositories_found
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.Job
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.onStart
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import org.jetbrains.compose.resources.getString
import zed.rainxch.githubstore.core.domain.Platform
import zed.rainxch.githubstore.core.domain.model.PlatformType
import zed.rainxch.githubstore.core.domain.repository.FavouritesRepository
import zed.rainxch.githubstore.core.domain.repository.InstalledAppsRepository
import zed.rainxch.githubstore.core.domain.use_cases.SyncInstalledAppsUseCase
import zed.rainxch.githubstore.core.presentation.model.DiscoveryRepository
import zed.rainxch.githubstore.feature.home.domain.repository.HomeRepository
import zed.rainxch.githubstore.feature.home.presentation.model.HomeCategory

class HomeViewModel(
    private val homeRepository: HomeRepository,
    private val installedAppsRepository: InstalledAppsRepository,
    private val platform: Platform,
    private val syncInstalledAppsUseCase: SyncInstalledAppsUseCase,
    private val favouritesRepository: FavouritesRepository
) : ViewModel() {

    private var hasLoadedInitialData = false
    private var currentJob: Job? = null
    private var nextPageIndex = 1

    private val _state = MutableStateFlow(HomeState())
    val state = _state
        .onStart {
            if (!hasLoadedInitialData) {
                syncSystemState()

                loadPlatform()
                loadRepos(isInitial = true)
                observeInstalledApps()
                observeFavourites()

                hasLoadedInitialData = true
            }
        }
        .stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(5_000L),
            initialValue = HomeState()
        )

    private fun syncSystemState() {
        viewModelScope.launch {
            try {
                val result = syncInstalledAppsUseCase()
                if (result.isFailure) {
                    Logger.w { "Initial sync had issues: ${result.exceptionOrNull()?.message}" }
                }
            } catch (e: Exception) {
                Logger.e { "Initial sync failed: ${e.message}" }
            }
        }
    }

    private fun loadPlatform() {
        _state.update {
            it.copy(isAppsSectionVisible = platform.type == PlatformType.ANDROID)
        }
    }

    private fun observeInstalledApps() {
        viewModelScope.launch {
            installedAppsRepository.getAllInstalledApps().collect { installedApps ->
                val installedMap = installedApps.associateBy { it.repoId }
                _state.update { current ->
                    current.copy(
                        repos = current.repos.map { homeRepo ->
                            val app = installedMap[homeRepo.repository.id]
                            homeRepo.copy(
                                isInstalled = app != null,
                                isUpdateAvailable = app?.isUpdateAvailable ?: false
                            )
                        },
                        isUpdateAvailable = installedMap.any { it.value.isUpdateAvailable }
                    )
                }
            }
        }
    }

    private fun loadRepos(isInitial: Boolean = false, category: HomeCategory? = null) {
        if (_state.value.isLoading || _state.value.isLoadingMore) {
            Logger.d { "Already loading, skipping..." }
            return
        }

        currentJob?.cancel()

        if (isInitial) {
            nextPageIndex = 1
        }

        val targetCategory = category ?: _state.value.currentCategory

        Logger.d { "Loading repos: category=$targetCategory, page=$nextPageIndex, isInitial=$isInitial" }

        currentJob = viewModelScope.launch {

            _state.update {
                it.copy(
                    isLoading = isInitial,
                    isLoadingMore = !isInitial,
                    errorMessage = null,
                    currentCategory = targetCategory,
                    repos = if (isInitial) emptyList() else it.repos
                )
            }

            try {
                val flow = when (targetCategory) {
                    HomeCategory.TRENDING -> homeRepository.getTrendingRepositories(nextPageIndex)
                    HomeCategory.NEW -> homeRepository.getNew(nextPageIndex)
                    HomeCategory.RECENTLY_UPDATED -> homeRepository.getRecentlyUpdated(nextPageIndex)
                }

                flow.collect { paginatedRepos ->
                    Logger.d { "Received ${paginatedRepos.repos.size} repos, hasMore=${paginatedRepos.hasMore}, nextPage=${paginatedRepos.nextPageIndex}" }

                    this@HomeViewModel.nextPageIndex = paginatedRepos.nextPageIndex

                    val installedAppsMap = installedAppsRepository
                        .getAllInstalledApps()
                        .first()
                        .associateBy { it.repoId }

                    val favoritesMap = favouritesRepository
                        .getAllFavorites()
                        .first()
                        .associateBy { it.repoId }

                    val newReposWithStatus = paginatedRepos.repos.map { repo ->
                        val app = installedAppsMap[repo.id]
                        val favourite = favoritesMap[repo.id]

                        DiscoveryRepository(
                            isInstalled = app != null,
                            isFavourite = favourite != null,
                            isUpdateAvailable = app?.isUpdateAvailable ?: false,
                            repository = repo
                        )
                    }

                    _state.update { currentState ->
                        val rawList = currentState.repos + newReposWithStatus
                        val uniqueList = rawList.distinctBy { it.repository.fullName }

                        currentState.copy(
                            repos = uniqueList,
                            hasMorePages = paginatedRepos.hasMore,
                            errorMessage = if (uniqueList.isEmpty() && !paginatedRepos.hasMore) {
                                getString(Res.string.no_repositories_found)
                            } else null
                        )
                    }
                }

                Logger.d { "Flow completed" }
                _state.update {
                    it.copy(isLoading = false, isLoadingMore = false)
                }

            } catch (t: Throwable) {
                if (t is CancellationException) {
                    Logger.d { "Load cancelled (expected)" }
                    throw t
                }

                Logger.e { "Load failed: ${t.message}" }
                _state.update {
                    it.copy(
                        isLoading = false,
                        isLoadingMore = false,
                        errorMessage = t.message ?: getString(Res.string.home_failed_to_load_repositories)
                    )
                }
            }
        }
    }

    fun onAction(action: HomeAction) {
        when (action) {
            HomeAction.Refresh -> {
                viewModelScope.launch {
                    syncInstalledAppsUseCase()
                    nextPageIndex = 1
                    loadRepos(isInitial = true)
                }
            }

            HomeAction.Retry -> {
                nextPageIndex = 1
                loadRepos(isInitial = true)
            }

            HomeAction.LoadMore -> {
                Logger.d { "LoadMore action: isLoading=${_state.value.isLoading}, isLoadingMore=${_state.value.isLoadingMore}, hasMore=${_state.value.hasMorePages}" }

                if (!_state.value.isLoadingMore && !_state.value.isLoading && _state.value.hasMorePages) {
                    loadRepos(isInitial = false)
                }
            }

            is HomeAction.SwitchCategory -> {
                if (_state.value.currentCategory != action.category) {
                    nextPageIndex = 1
                    loadRepos(isInitial = true, category = action.category)
                }
            }

            is HomeAction.OnRepositoryClick -> {
                /* Handled in composable */
            }

            HomeAction.OnSearchClick -> {
                /* Handled in composable */
            }

            HomeAction.OnSettingsClick -> {
                /* Handled in composable */
            }

            HomeAction.OnAppsClick -> {
                /* Handled in composable */
            }
        }
    }

    private fun observeFavourites() {
        viewModelScope.launch {
            favouritesRepository.getAllFavorites().collect { favourites ->
                val favouritesMap = favourites.associateBy { it.repoId }
                _state.update { current ->
                    current.copy(
                        repos = current.repos.map { homeRepo ->
                            homeRepo.copy(
                                isFavourite = favouritesMap.containsKey(homeRepo.repository.id)
                            )
                        }
                    )
                }
            }
        }
    }

    override fun onCleared() {
        super.onCleared()
        currentJob?.cancel()
    }
}