package tech.lp2p.thor.model

import android.graphics.Bitmap
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.compose.ui.graphics.ImageBitmap
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.eygraber.uri.Uri
import io.github.remmerw.idun.extractMimeType
import io.github.remmerw.idun.extractName
import io.github.remmerw.idun.extractSize
import io.github.remmerw.loki.parseMagnetUri
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.launch
import tech.lp2p.thor.R
import tech.lp2p.thor.data.Bookmark
import tech.lp2p.thor.data.Task
import tech.lp2p.thor.debug

class StateModel() : ViewModel() {

    var showTasks: Boolean by mutableStateOf(false)

    val online: Flow<Boolean> = flow {
        while (true) {
            val latestOnline = platform().isNetworkConnected()
            emit(latestOnline) // Emits the result of the request to the flow
            delay(1000) // Suspends the coroutine for some time
        }
    }

    fun removeTask(task: Task) {
        viewModelScope.launch {
            platform().removeTask(task)
        }
    }

    fun cancelTask(task: Task) {
        viewModelScope.launch {
            platform().cancelTask(task)
        }
    }

    fun activeTasks(): Flow<Boolean> {
        return platform().activeTasks()
    }

    fun tasks(pid: Long): Flow<List<Task>> {
        return platform().getTasks(pid)
    }

    fun bookmarks(): Flow<List<Bookmark>> {
        return platform().getBookmarks()
    }

    fun startTask(task: Task, uuid: String) {
        viewModelScope.launch {
            platform().startTask(task, uuid)
        }
    }

    fun hasBookmark(url: String?): Flow<Boolean> {
        return platform().hasBookmark(url)
    }

    fun alterBookmark(
        url: String, title: String?, bitmap: Bitmap?,
        onWarningRequest: (Int, String) -> Unit
    ) {
        viewModelScope.launch {
            val platform = platform()
            var bookmark = platform.getBookmark(url)
            val bookmarkTitle = title ?: url
            if (bookmark != null) {
                platform.deleteBookmark(bookmark)

                onWarningRequest.invoke(
                    R.string.bookmark_removed,
                    bookmarkTitle
                )

            } else {
                val icon = platform().bytesBitmap(bitmap)
                bookmark = Bookmark(0, url, bookmarkTitle, icon)
                platform.storeBookmark(bookmark)
                onWarningRequest.invoke(
                    R.string.bookmark_added,
                    bookmarkTitle
                )
            }
        }
    }


    fun removeBookmark(bookmark: Bookmark) {
        viewModelScope.launch {
            platform().deleteBookmark(bookmark)
        }
    }


    fun reset() {
        viewModelScope.launch {
            platform().reset()
        }
        platform().cleanCache()
    }

    fun showTask(task: Task, onWarningRequest: (String) -> Unit) {
        viewModelScope.launch {
            platform().showTask(task, onWarningRequest)
        }
    }

    fun pnsDownloader(uri: String) {
        viewModelScope.launch {

            try {
                val pnsUri = Uri.parse(uri)

                val tasks = platform().tasks()
                val name = pnsUri.extractName()
                require(name.isNotBlank()) { "Invalid pns Uri [name]" }
                val mimeType = pnsUri.extractMimeType()
                require(mimeType.isNotBlank()) { "Invalid pns Uri [mimeType]" }
                val size = pnsUri.extractSize()

                val taskId = tasks.insert(
                    Task(
                        0, 0,
                        name = name,
                        mimeType = mimeType,
                        uri = uri,
                        size = size,
                        work = null,
                        active = true,
                        finished = false,
                        progress = 0f
                    )
                )
                val uuid = platform().pnsDownloader(taskId)
                tasks.work(taskId, uuid)
            } catch (throwable: Throwable) {
                debug("StateModel", throwable)
            }

        }
    }

    fun magnetDownloader(uri: String) {
        viewModelScope.launch {

            val magnetUri = parseMagnetUri(uri)


            var name = uri
            if (magnetUri.displayName != null) {
                name = checkNotNull(magnetUri.displayName)
            }

            val taskId = platform().storeTask(
                Task(
                    0, 0,
                    name = name,
                    mimeType = "vnd.android.document/directory",
                    uri = uri,
                    size = -1,
                    work = null,
                    active = true,
                    finished = false,
                    progress = 0f
                )
            )
            val uuid = platform().magnetDownloader(taskId)
            platform().setTaskWork(taskId, uuid)

        }
    }

    fun fileDownloader(
        uri: String, filename: String, mimeType: String, size: Long
    ) {
        viewModelScope.launch {
            val taskId = platform().storeTask(
                Task(
                    0, 0, filename, mimeType, uri, size,
                    null,
                    active = true,
                    finished = false,
                    progress = 0f
                )
            )
            val uuid = platform().fileDownloader(taskId)
            platform().setTaskWork(taskId, uuid)
        }
    }


    fun resetHomepage() {
        viewModelScope.launch {
            platform().removeHomepage()
        }
    }

    fun homePage(uri: String, title: String, bitmap: ImageBitmap?) {
        viewModelScope.launch {
            val icon = platform().bytesBitmap(bitmap)
            platform().setHomepage(uri, title, icon)
        }
    }


    fun homepageUri(default: String): Flow<String> {
        return platform().getHomepageUri(default)
    }

    fun purgeTasks() {
        viewModelScope.launch {
            platform().tasks().purge()
        }
    }


}