package tech.lp2p.thor.model

import android.Manifest
import android.annotation.SuppressLint
import android.content.ContentValues
import android.content.Context
import android.content.Context.CONNECTIVITY_SERVICE
import android.content.Intent
import android.content.Intent.CATEGORY_BROWSABLE
import android.graphics.Bitmap
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.provider.MediaStore
import android.webkit.CookieManager
import android.webkit.MimeTypeMap
import androidx.annotation.RequiresPermission
import androidx.compose.ui.graphics.ImageBitmap
import androidx.compose.ui.graphics.asImageBitmap
import androidx.core.net.toUri
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.PreferenceDataStoreFactory
import androidx.datastore.preferences.core.Preferences
import androidx.room.Room
import com.eygraber.uri.Uri
import io.github.alexzhirkevich.qrose.toByteArray
import io.github.remmerw.grid.WorkManager
import io.github.remmerw.nott.Store
import kotlinx.coroutines.flow.Flow
import kotlinx.io.files.Path
import kotlinx.io.files.SystemFileSystem
import okio.Path.Companion.toPath
import tech.lp2p.thor.R
import tech.lp2p.thor.data.Addresses
import tech.lp2p.thor.data.Bookmark
import tech.lp2p.thor.data.Bookmarks
import tech.lp2p.thor.data.Task
import tech.lp2p.thor.data.Tasks
import tech.lp2p.thor.data.homepage
import tech.lp2p.thor.data.homepageUri
import tech.lp2p.thor.debug
import java.util.regex.Pattern

class Platform(
    private val datastore: DataStore<Preferences>,
    private val tasks: Tasks,
    private val bookmarks: Bookmarks,
    private val store: Store,
    private val workManager: WorkManager,
    val context: Context
) {

    fun workManager(): WorkManager {
        return workManager
    }

    fun datastore(): DataStore<Preferences> {
        return datastore
    }

    fun tasks(): Tasks {
        return tasks
    }

    fun bookmarks(): Bookmarks {
        return bookmarks
    }

    fun store(): Store {
        return store
    }

    suspend fun removeHomepage() {
        tech.lp2p.thor.data.removeHomepage(datastore())
    }

    fun getHomepageUri(default: String): Flow<String> {
        return homepageUri(datastore(), default)
    }

    suspend fun setHomepage(uri: String, title: String, icon: ByteArray?) {
        homepage(datastore(), uri, title, icon)
    }

    suspend fun storeBookmark(bookmark: Bookmark) {
        bookmarks().insert(bookmark)
    }

    fun hasBookmark(url: String?): Flow<Boolean> {
        return bookmarks().hasBookmark(url)
    }

    suspend fun deleteBookmark(bookmark: Bookmark) {
        bookmarks().delete(bookmark)
    }

    suspend fun getBookmark(url: String): Bookmark? {
        return bookmarks().bookmark(url)
    }

    fun getBookmarks(): Flow<List<Bookmark>> {
        return bookmarks().bookmarks()
    }


    suspend fun startTask(task: Task, uuid: String) {
        tasks().active(task.id)
        tasks().work(task.id, uuid)
    }

    suspend fun removeTask(task: Task) {
        tasks().delete(task)
    }


    fun activeTasks(): Flow<Boolean> {
        return tasks().active()
    }

    fun getTasks(pid: Long): Flow<List<Task>> {
        return tasks().tasks(pid)
    }

    suspend fun storeTask(task: Task): Long {
        return tasks().insert(task)
    }

    suspend fun setTaskWork(taskId: Long, uuid: String) {
        tasks().work(taskId, uuid)
    }

    suspend fun setTaskActive(taskId: Long) {
        tasks().active(taskId)
    }

    suspend fun setTaskInactive(taskId: Long) {
        tasks().inactive(taskId)
    }

    suspend fun setTaskFinished(taskId: Long, url: String) {
        tasks().finished(taskId, url)
    }

    suspend fun setTaskFinished(taskId: Long) {
        tasks().finished(taskId)
    }

    suspend fun setTaskProgress(taskId: Long, progress: Float) {
        tasks().progress(taskId, progress)
    }

    suspend fun getTask(taskId: Long): Task {
        return tasks().task(taskId)
    }

    suspend fun createOrGetTask(
        pid: Long, name: String, mimeType: String, uri: String, size: Long
    ): Long {
        return tasks().createOrGet(pid, name, mimeType, uri, size)
    }

    fun downloadsUri(mimeType: String, name: String, path: String): android.net.Uri? {
        val contentValues = ContentValues()
        contentValues.put(MediaStore.Downloads.DISPLAY_NAME, name)
        contentValues.put(MediaStore.Downloads.MIME_TYPE, mimeType)
        contentValues.put(MediaStore.Downloads.RELATIVE_PATH, path)

        val contentResolver = context.contentResolver
        return contentResolver.insert(MediaStore.Downloads.EXTERNAL_CONTENT_URI, contentValues)
    }


    fun bytesBitmap(bitmap: ImageBitmap?): ByteArray? {
        if (bitmap == null) {
            return null
        }
        return bitmap.toByteArray(Bitmap.CompressFormat.PNG)
    }

    fun bytesBitmap(bitmap: Bitmap?): ByteArray? {
        return bytesBitmap(bitmap?.asImageBitmap())
    }

    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    fun isNetworkConnected(): Boolean {
        val connectivityManager = context.getSystemService(
            CONNECTIVITY_SERVICE
        ) as ConnectivityManager
        val nw = connectivityManager.activeNetwork ?: return false
        val actNw = connectivityManager.getNetworkCapabilities(nw)
        return actNw != null && (actNw.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET))
    }

    fun showTask(task: Task, onWarningRequest: (String) -> Unit) {
        try {
            val intent = Intent(Intent.ACTION_VIEW, task.uri.toUri())
            intent.addCategory(CATEGORY_BROWSABLE)
            intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
        } catch (_: Throwable) {
            onWarningRequest.invoke(
                context.getString(R.string.no_activity_found_to_handle_uri)
            )
        }
    }

    fun pruneWork() {  // cancel all jobs
        workManager().pruneWork()
    }


    suspend fun cancelTask(task: Task) {
        tasks().inactive(task.id)
        val uuid = task.work
        if (uuid != null) {
            workManager().cancel(uuid)
        }
    }

    suspend fun reset() {
        tasks().reset()

        pruneWork()

        CookieManager.getInstance().removeAllCookies(null)
        CookieManager.getInstance().flush()
    }


    fun pnsDownloader(taskId: Long): String {
        return workManager().start(PnsWorker(context, taskId))
    }


    fun fileDownloader(taskId: Long): String {
        return workManager().start(FileWorker(context, taskId))
    }

    fun magnetDownloader(taskId: Long): String {
        return workManager().start(MagnetWorker(context, taskId))
    }

    fun cacheDir(): Path {
        return Path(context.cacheDir.absolutePath)
    }

    fun cleanCache() {
        deleteRecursively(cacheDir(), false)
    }
}

fun initializePlatform(context: Context) {
    val dataStore = createDataStore(context)
    val tasks = createTasks(context)
    val bookmarks = createBookmarks(context)
    val store = createStore(context)
    platform = Platform(
        dataStore, tasks, bookmarks, store,
        WorkManager(), context
    )
}


internal fun deleteRecursively(path: Path, deleteDirectory: Boolean, mustExist: Boolean = false) {
    val isDirectory = SystemFileSystem.metadataOrNull(path)?.isDirectory ?: false
    if (isDirectory) {
        for (child in SystemFileSystem.list(path)) {
            deleteRecursively(child, true, mustExist)
        }
        if (deleteDirectory) {
            try {
                SystemFileSystem.delete(path, mustExist)
            } catch (_: Throwable) {
                debug("Thor", "Deletion failed for $path")
            }
        }
    } else {
        try {
            SystemFileSystem.delete(path, mustExist)
        } catch (_: Throwable) {
            debug("Thor", "Deletion failed for $path")
        }
    }
}


private fun uriTitle(url: String?): String {
    if (url.isNullOrBlank()) {
        return ""
    }
    val uri = Uri.parse(url)
    val host = uri.host!!
    if (host.isNotBlank()) {
        return host
    } else {
        val paths = uri.pathSegments
        return if (paths.isNotEmpty()) {
            paths[paths.size - 1]
        } else {
            ""
        }
    }
}

fun infoTitle(title: String?, url: String?): String {
    return if (title.isNullOrEmpty()) {
        uriTitle(url)
    } else {
        title
    }
}

private val CONTENT_DISPOSITION_PATTERN: Pattern = Pattern.compile(
    "attachment;\\s*filename\\s*=\\s*(\"?)([^\"]*)\\1(\"?);",
    Pattern.CASE_INSENSITIVE
)


fun parseContentDisposition(contentDisposition: String?): String? {
    try {
        val m = CONTENT_DISPOSITION_PATTERN.matcher(contentDisposition.toString())
        if (m.find()) {
            return m.group(2)
        }
    } catch (_: IllegalStateException) {
        // This function is defined as returning null when it can't parse the header
    }
    return null
}


fun mimeType(name: String): String {
    val extension = getExtension(name)
    if (extension.isNotEmpty()) {
        val mimeType = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension)
        if (mimeType != null) {
            return mimeType
        }
    } else {
        // no extension directory assumed
        return "vnd.android.document/directory"
    }
    return "application/octet-stream"
}


private fun getExtension(filename: String): String {
    if (filename.contains(".")) {
        return filename.substring(filename.lastIndexOf('.') + 1)
    }
    return ""
}


private fun createDataStore(context: Context): DataStore<Preferences> = createDataStore(
    producePath = { context.filesDir.resolve(dataStoreFileName).absolutePath }
)


private fun createTasks(ctx: Context): Tasks {
    return Room.databaseBuilder(
        ctx,
        Tasks::class.java, "tasks.db"
    ).fallbackToDestructiveMigration(true).build()
}


private fun createBookmarks(ctx: Context): Bookmarks {
    return Room.databaseBuilder(
        ctx,
        Bookmarks::class.java, "bookmarks.db"
    ).fallbackToDestructiveMigration(true).build()
}


private fun createStore(ctx: Context): Addresses {
    return Room.databaseBuilder(
        ctx,
        Addresses::class.java, "store.db"
    ).fallbackToDestructiveMigration(true).build()
}


private fun createDataStore(producePath: () -> String): DataStore<Preferences> =
    PreferenceDataStoreFactory.createWithPath(
        produceFile = { producePath().toPath() }
    )

internal const val dataStoreFileName = "settings.preferences_pb"

@SuppressLint("StaticFieldLeak")
internal var platform: Platform? = null

fun platform(): Platform = platform!!





