package tech.lp2p.thor.model

import android.content.Context
import android.net.Uri
import android.os.Environment
import io.github.remmerw.grid.Work
import io.github.remmerw.loki.core.StorageUnit
import io.github.remmerw.loki.download
import io.github.remmerw.loki.parseMagnetUri
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.io.asSink
import kotlinx.io.buffered
import tech.lp2p.thor.debug
import java.io.File

class MagnetWorker(val context: Context, val taskId: Long) : Work {


    private suspend fun uploadStorageUnit(
        storageUnit: StorageUnit,
        name: String,
        taskId: Long
    ) {
        val taskIds: MutableList<Long> = mutableListOf()

        val paths = storageUnit.relPaths()
        val lastIndex = paths.size - 1
        val dirs: MutableList<String> = mutableListOf()
        dirs.add(name)
        val parentTasks: MutableList<Long> = mutableListOf()
        parentTasks.add(taskId)
        paths.forEachIndexed { index, child ->
            if (index == lastIndex) {

                val targetUri = targetUri(storageUnit, dirs)

                val subTaskId = platform().createOrGetTask(
                    parentTasks.last(),
                    child,
                    mimeType(child),
                    targetUri.toString(),
                    storageUnit.size()
                )
                platform().setTaskActive(subTaskId)
                uploadFile(storageUnit, targetUri, subTaskId)
            } else {
                dirs.add(child)

                val path = Environment.DIRECTORY_DOWNLOADS + File.separator + java.lang.String.join(
                    (File.separator), dirs
                )
                val targetUri = platform().downloadsUri(
                    "vnd.android.document/directory", child, path
                )
                checkNotNull(targetUri)

                // directory
                val subTaskId = platform().createOrGetTask(
                    parentTasks.last(), path,
                    "vnd.android.document/directory", targetUri.toString(),
                    storageUnit.size(),
                )
                parentTasks.add(subTaskId)
                platform().setTaskActive(subTaskId)
                taskIds.add(subTaskId)
            }
        }

        taskIds.forEach { taskId ->
            platform().setTaskFinished(taskId)
        }
    }

    private
    fun targetUri(storageUnit: StorageUnit, dirs: List<String>): Uri {
        val name = storageUnit.name()

        val mimeType = mimeType(name)
        val path = Environment.DIRECTORY_DOWNLOADS + File.separator + java.lang.String.join(
            (File.separator), dirs
        )

        val targetUri = platform().downloadsUri(mimeType, name, path)
        checkNotNull(targetUri)
        return targetUri
    }

    private suspend fun uploadFile(
        storageUnit: StorageUnit, targetUri: Uri, taskId: Long
    ) {

        val contentResolver = context.contentResolver

        try {
            withContext(Dispatchers.IO) {
                contentResolver.openOutputStream(targetUri)!!.asSink().buffered().use { sink ->
                    storageUnit.transferTo(sink)
                }
            }
            platform().setTaskFinished(taskId)
        } catch (throwable: Throwable) {
            contentResolver.delete(targetUri, null, null)
            throw throwable
        }
    }


    override suspend fun run() {

        val platform = platform()
        platform.setTaskActive(taskId)

        try {
            withContext(Dispatchers.IO) {
                val task = platform.getTask(taskId)

                debug("Magnet", task.uri)
                val magnetUri = parseMagnetUri(task.uri)


                val cache = platform().cacheDir()

                var oldProgress = 0.0f

                val storage = download(
                    magnetUri = magnetUri,
                    directory = cache,
                    store = platform.store()
                ) { state ->
                    val completePieces = state.piecesComplete
                    val totalPieces = state.piecesTotal
                    if (totalPieces > 0 && completePieces >= 0) {
                        val progress = (completePieces * 1.0f) / (totalPieces * 1.0f)
                        debug("Magnet", "pieces : $completePieces/$totalPieces")
                        if (progress > oldProgress) {
                            oldProgress = progress
                            launch {
                                platform.setTaskProgress(taskId, progress)
                            }
                        }
                    }
                }


                if (isActive) {

                    storage.storageUnits().forEach { storageUnit ->
                        uploadStorageUnit(storageUnit, task.name, taskId)
                    }

                    val path = Environment.DIRECTORY_DOWNLOADS + File.separator
                    val targetUri = platform().downloadsUri(
                        mimeType = "vnd.android.document/directory",
                        name = task.name,
                        path = path
                    )
                    checkNotNull(targetUri)
                    platform.setTaskFinished(taskId, targetUri.toString())

                    storage.delete()
                }
            }
        } catch (throwable: Throwable) {
            debug("DownloadMagnetWorker", throwable)
        } finally {
            platform.setTaskInactive(taskId)
        }
    }
}