package tech.lp2p.thor.model

import android.content.Context
import android.os.Environment
import io.github.remmerw.grid.Work
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ensureActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import tech.lp2p.thor.debug
import java.io.InputStream
import java.io.OutputStream
import java.net.HttpURLConnection
import java.net.URL

class FileWorker(val context: Context, val taskId: Long) : Work {

    fun copy(
        inputStream: InputStream, outputStream: OutputStream,
        read: Long, size: Long, consumer: (Int) -> Unit = {}
    ): Long {
        var x = read
        val buf = ByteArray(Short.MAX_VALUE.toInt())
        var remember = 0
        var n: Int
        while ((inputStream.read(buf).also { n = it }) > 0) {
            outputStream.write(buf, 0, n)
            x += n.toLong()

            if (size > 0) {
                val percent = ((x * 100.0f) / size).toInt()
                if (percent > remember) {
                    remember = percent
                    consumer.invoke(percent)
                }
            }
        }
        return x
    }

    override suspend fun run() {


        val platform = platform()

        platform.setTaskActive(taskId)

        try {
            withContext(Dispatchers.IO) {

                val task = platform.getTask(taskId)
                val uri = task.uri

                val urlCon = URL(uri)

                val downloads = platform().downloadsUri(
                    mimeType = task.mimeType,
                    name = task.name,
                    path = Environment.DIRECTORY_DOWNLOADS
                )
                checkNotNull(downloads)
                val contentResolver = context.contentResolver

                try {
                    contentResolver.openOutputStream(downloads).use { os ->
                        checkNotNull(os)
                        HttpURLConnection.setFollowRedirects(false)

                        val huc = urlCon.openConnection() as HttpURLConnection

                        huc.readTimeout = 30000 // 30 sec
                        huc.connect()

                        huc.inputStream.use { fis ->
                            copy(fis, os, 0L, task.size) { progress: Int ->
                                ensureActive()
                                launch {
                                    platform.setTaskProgress(taskId, progress / 100.0f)
                                }
                            }
                        }
                    }
                    platform.setTaskFinished(taskId, downloads.toString())
                } catch (throwable: Throwable) {
                    contentResolver.delete(downloads, null, null)
                    throw throwable
                }
            }
        } catch (throwable: Throwable) {
            debug("DownloadFileWorker", throwable)
        } finally {
            platform.setTaskInactive(taskId)
        }
    }
}