package tech.lp2p.thor.model

import android.content.Context
import android.database.Cursor
import android.os.Environment
import android.provider.OpenableColumns
import androidx.work.CoroutineWorker
import androidx.work.Data
import androidx.work.ExistingWorkPolicy
import androidx.work.OneTimeWorkRequest
import androidx.work.OneTimeWorkRequestBuilder
import androidx.work.WorkManager
import androidx.work.WorkerParameters
import io.github.remmerw.idun.newIdun
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.io.asSink
import java.util.UUID
import kotlin.concurrent.atomics.ExperimentalAtomicApi


class DownloadPnsWorker(context: Context, params: WorkerParameters) :
    CoroutineWorker(context, params) {


    @OptIn(ExperimentalAtomicApi::class)
    override suspend fun doWork(): Result {

        val taskId = inputData.getLong(TID, 0)
        val tasks = platform().tasks()
        val task = tasks.task(taskId)

        tasks.active(taskId)
        try {
            withContext(Dispatchers.IO) {

                val request = task.uri
                val idun = newIdun(store = platform().store())

                val name = task.name
                val mimeType = task.mimeType


                val targetUri = platform().downloadsUri(
                    mimeType, name,
                    Environment.DIRECTORY_DOWNLOADS
                )
                checkNotNull(targetUri)
                val contentResolver = applicationContext.contentResolver
                checkNotNull(contentResolver)
                var offset = 0L

                val cursor: Cursor? = contentResolver.query(
                    targetUri, null, null, null, null
                )
                if (cursor != null) {
                    cursor.moveToFirst()
                    val value = cursor.getColumnIndex(OpenableColumns.SIZE)
                    offset = cursor.getLong(value)
                    cursor.close()
                }

                contentResolver.openOutputStream(targetUri, "wa").use { os ->
                    checkNotNull(os)

                    idun.transferTo(os.asSink(), request, offset) { progress ->
                        launch {
                            tasks.progress(taskId, progress)
                        }
                    }
                }
                tasks.finished(taskId, targetUri.toString())

            }
        } catch (_: Throwable) {
            return Result.failure()
        } finally {
            tasks.inactive(taskId)
        }

        return if (isStopped) {
            Result.retry()
        } else {
            Result.success()
        }
    }


    companion object {

        private const val TID = "tid"
        private fun getWork(taskId: Long): OneTimeWorkRequest {
            val data = Data.Builder().putLong(TID, taskId)

            return OneTimeWorkRequestBuilder<DownloadPnsWorker>()
                .addTag(DOWNLOADS_TAG)
                .setInputData(data.build())
                .build()
        }

        fun download(context: Context, taskId: Long): UUID {
            val work = getWork(taskId)
            WorkManager.Companion.getInstance(context).enqueueUniqueWork(
                taskId.toString(), ExistingWorkPolicy.KEEP, work
            )
            return work.id
        }
    }
}