package tech.lp2p.odin.ui

import androidx.compose.animation.animateContentSize
import androidx.compose.foundation.background
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.offset
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Delete
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.SwipeToDismissBox
import androidx.compose.material3.SwipeToDismissBoxState
import androidx.compose.material3.SwipeToDismissBoxValue
import androidx.compose.material3.Text
import androidx.compose.material3.minimumInteractiveComponentSize
import androidx.compose.material3.rememberSwipeToDismissBoxState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import io.github.remmerw.saga.Entity
import tech.lp2p.odin.Entities
import tech.lp2p.odin.model.StateModel
import tech.lp2p.odin.model.mediaResource
import tech.lp2p.odin.platform

@Composable
fun FileInfo(
    stateModel: StateModel,
    entity: Entity
) {

    var dismissState: SwipeToDismissBoxState? = null
    val threshold = .5f
    dismissState = rememberSwipeToDismissBoxState(
        confirmValueChange = { value ->
            if (value == SwipeToDismissBoxValue.EndToStart && dismissState!!.progress > threshold) {
                stateModel.delete(entity)
                true
            } else {
                false
            }
        },
        positionalThreshold = { it * threshold }
    )


    SwipeToDismissBox(
        modifier = Modifier.animateContentSize(),
        state = dismissState,
        backgroundContent = {

            Box(
                modifier = Modifier
                    .fillMaxSize()
                    .background(MaterialTheme.colorScheme.errorContainer),
                contentAlignment = Alignment.CenterEnd

            ) {
                Icon(
                    Icons.Filled.Delete,
                    stringResource(android.R.string.untitled),
                    modifier = Modifier.minimumInteractiveComponentSize()
                )
            }
        },
        enableDismissFromEndToStart = true,
        enableDismissFromStartToEnd = false,
    ) {
        FileItem(stateModel, entity)
    }

}

@Composable
fun FileItem(
    stateModel: StateModel,
    file: Entity
) {
    val attributes by stateModel.fileAttributes(file).collectAsStateWithLifecycle()
    val mimeType = attributes[Entities.MIME]!!.toString()
    val name = attributes[Entities.NAME]!!.toString()
    val size = attributes[Entities.SIZE]!!.toLong()

    var showInfo: Boolean by remember { mutableStateOf(false) }
    Column(
        modifier = Modifier
            .clickable { showInfo = true }
            .fillMaxWidth()
            .background(MaterialTheme.colorScheme.surface)

    ) {
        Row(
            modifier = Modifier
                .padding(8.dp)
                .fillMaxWidth()
        ) {


            Box(
                contentAlignment = Alignment.Center,
                modifier = Modifier
                    .background(
                        color = MaterialTheme.colorScheme.secondaryContainer,
                        shape = RoundedCornerShape(30.dp)
                    )
                    .size(48.dp)
            ) {
                Icon(
                    imageVector = mediaResource(mimeType),
                    contentDescription = stringResource(android.R.string.untitled),
                    modifier = Modifier.size(24.dp),
                    tint = MaterialTheme.colorScheme.secondary
                )
            }

            Column(
                modifier = Modifier
                    .offset(8.dp, 0.dp)
                    .align(Alignment.CenterVertically)
                    .weight(1.0f, true)
            ) {
                Text(
                    text = name, maxLines = 1,
                    style = MaterialTheme.typography.bodyMedium, softWrap = true
                )
                Text(
                    text = fileInfoSize(size), maxLines = 1,
                    style = MaterialTheme.typography.bodySmall, softWrap = true
                )
            }

        }
    }
    if (showInfo) {
        val uri = io.github.remmerw.idun.pnsUri(
            platform().peerId(), file.uid, name, mimeType, size
        )
        InfoDialog(
            uri = uri,
            onDismissRequest = {
                showInfo = false
            }
        )
    }
}


private fun fileInfoSize(size: Long): String {
    val fileSize: String

    if (size < 1000) {
        fileSize = size.toString()
        return "$fileSize B"
    } else if (size < 1000 * 1000) {
        fileSize = (size / 1000).toDouble().toString()
        return "$fileSize KB"
    } else {
        fileSize = (size / (1000 * 1000)).toDouble().toString()
        return "$fileSize MB"
    }
}

