package tech.lp2p.odin.model

import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.net.toUri
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.eygraber.uri.Uri
import io.github.remmerw.idun.extractMimeType
import io.github.remmerw.idun.extractName
import io.github.remmerw.idun.extractSize
import io.github.remmerw.saga.Entity
import io.github.remmerw.saga.Key
import io.github.remmerw.saga.Value
import io.github.remmerw.saga.normalizeValue
import io.github.remmerw.saga.toValue
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.absolutePath
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import tech.lp2p.odin.Entities
import tech.lp2p.odin.debug
import tech.lp2p.odin.platform

class StateModel() : ViewModel() {

    val online: Flow<Boolean> = flow {
        while (true) {
            val latestOnline = platform().isNetworkConnected()
            emit(latestOnline) // Emits the result of the request to the flow
            delay(1000) // Suspends the coroutine for some time
        }
    }

    val activeTasks: Flow<Boolean> = flow { // todo test
        val tasks = platform().tasks()
        while (true) {
            val latestActive = tasks.getChildren().filter { entity ->
                val active = tasks.getAttribute(entity, Entities.ACTIVE)
                if (active != null) {
                    if (active.toString().toBoolean()) {
                        return@filter true
                    }
                }
                return@filter false
            }.firstOrNull()
            emit(latestActive != null) // Emits the result of the request to the flow
            delay(3000) // Suspends the coroutine for some time
        }
    }

    fun fileAttributes(entity: Entity): StateFlow<Map<Key, Value>> {
        return platform().files().attributes(entity).stateIn(
            scope = viewModelScope,
            started = SharingStarted.Eagerly,
            initialValue = emptyMap()
        )
    }

    fun taskAttributes(entity: Entity): StateFlow<Map<Key, Value>> {
        return platform().tasks().attributes(entity).stateIn(
            scope = viewModelScope,
            started = SharingStarted.Eagerly,
            initialValue = emptyMap()
        )
    }

    fun showTask(task: Entity, onWarningRequest: (String) -> Unit) {
        platform().showTask(task, onWarningRequest)
    }

    fun sharePage(onWarningRequest: (String) -> Unit) {
        platform().sharePage(onWarningRequest)
    }


    fun removeTask(task: Entity) {
        platform().tasks().removeEntity(task)
    }

    fun purgeTasks() {
        val tasks = platform().tasks()

        tasks.getChildren().forEach { task ->
            tasks.setAttributes(
                task, mapOf(
                    Entities.ACTIVE to false.toValue(),
                    Entities.FINISHED to false.toValue()
                )
            )

        }
    }

    fun tasks(): StateFlow<List<Entity>> {
        return platform().tasks().children.stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(),
            initialValue = emptyList()
        )
    }

    fun cancelTask(task: Entity) {
        platform().cancelTask(task)
    }

    fun loadUrisFile(uris: List<PlatformFile>) {

        if (uris.isNotEmpty()) {

            for (uriStr in uris) {
                val uri = uriStr.absolutePath().toUri()

                val name = platform().fileName(uri)
                val mimeType = platform().mimeType(uri)
                val size = platform().fileSize(uri)


                platform().files().createEntity(
                    Entities.FILE, attributes = mapOf(
                        Entities.URI to uri.toString().toValue(),
                        Entities.NAME to normalizeValue(name),
                        Entities.MIME to mimeType.toValue(),
                        Entities.SIZE to size.toValue()
                    )
                )
            }
        }
    }

    var showTasks: Boolean by mutableStateOf(false)

    fun pnsDownloader(uri: String) {


        try {
            val pnsUri = Uri.parse(uri)

            val tasks = platform().tasks()
            val name = pnsUri.extractName()
            require(name.isNotBlank()) { "Invalid pns Uri [name]" }
            val mimeType = pnsUri.extractMimeType()
            require(mimeType.isNotBlank()) { "Invalid pns Uri [mimeType]" }
            pnsUri.extractSize()

            val task = tasks.createEntity(
                Entities.TASK,
                mapOf(
                    Entities.NAME to normalizeValue(name),
                    Entities.MIME to mimeType.toValue(),
                    Entities.URI to uri.toValue(),
                    Entities.ACTIVE to true.toValue(),
                    Entities.FINISHED to false.toValue(),
                    Entities.PROGRESS to 0F.toValue()
                )
            )
            platform().pnsDownloader(task)
        } catch (throwable: Throwable) {
            debug("StateModel", throwable)
        }


    }

    val numReservations: Flow<Int> = flow {
        while (true) {
            val latestRelays = platform().numReservations()
            emit(latestRelays) // Emits the result of the request to the flow
            delay(500) // Suspends the coroutine for some time
        }
    }

    val numConnections: Flow<Int> = flow {
        while (true) {
            val latestConnections = platform().numIncomingConnections()
            emit(latestConnections) // Emits the result of the request to the flow
            delay(500) // Suspends the coroutine for some time
        }
    }

    fun fileInfos(): StateFlow<List<Entity>> {
        return platform().fileInfos().stateIn(
            scope = viewModelScope,
            started = SharingStarted.WhileSubscribed(),
            initialValue = emptyList()
        )
    }

    fun delete(fileInfo: Entity) {
        viewModelScope.launch {
            platform().delete(fileInfo)
        }
    }

}