package tech.lp2p.odin.model

import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.core.net.toUri
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.eygraber.uri.Uri
import io.github.remmerw.idun.extractMimeType
import io.github.remmerw.idun.extractName
import io.github.remmerw.idun.extractSize
import io.github.vinceglb.filekit.PlatformFile
import io.github.vinceglb.filekit.absolutePath
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.launch
import tech.lp2p.odin.data.FileInfo
import tech.lp2p.odin.data.Task
import tech.lp2p.odin.debug
import tech.lp2p.odin.platform

class StateModel() : ViewModel() {

    val online: Flow<Boolean> = flow {
        while (true) {
            val latestOnline = platform().isNetworkConnected()
            emit(latestOnline) // Emits the result of the request to the flow
            delay(1000) // Suspends the coroutine for some time
        }
    }

    fun showTask(task: Task, onWarningRequest: (String) -> Unit) {
        viewModelScope.launch {
            platform().showTask(task, onWarningRequest)
        }
    }

    fun sharePage(onWarningRequest: (String) -> Unit) {
        viewModelScope.launch {
            platform().sharePage(onWarningRequest)
        }
    }

    fun activeTasks(): Flow<Boolean> {
        return platform().tasks().active()
    }

    fun removeTask(task: Task) {
        viewModelScope.launch {
            platform().tasks().delete(task)
        }
    }

    fun purgeTasks() {
        viewModelScope.launch {
            platform().tasks().purge()
        }
    }

    fun tasks(): Flow<List<Task>> {
        return platform().tasks().tasks()
    }

    fun cancelTask(task: Task) {
        viewModelScope.launch {
            platform().tasks().inactive(task.id)
        }
    }

    fun loadUrisFile(uris: List<PlatformFile>) {
        viewModelScope.launch {
            if (uris.isNotEmpty()) {

                val names = platform().fileNames().toMutableList()

                for (uriStr in uris) {
                    val uri = uriStr.absolutePath().toUri()

                    val displayName = platform().fileName(uri)
                    val mimeType = platform().mimeType(uri)
                    val size = platform().fileSize(uri)

                    val name = platform().getUniqueName(names, displayName)
                    try {

                        val fileInfo = FileInfo(
                            0, uri.toString(), name, mimeType, size
                        )
                        platform().files().storeFileInfo(fileInfo)
                    } catch (throwable: Throwable) {
                        debug("MainView", throwable)
                    } finally {
                        names.add(name) // just for safety
                    }
                }
            }
        }
    }

    fun startTask(task: Task, uuid: String) {
        viewModelScope.launch {
            val tasks = platform().tasks()
            tasks.active(task.id)
            tasks.work(task.id, uuid)
        }
    }

    var showTasks: Boolean by mutableStateOf(false)

    fun pnsDownloader(uri: String) {
        viewModelScope.launch {

            try {
                val pnsUri = Uri.parse(uri)

                val tasks = platform().tasks()
                val name = pnsUri.extractName()
                require(name.isNotBlank()) { "Invalid pns Uri [name]" }
                val mimeType = pnsUri.extractMimeType()
                require(mimeType.isNotBlank()) { "Invalid pns Uri [mimeType]" }
                val size = pnsUri.extractSize()

                val taskId = tasks.insert(
                    Task(
                        id = 0,
                        name = name,
                        mimeType = mimeType,
                        uri = uri,
                        size = size,
                        work = null,
                        active = true,
                        finished = false,
                        progress = 0f
                    )
                )
                val uuid = platform().pnsDownloader(taskId)
                tasks.work(taskId, uuid)
            } catch (throwable: Throwable) {
                debug("StateModel", throwable)
            }

        }
    }

    val numReservations: Flow<Int> = flow {
        while (true) {
            val latestRelays = platform().numReservations()
            emit(latestRelays) // Emits the result of the request to the flow
            delay(500) // Suspends the coroutine for some time
        }
    }

    val numConnections: Flow<Int> = flow {
        while (true) {
            val latestConnections = platform().numIncomingConnections()
            emit(latestConnections) // Emits the result of the request to the flow
            delay(500) // Suspends the coroutine for some time
        }
    }

    fun fileInfos(): Flow<List<FileInfo>> {
        return platform().fileInfos()
    }

    fun delete(fileInfo: FileInfo) {
        viewModelScope.launch {
            platform().delete(fileInfo)
        }
    }


}