package tech.lp2p.odin.model

import android.content.Context
import android.database.Cursor
import android.os.Environment
import android.provider.OpenableColumns
import com.eygraber.uri.Uri
import io.github.remmerw.grid.Work
import io.github.remmerw.idun.extractPeerId
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlinx.io.asSink
import tech.lp2p.odin.ODIN_PORT
import tech.lp2p.odin.debug
import tech.lp2p.odin.platform
import java.net.InetAddress
import java.net.InetSocketAddress


class PnsWorker(val context: Context, val taskId: Long) : Work {

    override suspend fun run() {

        val tasks = platform().tasks()
        val task = tasks.task(taskId)

        tasks.active(taskId)
        try {
            withContext(Dispatchers.IO) {

                val request = task.uri
                val peerId = Uri.parse(request).extractPeerId()
                val localPeer = peerId == platform().peerId()
                val idun = platform().idun()

                if (localPeer) {
                    // its the same peerId (testing purpose local machine)
                    idun.reachable(
                        peerId, InetSocketAddress(
                            InetAddress.getLoopbackAddress(), ODIN_PORT
                        )
                    )
                }


                val name = task.name
                val mimeType = task.mimeType


                val targetUri = platform().downloadsUri(
                    mimeType, name,
                    Environment.DIRECTORY_DOWNLOADS
                )
                checkNotNull(targetUri)
                val contentResolver = context.contentResolver
                checkNotNull(contentResolver)
                var offset = 0L

                val cursor: Cursor? = contentResolver.query(
                    targetUri, null, null, null, null
                )
                if (cursor != null) {
                    cursor.moveToFirst()
                    val value = cursor.getColumnIndex(OpenableColumns.SIZE)
                    offset = cursor.getLong(value)
                    cursor.close()
                }

                contentResolver.openOutputStream(targetUri, "wa").use { os ->
                    checkNotNull(os)

                    idun.transferTo(os.asSink(), request, offset) { progress ->
                        launch {
                            tasks.progress(taskId, progress)
                        }
                    }
                }
                tasks.finished(taskId, targetUri.toString())

            }
        } catch (throwable: Throwable) {
            debug("PnsWorker", throwable)
        } finally {
            tasks.inactive(taskId)
        }

    }


    companion object {

        fun download(context: Context, taskId: Long): String {
            val workManager = platform().workManager()
            return workManager.start(PnsWorker(context, taskId))

        }
    }
}