package tech.lp2p.odin.ui

import androidx.compose.foundation.background
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.safeDrawingPadding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.outlined.Download
import androidx.compose.material.icons.outlined.Downloading
import androidx.compose.material.icons.outlined.Home
import androidx.compose.material.icons.outlined.MoreVert
import androidx.compose.material3.Badge
import androidx.compose.material3.BadgedBox
import androidx.compose.material3.BottomAppBar
import androidx.compose.material3.BottomAppBarDefaults
import androidx.compose.material3.DropdownMenu
import androidx.compose.material3.DropdownMenuItem
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.FloatingActionButtonDefaults
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.ModalBottomSheet
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarDuration
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBar
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberModalBottomSheetState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.LocalFocusManager
import androidx.compose.ui.platform.LocalUriHandler
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import io.github.vinceglb.filekit.dialogs.FileKitMode
import io.github.vinceglb.filekit.dialogs.compose.rememberFilePickerLauncher
import tech.lp2p.odin.R
import tech.lp2p.odin.model.StateModel
import kotlin.uuid.ExperimentalUuidApi


@OptIn(ExperimentalMaterial3Api::class, ExperimentalUuidApi::class)
@Composable
fun MainView(stateModel: StateModel) {


    val launcher = rememberFilePickerLauncher(
        mode = FileKitMode.Multiple()
    ) { files ->
        if (files != null) {
            stateModel.loadUrisFile(files)
        }
    }

    var showMenu: Boolean by remember { mutableStateOf(false) }
    var showHome: Boolean by remember { mutableStateOf(false) }


    val snackbarHostState = remember { SnackbarHostState() }
    var warning: String by remember { mutableStateOf("") }

    if (warning.isNotBlank()) {
        LaunchedEffect(warning) {
            snackbarHostState.showSnackbar(
                message = warning,
                duration = SnackbarDuration.Short
            )
        }
    }

    val fileInfos by stateModel.fileInfos().collectAsState(emptyList())
    val numReservations by stateModel.numReservations.collectAsState(0)
    val numConnections by stateModel.numConnections.collectAsState(0)
    val hasActiveTasks by stateModel.activeTasks().collectAsState(false)
    val online by stateModel.online.collectAsState(true)


    val focusManager = LocalFocusManager.current
    val uriHandler = LocalUriHandler.current



    Scaffold(
        snackbarHost = {
            SnackbarHost(hostState = snackbarHostState)
        },
        topBar = {
            TopAppBar(
                title = {
                    Text(
                        text = stringResource(R.string.app_name)
                    )
                },
                navigationIcon = {
                    IconButton(onClick = { showHome = true }) {
                        Icon(
                            imageVector = Icons.Outlined.Home,
                            contentDescription = stringResource(R.string.home)
                        )
                    }
                },
                actions = {
                    IconButton(onClick = {
                        showMenu = true
                    }) {
                        Icon(
                            imageVector = Icons.Outlined.MoreVert,
                            contentDescription = stringResource(android.R.string.untitled)
                        )
                    }
                    DropdownMenu(
                        expanded = showMenu,
                        onDismissRequest = { showMenu = false })
                    {
                        DropdownMenuItem(
                            text = { Text(text = stringResource(R.string.documentation)) },

                            onClick = {
                                uriHandler.openUri("https://gitlab.com/lp2p/odin")
                            })

                    }
                },
                scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior()
            )
        },
        bottomBar = {
            BottomAppBar(
                actions = {

                    BadgedBox(
                        badge = {
                            Badge {
                                Text(text = numReservations.toString())
                            }
                        },
                        modifier = Modifier.padding(16.dp, 0.dp, 16.dp, 0.dp)
                    ) {
                        Icon(
                            painter = painterResource(R.drawable.access_point_network),
                            contentDescription = stringResource(android.R.string.untitled)
                        )
                    }

                    BadgedBox(
                        badge = {
                            Badge {
                                Text(text = numConnections.toString())
                            }
                        },
                        modifier = Modifier.padding(16.dp, 0.dp, 16.dp, 0.dp)
                    ) {
                        Icon(
                            painterResource(R.drawable.lan_connect),
                            contentDescription = stringResource(R.string.connections)
                        )
                    }


                    IconButton(
                        onClick = { stateModel.showTasks = true }) {
                        Icon(
                            imageVector = if (hasActiveTasks) {
                                Icons.Outlined.Downloading
                            } else {
                                Icons.Outlined.Download
                            },
                            contentDescription = stringResource(
                                R.string.downloads
                            )
                        )
                    }

                },
                floatingActionButton = {
                    FloatingActionButton(
                        onClick = { launcher.launch() },
                        containerColor = BottomAppBarDefaults.bottomAppBarFabColor,
                        elevation = FloatingActionButtonDefaults.bottomAppBarFabElevation()
                    ) {
                        Icon(
                            painter = painterResource(R.drawable.plus_thick),
                            contentDescription = stringResource(R.string.files)
                        )
                    }
                }
            )
        },
    ) { innerPadding ->


        val listState = rememberLazyListState()
        Column(
            modifier = Modifier
                .padding(innerPadding)
                .fillMaxSize()
                .pointerInput(Unit) {
                    detectTapGestures(onTap = {
                        focusManager.clearFocus()
                    })
                }
        ) {

            if (!online) {
                Text(
                    text = stringResource(R.string.offline),
                    style = MaterialTheme.typography.labelLarge,
                    color = MaterialTheme.colorScheme.error,
                    textAlign = TextAlign.Center,
                    modifier = Modifier
                        .fillMaxWidth()
                        .align(Alignment.CenterHorizontally)
                        .background(MaterialTheme.colorScheme.onError)
                )
            }

            LazyColumn(
                modifier = Modifier.fillMaxSize(),
                state = listState
            ) {
                items(items = fileInfos, key = { fileItem ->
                    fileItem.cid
                }) { fileInfo ->
                    FileInfo(stateModel, fileInfo) { msg -> warning = msg }
                }
            }
        }

    }


    if (showHome) {
        val sheetState = rememberModalBottomSheetState(
            skipPartiallyExpanded = true
        )
        ModalBottomSheet(
            sheetState = sheetState,
            onDismissRequest = { showHome = false }) {
            HomeView()
        }
    }

    if (stateModel.showTasks) {
        val sheetState = rememberModalBottomSheetState(
            skipPartiallyExpanded = true
        )
        ModalBottomSheet(
            containerColor = Color.Black.copy(0.3f),
            modifier = Modifier
                .fillMaxHeight()
                .safeDrawingPadding(),
            sheetState = sheetState,
            onDismissRequest = { stateModel.showTasks = false }) {

            TasksView(
                stateModel,
                onWarningRequest = { it ->
                    warning = it
                })
        }
    }
}