package tech.lp2p.odin

import android.Manifest
import android.annotation.SuppressLint
import android.content.ContentValues
import android.content.Context
import android.content.Context.CONNECTIVITY_SERVICE
import android.content.Intent
import android.content.Intent.CATEGORY_BROWSABLE
import android.database.Cursor
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.Uri
import android.provider.MediaStore
import android.provider.OpenableColumns
import androidx.annotation.RequiresPermission
import androidx.core.net.toUri
import androidx.datastore.core.DataStore
import androidx.datastore.preferences.core.PreferenceDataStoreFactory
import androidx.datastore.preferences.core.Preferences
import androidx.room.Room
import androidx.work.WorkManager
import io.github.remmerw.borr.Keys
import io.github.remmerw.borr.PeerId
import io.github.remmerw.borr.generateKeys
import io.github.remmerw.dagr.Data
import io.github.remmerw.idun.Idun
import io.github.remmerw.idun.Storage
import io.github.remmerw.idun.newIdun
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.runBlocking
import kotlinx.io.Buffer
import kotlinx.io.asSource
import kotlinx.io.buffered
import okio.Path.Companion.toPath
import tech.lp2p.odin.data.Addresses
import tech.lp2p.odin.data.FileInfo
import tech.lp2p.odin.data.Files
import tech.lp2p.odin.data.Task
import tech.lp2p.odin.data.Tasks
import tech.lp2p.odin.data.getPrivateKey
import tech.lp2p.odin.data.getPublicKey
import tech.lp2p.odin.data.setPrivateKey
import tech.lp2p.odin.data.setPublicKey
import tech.lp2p.odin.model.DownloadPnsWorker
import java.io.File
import java.net.InetSocketAddress
import java.net.NetworkInterface
import java.util.UUID
import kotlin.time.measureTime


class Platform(
    private val context: Context,
    private val tasks: Tasks,
    private val files: Files,
    private val idun: Idun
) {


    fun mimeType(uri: Uri): String {
        var mimeType = context.contentResolver.getType(uri)
        if (mimeType == null) {
            mimeType = "application/octet-stream"
        }
        return mimeType
    }

    fun nameWithoutExtension(file: String): String {
        val fileName = File(file).name
        val dotIndex = fileName.lastIndexOf('.')
        return if ((dotIndex == -1)) fileName else fileName.substring(0, dotIndex)
    }

    fun fileName(uri: Uri): String {
        var filename: String? = null

        val contentResolver = context.contentResolver

        contentResolver.query(
            uri,
            null, null, null, null
        ).use { cursor ->
            cursor!!.moveToFirst()
            val nameIndex = cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME)
            filename = cursor.getString(nameIndex)
        }


        if (filename == null) {
            filename = uri.lastPathSegment
        }

        if (filename == null) {
            filename = "file_name_not_detected"
        }

        return filename
    }

    fun fileSize(uri: Uri): Long {
        val contentResolver = context.contentResolver

        contentResolver.openFileDescriptor(uri, "r").use { fd ->
            return fd!!.statSize
        }
    }


    fun getUniqueName(names: List<String>, name: String): String {
        return getName(names, name, 0)
    }


    fun fileExtension(fullName: String): String {
        val dotIndex = fullName.lastIndexOf('.')
        return if ((dotIndex == -1)) "" else fullName.substring(dotIndex + 1)
    }


    fun getName(names: List<String>, name: String, index: Int): String {

        var searchName = name
        if (index > 0) {
            try {
                val base = nameWithoutExtension(name)
                val extension = fileExtension(name)
                if (extension.isEmpty()) {
                    searchName = "$searchName ($index)"
                } else {
                    val end = " ($index)"
                    if (base.endsWith(end)) {
                        val realBase = base.substring(0, base.length - end.length)
                        searchName = "$realBase ($index).$extension"
                    } else {
                        searchName = "$base ($index).$extension"
                    }
                }
            } catch (_: Throwable) {
                searchName = "$searchName ($index)" // just backup
            }
        }

        if (names.contains(searchName)) {
            return getName(names, name, (index + 1))
        }
        return searchName
    }

    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    fun isNetworkConnected(): Boolean {
        val connectivityManager = context.getSystemService(
            CONNECTIVITY_SERVICE
        ) as ConnectivityManager
        val nw = connectivityManager.activeNetwork ?: return false
        val actNw = connectivityManager.getNetworkCapabilities(nw)
        return actNw != null && (actNw.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
                || actNw.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET))
    }

    fun files(): Files {
        return files
    }

    fun idun(): Idun {
        return idun
    }

    fun peerId(): PeerId {
        return idun().peerId()
    }

    suspend fun fileNames(): List<String> {
        return files().fileNames()
    }

    suspend fun startup() {
        idun().startup(port = ODIN_PORT, object : Storage {
            override suspend fun getData(cid: Long, offset: Long): Data {

                val fileInfo = files().fileInfo(cid)


                if (fileInfo != null) {

                    val uri = fileInfo.uri.toUri()
                    val contentResolver = context.contentResolver
                    checkNotNull(contentResolver)
                    var size = 0L

                    val cursor: Cursor? = contentResolver.query(
                        uri, null, null, null, null
                    )
                    if (cursor != null) {
                        cursor.moveToFirst()
                        val value = cursor.getColumnIndex(OpenableColumns.SIZE)
                        size = cursor.getLong(value)
                        cursor.close()
                    }

                    val length = (size - offset)
                    val source = contentResolver.openInputStream(uri)!!.asSource().buffered()
                    source.skip(offset)
                    return Data(source, length)
                }
                return Data(Buffer(), 0)

            }
        })
    }

    fun numReservations(): Int {
        return idun().numReservations()
    }

    fun numIncomingConnections(): Int {
        return idun().numIncomingConnections()
    }

    fun fileInfos(): Flow<List<FileInfo>> {
        return files().filesDao().flowFileInfos()
    }

    suspend fun delete(fileInfo: FileInfo) {
        files().filesDao().delete(fileInfo)
    }

    fun observedAddresses(): List<InetSocketAddress> {
        return idun().publishedAddresses()
    }


    suspend fun publishPeeraddrs(
        addresses: List<InetSocketAddress>,
        maxPublifications: Int = 60,
        timeout: Int = 60
    ) {
        idun().publishAddresses(addresses, maxPublifications, timeout)
    }

    fun shutdown() {
        idun().shutdown()
    }

    fun downloadsUri(mimeType: String, name: String, path: String): Uri? {
        val contentValues = ContentValues()
        contentValues.put(MediaStore.Downloads.DISPLAY_NAME, name)
        contentValues.put(MediaStore.Downloads.MIME_TYPE, mimeType)
        contentValues.put(MediaStore.Downloads.RELATIVE_PATH, path)

        val contentResolver = context.contentResolver
        return contentResolver.insert(MediaStore.Downloads.EXTERNAL_CONTENT_URI, contentValues)
    }

    fun cancelTask(task: Task) {
        val uuid = workUUID(task)
        if (uuid != null) {
            WorkManager.getInstance(context).cancelWorkById(uuid)
        }
    }


    fun pruneWork() {
        WorkManager.getInstance(context).pruneWork()
    }


    fun pnsDownloader(taskId: Long): String {
        return DownloadPnsWorker.download(context, taskId).toString()
    }

    fun tasks(): Tasks {
        return tasks
    }

    fun showTask(task: Task, onWarningRequest: (String) -> Unit) {
        try {
            val intent = Intent(Intent.ACTION_VIEW, task.uri.toUri())
            intent.addCategory(CATEGORY_BROWSABLE)
            intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(intent)
        } catch (_: Throwable) {
            onWarningRequest.invoke(
                context.getString(R.string.no_app_found_to_handle_uri)
            )
        }
    }

    fun sharePageUri(uri: String, onWarningRequest: (String) -> Unit) {
        try {
            val intent = Intent(Intent.ACTION_SEND)
            intent.putExtra(
                Intent.EXTRA_SUBJECT, context.getString(R.string.share_link)
            )
            intent.putExtra(Intent.EXTRA_TEXT, uri)
            intent.setType("text/plain")
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            intent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)

            val chooser = Intent.createChooser(
                intent, context.getString(R.string.share)
            )

            chooser.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP)
            chooser.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
            context.startActivity(chooser)
        } catch (_: Throwable) {
            onWarningRequest.invoke(context.getString(R.string.no_app_found_to_handle_uri))
        }

    }

}


const val ODIN_PORT: Int = 5001
const val DOWNLOADS_TAG: String = "downloads"

@SuppressLint("StaticFieldLeak")
internal var platform: Platform? = null

fun platform(): Platform = platform!!


private fun workUUID(task: Task): UUID? {
    if (task.work != null) {
        return UUID.fromString(task.work)
    }
    return null
}

fun initializePlatform(context: Context) {

    val time = measureTime {

        val tasks = createTasks(context)
        val datastore = createDataStore(context)
        val files = createFiles(context)
        val store = createStore(context)

        val idun = newIdun(
            keys = keys(datastore),
            store = store
        )

        platform = Platform(
            context = context,
            tasks = tasks,
            files = files,
            idun = idun
        )
    }

    debug("App", "App started " + time.inWholeMilliseconds)
}


fun siteLocalAddresses(): List<InetSocketAddress> {
    val inetAddresses: MutableList<InetSocketAddress> = ArrayList()

    try {
        val interfaces = NetworkInterface.getNetworkInterfaces()
        for (networkInterface in interfaces) {
            if (networkInterface.isUp()) {
                val addresses = networkInterface.getInetAddresses()
                for (inetAddress in addresses) {
                    if (inetAddress.isSiteLocalAddress) {
                        inetAddresses.add(
                            InetSocketAddress(inetAddress, ODIN_PORT)
                        )
                    }
                }
            }
        }
    } catch (throwable: Throwable) {
        debug("Platform", throwable)
    }
    return inetAddresses
}


private fun createStore(context: Context): Addresses {
    return Room.databaseBuilder(
        context,
        Addresses::class.java, "store.db"
    ).fallbackToDestructiveMigration(true).build()
}

private fun createFiles(context: Context): Files {
    return Room.inMemoryDatabaseBuilder(context, Files::class.java).build()
}


private const val dataStoreFileName = "settings.preferences_pb"


private fun keys(datastore: DataStore<Preferences>): Keys {
    return runBlocking {
        val privateKey = getPrivateKey(datastore).first()
        val publicKey = getPublicKey(datastore).first()
        if (privateKey.isNotEmpty() && publicKey.isNotEmpty()) {
            return@runBlocking Keys(PeerId(publicKey), privateKey)
        } else {
            val keys = generateKeys()
            setPrivateKey(datastore, keys.privateKey)
            setPublicKey(datastore, keys.peerId.hash)
            return@runBlocking keys
        }
    }
}

private fun createDataStore(producePath: () -> String): DataStore<Preferences> =
    PreferenceDataStoreFactory.createWithPath(
        produceFile = { producePath().toPath() }
    )

private fun createDataStore(context: Context): DataStore<Preferences> =
    createDataStore(
        producePath = { context.filesDir.resolve(dataStoreFileName).absolutePath }
    )


private fun createTasks(ctx: Context): Tasks {
    return Room.databaseBuilder(
        ctx,
        Tasks::class.java, "tasks.db"
    ).fallbackToDestructiveMigration(true).build()
}