package tech.lp2p.odin

import android.Manifest
import android.content.Intent
import android.net.Uri
import android.os.Build
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.activity.enableEdgeToEdge
import androidx.annotation.RequiresPermission
import androidx.compose.foundation.isSystemInDarkTheme
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.safeDrawingPadding
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.dynamicDarkColorScheme
import androidx.compose.material3.dynamicLightColorScheme
import androidx.compose.material3.lightColorScheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalView
import androidx.core.net.toUri
import androidx.core.util.Consumer
import androidx.lifecycle.viewmodel.compose.viewModel
import tech.lp2p.odin.model.StateModel
import tech.lp2p.odin.ui.MainView

class Activity : ComponentActivity() {

    @RequiresPermission(Manifest.permission.ACCESS_NETWORK_STATE)
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)

        enableEdgeToEdge()

        setContent {
            val stateModel: StateModel = viewModel { StateModel() }


            DisposableEffect(Unit) {
                val listener = Consumer<Intent> { intent ->

                    val uri = intentUri(intent)
                    if (uri != null) {
                        when (uri.scheme) {
                            "pns" -> {
                                stateModel.pnsDownloader(uri.toString())
                                stateModel.showTasks = true
                            }
                        }
                    }
                }
                addOnNewIntentListener(listener)
                onDispose { removeOnNewIntentListener(listener) }
            }

            KeepScreenOn()

            val uri by remember { mutableStateOf(intentUri(intent)) }
            LaunchedEffect(uri) {
                if (uri != null) {
                    when (uri!!.scheme) {
                        "pns" -> {
                            stateModel.pnsDownloader(uri!!.toString())
                            stateModel.showTasks = true
                        }
                    }
                }
            }

            AppTheme {
                Box(modifier = Modifier.safeDrawingPadding()) {
                    MainView(stateModel)
                }
            }
        }
    }

    @Composable
    fun AppTheme(useDarkTheme: Boolean = isSystemInDarkTheme(), content: @Composable () -> Unit) {
        val context = LocalContext.current

        val lightingColorScheme = lightColorScheme()

        val colorScheme =
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.S) {
                if (useDarkTheme)
                    dynamicDarkColorScheme(context) else dynamicLightColorScheme(context)
            } else {
                lightingColorScheme
            }

        MaterialTheme(
            colorScheme = colorScheme,
            content = content
        )
    }

    @Composable
    fun KeepScreenOn() {
        val currentView = LocalView.current
        DisposableEffect(Unit) {
            currentView.keepScreenOn = true
            onDispose {
                currentView.keepScreenOn = false
            }
        }
    }


    fun intentUri(intent: Intent): Uri? {
        val action = intent.action
        if (Intent.ACTION_VIEW == action) {
            val uri = intent.data
            if (uri != null) {
                return uri
            }
        }
        if (Intent.ACTION_SEND == action) {
            if (intent.type == "text/plain") {
                val query = intent.getStringExtra(Intent.EXTRA_TEXT)

                try {
                    if (!query.isNullOrEmpty()) {
                        return query.toUri()
                    }
                } catch (throwable: Throwable) {
                    debug("Activity", throwable)
                }
            }
        }
        return null
    }
}


