// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'constants.dart';
import 'screen.dart';

enum TooltipIconButtonLayout { iconOnLeft, iconOnRight }

/// A button with an icon which displays a tooltip on hover.
class TooltipIconButton extends StatelessWidget {
  final void Function()? onPressedCallback;
  final String text;
  final IconData iconData;
  final double? elevation;
  final TooltipIconButtonLayout? layout;
  final Color? backgroundColor;

  const TooltipIconButton({
    super.key,
    required this.onPressedCallback,
    required this.text,
    required this.iconData,
    this.elevation = 1,
    this.layout,
    this.backgroundColor,
  });

  @override
  Widget build(BuildContext context) {
    final ColorScheme colorScheme = Theme.of(context).colorScheme;

    final Color disabledBackgroundColor = backgroundColor ?? colorScheme.surface;
    final Color enabledBackgroundColor = backgroundColor ?? colorScheme.surface;
    final Color foregroundColor = colorScheme.primary;
    final Color foregroundColorDisabled = Color.alphaBlend(
      colorScheme.primary.withValues(alpha: BS.disableControlOpacity),
      backgroundColor ?? colorScheme.surface,
    );
    Icon icon = Icon(
      iconData,
      color: onPressedCallback == null ? foregroundColorDisabled : foregroundColor,
      size: Screen.isCompact(context) ? BS.buttonIconSizeCompact : BS.buttonIconSizeStandard,
    );
    final Color labelColor = Color.alphaBlend(
      colorScheme.primary.withValues(alpha: BS.tappableMildOpacity),
      backgroundColor ?? colorScheme.surface,
    );

    return Material(
      type: MaterialType.transparency,
      child: Tooltip(
        message: text,
        child: ElevatedButton(
          onPressed: onPressedCallback,
          style:
              ElevatedButton.styleFrom(
                elevation: elevation,
                visualDensity: VisualDensity.compact,
                padding: EdgeInsets.only(left: BS.paddingMedium, right: BS.paddingMedium),
              ).copyWith(
                shape: WidgetStateProperty<RoundedRectangleBorder>.fromMap(
                  <WidgetStatesConstraint, RoundedRectangleBorder>{
                    WidgetState.disabled: RoundedRectangleBorder(
                      borderRadius: BorderRadius.circular(BS.cornerRadius),
                      side: BorderSide(color: foregroundColorDisabled),
                    ),
                    WidgetState.any: RoundedRectangleBorder(
                      borderRadius: BorderRadius.circular(BS.cornerRadius),
                      side: BorderSide(color: foregroundColor),
                    ),
                  },
                ),
                backgroundColor: WidgetStateProperty<Color>.fromMap(<WidgetStatesConstraint, Color>{
                  WidgetState.disabled: disabledBackgroundColor,
                  WidgetState.any: enabledBackgroundColor,
                }),
              ),
          child: Row(
            mainAxisSize: MainAxisSize.min,
            mainAxisAlignment: MainAxisAlignment.spaceBetween,
            children: [
              if (layout == TooltipIconButtonLayout.iconOnLeft)
                ...(() {
                  return [icon, const SizedBox(width: BS.paddingSmall)];
                }()),
              Flexible(
                child: Container(
                  constraints: BoxConstraints.loose(const Size.fromWidth(BS.buttonMaxWidth)),
                  child: Text(
                    text,
                    overflow: TextOverflow.ellipsis,
                    style: TextStyle(
                      color: onPressedCallback == null ? foregroundColorDisabled : labelColor,
                      fontWeight: FontWeight.bold,
                    ),
                  ),
                ),
              ),
              if (layout == TooltipIconButtonLayout.iconOnRight)
                ...(() {
                  return [const SizedBox(width: BS.paddingSmall), icon];
                }()),
            ],
          ),
        ),
      ),
    );
  }
}
