// SPDX-License-Identifier: GPL-3.0-only

import 'package:BendyStraw/app_state.dart';
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';

import 'constants.dart';

class TappableEditableTextField extends StatefulWidget {
  const TappableEditableTextField({
    super.key,
    required this.foregroundColor,
    required this.backgroundColor,
    required this.text,
    required this.onUpdate,
    this.tapToEditForegroundColor = Colors.white,
    this.tapToEditBackgroundColor = Colors.red,
    this.taptPromptText = 'Tap to edit',
    this.moveCaretToEndOfSelection = true,
    this.disableTapping = false,
    this.hintText,
    this.searchPlaylistName,
  });

  final Color foregroundColor;
  final Color backgroundColor;
  final Color tapToEditForegroundColor;
  final Color tapToEditBackgroundColor;
  final String text;
  final String taptPromptText;
  final String? searchPlaylistName;
  final Function(String) onUpdate;
  final bool moveCaretToEndOfSelection;
  final bool disableTapping;
  final String? hintText;

  @override
  State<TappableEditableTextField> createState() => _TappableEditableTextFieldState();
}

class _TappableEditableTextFieldState extends State<TappableEditableTextField> {
  late final TextEditingController _textEditingController =
      TextEditingController(text: widget.text);

  late final FocusNode _focusNode;

  final BorderRadius _borderRadius = BorderRadius.circular(BS.cornerRadius);

  late bool _areUnsavedChanges = false;
  late int _tapCount = 0;

  @override
  void initState() {
    super.initState();
    _focusNode = FocusNode();
  }

  @override
  void dispose() {
    // Clean up the focus node when the Form is disposed.
    _focusNode.dispose();

    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    // Initiate editing
    if (!widget.disableTapping && _tapCount == 2 && !_areUnsavedChanges) {
      _initiateEditing();
    }

    // Padding caclulated to make a visual match so that this widget lines up visually with
    // [StandardTableTitle].
    final double leftPadding = BS.paddingMedium - BS.paddingSuperSmall;

    final ColorScheme colorScheme = Theme.of(context).colorScheme;

    return TapRegion(
      onTapInside: widget.disableTapping
          ? null
          : (PointerDownEvent event) {
              if (_tapCount < 2) {
                _tapCount++;
              } else {
                _tapCount = 0;
              }
              setState(() {});
            },
      onTapOutside: widget.disableTapping
          ? null
          : (PointerDownEvent event) {
              _tapCount = 0;
              if (_areUnsavedChanges) {
                widget.onUpdate(_textEditingController.text);
              }

              FocusScopeNode focusScopeNode = FocusScope.of(context);
              if (!focusScopeNode.hasPrimaryFocus) {
                focusScopeNode.unfocus();
              }
              setState(() {});
            },
      child: Stack(
        children: [
          TextField(
            // Disabled unless we've been tapped twice, as we don't normally want it to react to
            // taps here, those are handled by the parent [TapRegion] widget.
            enabled: widget.disableTapping || _tapCount == 2,
            focusNode: _focusNode,
            style: TextStyle(
              fontSize: BS.fontSizeHeading,
              color: widget.foregroundColor,
              fontWeight: FontWeight.bold,
            ),
            controller: _textEditingController,
            inputFormatters: [
              /// Use this instead of [TextField.maxLength] as we don't
              /// want to display a character count.
              LengthLimitingTextInputFormatter(BS.playlistNameMaxLength)
            ],
            decoration: InputDecoration(
              fillColor: widget.backgroundColor,
              isCollapsed: true,
              hintText: widget.hintText,
              hintStyle: TextStyle(
                  color: Color.alphaBlend(
                      colorScheme.primary.withValues(alpha: BS.disableControlOpacity),
                      colorScheme.surface)),
              contentPadding: EdgeInsets.only(
                top: BS.paddingSmall,
                right: BS.paddingSmall,
                bottom: BS.paddingSmall,
                left: leftPadding,
              ),
              filled: true,
              border: OutlineInputBorder(
                borderRadius: _borderRadius,
                borderSide: const BorderSide(width: 0, style: BorderStyle.none),
              ),
            ),
            onChanged: (String value) {
              _areUnsavedChanges = true;
              if (widget.searchPlaylistName != null) {
                AppState.update('${widget.searchPlaylistName}-terms', value);
              }
              setState(() {});
            },
            onSubmitted: _onSubmitted,
          ),
          if (!widget.disableTapping && _tapCount == 1)
            Positioned.fill(
              child: TapRegion(
                child: Stack(
                  children: [
                    Positioned.fill(
                      child: ClipRRect(
                        borderRadius: _borderRadius,
                        child: Container(
                          decoration: BoxDecoration(
                            color: widget.tapToEditBackgroundColor,
                            borderRadius: _borderRadius,
                            border: Border.all(
                              color: widget.tapToEditForegroundColor,
                              strokeAlign: BorderSide.strokeAlignInside,
                            ),
                          ),
                        ),
                      ),
                    ),
                    Align(
                      alignment: Alignment.centerLeft,
                      child: Padding(
                        padding: EdgeInsets.only(
                          left: leftPadding,
                        ),
                        child: Text(
                          widget.taptPromptText,
                          overflow: TextOverflow.ellipsis,
                          style: TextStyle(
                            color: widget.tapToEditForegroundColor,
                            fontWeight: FontWeight.bold,
                          ),
                        ),
                      ),
                    ),
                  ],
                ),
              ),
            ),
        ],
      ),
    );
  }

  void _onSubmitted(String text) {
    widget.onUpdate(text);
    _tapCount = 0;
    _areUnsavedChanges = false;
  }

  void _initiateEditing() {
    // Focus field, this results in the contained text being selected which is not what we want.
    Future.microtask(() {
      _focusNode.requestFocus();
    });

    if (widget.moveCaretToEndOfSelection) {
      // After a delay (for [requestFocus] to complete), move the cursor to the end of the text.
      Future.delayed(Duration(milliseconds: 100), () {
        _textEditingController.selection =
            TextSelection.fromPosition(TextPosition(offset: _textEditingController.text.length));
      });
    }
  }
}
