// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';
import 'package:sqlite3/sqlite3.dart';

import 'app_state.dart';
import 'enum.dart';
import 'constants.dart';
import 'table_model.dart';

/// All the tables for a specific database.
/// These are specialised representations of the tables the app front end cares about, not a 1:1 mapping of everything in the *.db file.
/// A specific tab would listen to this.
class TablesModel with ChangeNotifier {
  Map<int, TableModel> _localPlaylists = {};
  Map<int, TableModel> _searchPlaylists = {};

  final TableModel _bookmarkedPlaylistsNotifier;

  final TableModel _channelSubscriptionsNotifier;

  final String dbPathHash;

  TablesModel({required this.dbPathHash})
    : _bookmarkedPlaylistsNotifier = TableModel(
        uid: BS.remotePlaylistsTableId,
        displayName: BS.knownTables[BS.remotePlaylistsTableId]!['displayName'],
        resultSet: ResultSet([], [], []),
        tableType: TableType.remotePlaylists,
        dbPathHash: dbPathHash,
      ),
      _channelSubscriptionsNotifier = TableModel(
        uid: BS.subscriptionsTableId,
        displayName: BS.knownTables[BS.subscriptionsTableId]!['displayName'],
        resultSet: ResultSet([], [], []),
        tableType: TableType.subscriptions,
        dbPathHash: dbPathHash,
      );

  void updateRemotePlaylistsResultSet(ResultSet resultSet) {
    AppState.debug('TableModel::updateRemotePlaylistsResultSet()');
    _bookmarkedPlaylistsNotifier.resultSet = resultSet;
    notifyListeners();
  }

  void updateChannelSubscriptionsResultSet(ResultSet resultSet) {
    AppState.debug('TableModel::updateChannelSubscriptionsResultSet()');
    _channelSubscriptionsNotifier.resultSet = resultSet;
    notifyListeners();
  }

  void setLocalPlaylists(Map<int, TableModel> playlists) {
    AppState.debug('TableModel::setLocalPlaylists()');
    _localPlaylists = playlists;
    AppState.debug('TablesModel::_localPlaylists.length: ${_localPlaylists.length}');
    //_localPlaylists = Map.from(playlists);
    notifyListeners();
  }

  void setSearchPlaylists(Map<int, TableModel> playlists) {
    AppState.debug('TableModel::setSearchPlaylists()');
    _searchPlaylists = playlists;
    AppState.debug('TablesModel::_searchPlaylists.length: ${_searchPlaylists.length}');
    //_searchPlaylists = Map.from(playlists);
    notifyListeners();
  }

  Map<int, TableModel> get localPlaylists {
    return _localPlaylists;
  }

  Map<int, TableModel> get searchPlaylists {
    return _searchPlaylists;
  }

  TableModel get bookmarkedPlaylistsNotifier {
    return _bookmarkedPlaylistsNotifier;
  }

  TableModel get channelSubscriptionsNotifier {
    return _channelSubscriptionsNotifier;
  }
}
