// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';
import 'package:sqlite3/sqlite3.dart' as sqlite3;

import 'app_state.dart';
import 'constants.dart';
import 'enum.dart';
import 'location.dart';
import 'playlist_header_widget.dart';
import 'search_info_button.dart';
import 'selected_items_buttons_widget.dart';
import 'simple_playlist_header_widget.dart';
import 'table_data_widget.dart';
import 'table_model.dart';

/// A widget for displaying and managing a single database table.
///
/// Put together the varius parts needed to display a table from the database:
/// - [PlaylistHeaderWidget]: The editable name of the playlist and buttons to perform actions on the playlist as a whole
/// - [TableDataWidget]: The data contained within the table
/// - [SelectedItemsButtonsWidget]: When items (eg streams) are selected in a table, these buttons perform action (eg copy/move) on those items
class TableWidget extends StatefulWidget {
  final String dbPath;

  final TableModel tableNotifier;

  const TableWidget({super.key, required this.tableNotifier, required this.dbPath});

  @override
  State<TableWidget> createState() => _TableWidgetState();
}

class _TableWidgetState extends State<TableWidget> {
  final expansibleController = ExpansibleController();
  bool haveRevealedFirstNonEmptySearchResult = false;

  @override
  void initState() {
    expansibleController.addListener(_expansibleControllerListener);

    super.initState();
  }

  /// The [Expansible] is drawn with different bottom padding depending on whether it is expanded or collapsed. The change in padding wasn't happening consistently though, so we use [setState] to force a rebuild.
  void _expansibleControllerListener() {
    Future.microtask(() {
      if (mounted) {
        setState(() {});
      }
    });
  }

  @override
  void dispose() {
    expansibleController.removeListener(_expansibleControllerListener);
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    late String pageStorageKeyPrefix = widget.tableNotifier.tablePath;

    ColorScheme colorScheme = Theme.of(context).colorScheme;

    late Color tappableBackgroundColor = Color.alphaBlend(
      colorScheme.primary.withValues(alpha: 0.07),
      colorScheme.surface,
    );
    late Color outlineColor = Color.alphaBlend(
      colorScheme.primary.withValues(alpha: 0.2),
      colorScheme.surface,
    );
    late Color dataTableBackgroundColor = Color.alphaBlend(
      colorScheme.tertiary.withValues(alpha: 0.0666),
      colorScheme.surface,
    );

    TableType tableType = widget.tableNotifier.tableType;
    dynamic tableUid = widget.tableNotifier.uid;
    String displayName = widget.tableNotifier.displayName;
    sqlite3.ResultSet resultSet = widget.tableNotifier.resultSet;

    TextStyle debugTextStyle = TextStyle(
      fontSize: BS.fontSizeExtraSmall,
      color: colorScheme.tertiary.withValues(alpha: BS.tappableMildOpacity),
    );

    /// If this is the search table and it is non-empty for the first time, open the expansible to
    /// make the results obvious for the user.
    if (tableType == TableType.search &&
        !haveRevealedFirstNonEmptySearchResult &&
        resultSet.isNotEmpty) {
      if (!expansibleController.isExpanded) {
        haveRevealedFirstNonEmptySearchResult = true;
        expansibleController.expand();
      }
    }

    Location tableLocation = Location(
      tableType: tableType,
      tableId: tableUid,
      dbPath: widget.dbPath,
    );

    return Padding(
      padding: EdgeInsetsGeometry.only(
        top: BS.paddingSmall,
        right: BS.paddingSmall,
        bottom: expansibleController.isExpanded ? BS.paddingMedium : 0,
        left: BS.paddingSmall,
      ),
      child: Container(
        padding: EdgeInsetsGeometry.only(
          top: BS.paddingSmall,
          right: BS.paddingMedium,
          bottom: BS.paddingSmall,
          left: BS.paddingSmall,
        ),
        child: Expansible(
          /// Unique key to avoid redraw errors with values not changing, loss of
          /// scroll position, collapsed state etc.
          key: PageStorageKey<String>('${pageStorageKeyPrefix}_Expansible'),

          controller: expansibleController,
          duration: AppState.getPreference('disableUiAnimations')
              ? Duration.zero
              : Duration(milliseconds: BS.expansibleDefaultDurationMs),

          headerBuilder: (context, animation) => Container(
            decoration: BoxDecoration(
              borderRadius: BorderRadius.only(
                topLeft: Radius.circular(BS.cornerRadius),
                topRight: Radius.circular(BS.cornerRadius),
              ),
              border: Border.all(color: outlineColor),
            ),
            child: Stack(
              children: [
                Positioned.fill(
                  child: TapRegion(
                    onTapInside: _onExpandTap,
                    child: Container(color: tappableBackgroundColor),
                  ),
                ),
                Padding(
                  padding: EdgeInsets.only(
                    top: BS.paddingSmall,
                    right: BS.paddingMedium,
                    bottom: BS.paddingExtraSmall,
                  ),
                  child: Row(
                    crossAxisAlignment: CrossAxisAlignment.start,
                    spacing: BS.paddingMedium,
                    children: [
                      ///
                      ///
                      /// Rest of the header plus the main content.
                      ///
                      Expanded(
                        child:
                            (tableType == TableType.localPlaylist || tableType == TableType.search)
                            ? PlaylistHeaderWidget(
                                key: PageStorageKey<String>(
                                  '${pageStorageKeyPrefix}_PlaylistHeaderWidget',
                                ),
                                name: displayName,
                                dbPath: widget.dbPath,
                                tableId: tableUid,
                                tableType: tableType,
                                itemCount: resultSet.length,
                              )
                            // Top tables (bookmarked playlists/channels)
                            : SimplePlaylistHeaderWidget(
                                name: displayName,
                                tableType: tableType,
                                itemCount: resultSet.length,
                              ),
                      ),
                    ],
                  ),
                ),

                /// Expand/contract icon
                Positioned.directional(
                  end: BS.paddingExtraSmall,
                  bottom: BS.paddingExtraSmall,
                  textDirection: TextDirection.ltr,
                  child: IgnorePointer(
                    child: Icon(
                      expansibleController.isExpanded ? Icons.expand_less : Icons.expand_more,
                      color: Color.alphaBlend(
                        colorScheme.primary.withValues(alpha: BS.tappableMildOpacity),
                        colorScheme.surface,
                      ),
                    ),
                  ),
                ),

                /// Search info icon (handles info dialog too)
                if (tableType == TableType.search)
                  Positioned.directional(
                    end: BS.paddingExtraSmall,
                    top: BS.paddingMedium,
                    textDirection: TextDirection.ltr,
                    child: SearchInfoButton(),
                  ),
              ],
            ),
          ),

          bodyBuilder: (context, animation) => Container(
            decoration: BoxDecoration(
              border: Border.all(color: outlineColor),
              borderRadius: BorderRadius.only(
                bottomLeft: Radius.circular(BS.cornerRadius),
                bottomRight: Radius.circular(BS.cornerRadius),
              ),
            ),
            child: Column(
              children: [
                Container(
                  color: dataTableBackgroundColor,
                  child: TableDataWidget(
                    key: PageStorageKey<String>('${pageStorageKeyPrefix}_TableDataWidget'),
                    resultSet: resultSet,
                    location: tableLocation,
                    tableNotifier: widget.tableNotifier,
                  ),
                ),
                Padding(
                  padding: const EdgeInsets.only(
                    top: BS.paddingMedium,
                    right: BS.paddingMedium,
                    bottom: BS.paddingMedium,
                    left: BS.paddingMedium + BS.paddingSmall,
                  ),
                  child: Column(
                    crossAxisAlignment: CrossAxisAlignment.stretch,
                    children: [
                      Row(
                        mainAxisSize: MainAxisSize.min,
                        children: [
                          SelectedItemsButtonsWidget(
                            location: tableLocation,

                            tableNotifier: widget.tableNotifier,
                          ),
                        ],
                      ),
                      if (!AppState.getPreference('tableUidIsHidden'))
                        Align(
                          alignment: Alignment.bottomRight,
                          child: Padding(
                            padding: const EdgeInsets.only(top: BS.paddingMedium),
                            child: Column(
                              crossAxisAlignment: CrossAxisAlignment.end,
                              children: [
                                Text(
                                  'Type: ${tableType.name}',
                                  textAlign: TextAlign.end,
                                  style: debugTextStyle,
                                ),
                                Text(
                                  'UID: ${tableUid}',
                                  textAlign: TextAlign.end,
                                  style: debugTextStyle,
                                ),
                                Text(
                                  'Key: ${widget.tableNotifier.tablePath}',
                                  textAlign: TextAlign.end,
                                  style: debugTextStyle,
                                ),
                              ],
                            ),
                          ),
                        ),
                    ],
                  ),
                ),
              ],
            ),
          ),
        ),
      ),
    );
  }

  void _onExpandTap(PointerDownEvent event) {
    if (expansibleController.isExpanded) {
      expansibleController.collapse();
    } else {
      expansibleController.expand();
    }
  }
}
