// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';
import 'package:sqlite3/sqlite3.dart';

import 'enum.dart';

class TableModel with ChangeNotifier {
  /// Can be [String] (Channel Subscriptions and Remote Playlists) or [int] (Local Playlists
  /// including Search).
  final dynamic uid;
  final TableType tableType;
  final String dbPathHash;
  final String tablePath;
  String displayName;
  ResultSet resultSet;
  List<int> _selectedRows = [];
  int primaryKeyColumnIndex;

  TableModel({
    required this.uid,
    required this.displayName,
    required this.dbPathHash,
    required this.resultSet,
    required this.tableType,
  }) : tablePath = dbPathHash + tableType.name + uid.toString(),
       primaryKeyColumnIndex = tableType.columns.indexWhere(
         (columnInfo) => columnInfo.isPrimaryKey,
       );

  set selectedRows(List<int> rowIndexOrUid) {
    _selectedRows = List.of(rowIndexOrUid);
    notifyListeners();
  }

  get selectedRows => _selectedRows;

  void clearRowSelection() => selectedRows = [];

  void selectAllRows() {
    int primaryKeyColumnIndex = tableType.columns.indexWhere(
      (columnInfo) => columnInfo.isPrimaryKey,
    );

    /// As we use the [selectedRows] setter, [notifyListeners] will be called.
    switch (tableType) {
      /// Use row indices
      case TableType.localPlaylist:
      case TableType.search:
        selectedRows = List.generate(resultSet.length, (int rowIndex) => rowIndex);

      /// Use row uids
      case TableType.remotePlaylists:
      case TableType.subscriptions:
        selectedRows = resultSet.rows.map((row) => row[primaryKeyColumnIndex] as int).toList();
    }
  }

  /// Depending on the [tableType] rows can be selected by their row index (in cases where duplicate
  /// items are allowed like local playlists) or their uid (in cases where rows are unique). This
  /// function works the same way in either situation.
  void setRowIsSelected({required int rowIndexOrUid, required bool select}) {
    if (select) {
      if (!_selectedRows.contains(rowIndexOrUid)) {
        _selectedRows.add(rowIndexOrUid);
        notifyListeners();
      }
    } else {
      if (_selectedRows.contains(rowIndexOrUid)) {
        _selectedRows.remove(rowIndexOrUid);
        notifyListeners();
      }
    }
  }
}
