// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'constants.dart';
import 'db_manager.dart';
import 'enum.dart';
import 'location.dart';
import 'screen.dart';

class TableHeader extends StatefulWidget {
  TableHeader({
    super.key,
    required this.location,
    required this.cellLayout,
    required this.primaryKeyColumnIndex,
    required this.primaryKeyColumnIsHidden,
    this.rightPadding = 0,
    this.sortableColor = Colors.yellow,
    this.nonSortableColor = Colors.red,
    this.triStateAllRowsAreSelected = false,
    this.selectAllOrNone,
  }) : columnsInfoList = location.tableType!.columns
           .where(
             (ColumnInfo columnInfo) =>
                 columnInfo.isVisible && !(primaryKeyColumnIsHidden && columnInfo.isPrimaryKey),
           )
           .toList();

  static const double height = 50;

  final Location location;
  final List<ColumnInfo> columnsInfoList;
  final Map<String, dynamic> cellLayout;
  final int primaryKeyColumnIndex;
  final double rightPadding;

  final Color sortableColor;
  final Color nonSortableColor;
  final bool? triStateAllRowsAreSelected;
  final bool primaryKeyColumnIsHidden;
  final ValueChanged? selectAllOrNone;

  @override
  State<TableHeader> createState() => _TableHeaderState();
}

class _TableHeaderState extends State<TableHeader> {
  ColumnSort columnSortedBy = ColumnSort.none;
  int currentSortedColumnIndex = -1;

  @override
  Widget build(BuildContext context) {
    ColorScheme colorScheme = Theme.of(context).colorScheme;
    final Color headerRowBorderColor = Color.alphaBlend(
      colorScheme.secondary.withValues(alpha: 0.2),
      colorScheme.surface,
    );
    final BorderSide headerRowBorderSide = BorderSide(
      color: headerRowBorderColor,
      width: BS.accentBorderThickness,
    );
    final bool screenIsCompact = Screen.isCompact(context);

    return Container(
      decoration: BoxDecoration(border: Border(bottom: headerRowBorderSide)),
      height: TableHeader.height,
      child: Padding(
        padding: EdgeInsets.only(right: widget.rightPadding),
        child: Row(
          children: [
            Checkbox(
              tristate: true,
              value: widget.triStateAllRowsAreSelected,
              onChanged: widget.selectAllOrNone,
            ),

            ...List.generate(widget.columnsInfoList.length, (cellIndex) {
              ColumnInfo columnInfo = widget.columnsInfoList[cellIndex];

              Widget paddedIcon = Padding(
                padding: EdgeInsets.only(
                  left: columnInfo.numeric
                      ? screenIsCompact
                            ? BS.paddingSmall
                            : BS.paddingMedium
                      : 0,
                  right: columnInfo.numeric
                      ? 0
                      : screenIsCompact
                      ? BS.paddingSmall
                      : BS.paddingMedium,
                ),
                child: cellIndex != currentSortedColumnIndex
                    ? Icon(
                        Icons.sort,
                        color: colorScheme.primary,
                        size: screenIsCompact
                            ? BS.buttonIconSizeCompact
                            : BS.buttonIconSizeStandard,
                      )
                    : switch (columnSortedBy) {
                        ColumnSort.none => Icon(Icons.sort, color: colorScheme.primary),
                        ColumnSort.ascending => Icon(Icons.expand_less, color: colorScheme.primary),
                        ColumnSort.descending => Icon(
                          Icons.expand_more,
                          color: colorScheme.primary,
                        ),
                      },
              );

              /// Cell
              return Expanded(
                flex: columnInfo.displaySize.flex,
                child: InkWell(
                  onTap: columnInfo.canSortOn
                      ? () {
                          _sortPlaylistTableByColumnIndex(
                            index: cellIndex,

                            /// Unless already sorted by ascending, sort by ascending
                            byAscending: columnSortedBy != ColumnSort.ascending,
                          );
                        }
                      : null,
                  child: DecoratedBox(
                    decoration: BoxDecoration(
                      border: BoxBorder.fromLTRB(
                        //borderRadius: _borderRadius,
                        left: BorderSide(
                          color: headerRowBorderColor,
                          width: 2,
                          strokeAlign: BorderSide.strokeAlignInside,
                        ),
                      ),
                    ),
                    child: Row(
                      children: [
                        if (columnInfo.canSortOn && columnInfo.numeric) paddedIcon,
                        Expanded(
                          child: Align(
                            alignment: columnInfo.numeric
                                ? Alignment.centerRight
                                : Alignment.centerLeft,
                            child: Padding(
                              padding: widget.cellLayout['padding'],
                              child: Text(
                                columnInfo.displayName,
                                overflow: TextOverflow.ellipsis,
                                style: widget.cellLayout['textStyle'].copyWith(
                                  color: columnInfo.canSortOn
                                      ? widget.sortableColor
                                      : widget.nonSortableColor,
                                  fontWeight: FontWeight.bold,
                                ),
                              ),
                            ),
                          ),
                        ),
                        if (columnInfo.canSortOn && !columnInfo.numeric) paddedIcon,
                      ],
                    ),
                  ),
                ),
              );
            }),
          ],
        ),
      ),
    );
  }

  /// Sort a playlist based on the contents of a column.
  ///
  /// Using the contents of the column at [columnIndex], sort the playlist in
  /// ascending order if [byAscending] is `true`, otherwise descending order.
  void _sortPlaylistTableByColumnIndex({required int index, required bool byAscending}) {
    dynamic columnConfig = widget.columnsInfoList[index];
    DbManager.sortLocalPlaylistByColumn(
      dbPath: widget.location.dbPath,
      playlistUid: widget.location.tableId,
      sortColumnIndex: index,
      isNumeric: columnConfig.numeric,
      streamUidColumnIndex: widget.primaryKeyColumnIndex,
      ascending: byAscending,
    );
    setState(() {
      currentSortedColumnIndex = index;
      columnSortedBy = byAscending ? ColumnSort.ascending : ColumnSort.descending;
    });
  }
}
