// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'constants.dart';
import 'database_viewer_widget.dart';
import 'databases.dart';
import 'db_manager.dart';
import 'tab_widget.dart';
import 'tables_model.dart';
import 'tooltip_icon_button.dart';
import 'utils.dart';

/// Manage tabs.
///
/// Multiple database files can be opened at once. Each database is displayed
/// in a tab. The tabs are added/removed here.
class TabManagerWidget extends StatefulWidget {
  final Databases databasesNotifier;

  const TabManagerWidget({
    super.key,
    required this.databasesNotifier,
  });

  @override
  State<TabManagerWidget> createState() => _TabManagerWidgetState();
}

class _TabManagerWidgetState extends AppState<TabManagerWidget> with TickerProviderStateMixin {
  @override
  List<String>? listenForChanges = ['preferences', 'currentSelectedDbPath'];

  late TabController _tabController = TabController(length: 0, vsync: this);
  final PageStorageBucket _bucket = PageStorageBucket();

  @override
  Widget build(BuildContext context) {
    _updateTabController();

    /// If no databases (tabs) are open, navigate back to start page
    if (widget.databasesNotifier.isEmpty) {
      Future.microtask(_returnToStartPage);
    }

    String currentSelectedDbPath = AppState.get('currentSelectedDbPath');
    ColorScheme currentSelectedColorScheme =
        DbManager.getDbCurrentThemeData(currentSelectedDbPath).colorScheme;

    return Column(
      children: [
        Container(
          /// Set tab bar background to hint which tab is open
          color: Color.alphaBlend(currentSelectedColorScheme.primary.withValues(alpha: 0.15),
              currentSelectedColorScheme.surface),

          /// This SizedBox is here to fix non-left-aligned tabs on Android
          child: Padding(
            padding: const EdgeInsets.only(
              right: BS.tabBarRightPadding,
            ),
            child: SizedBox(
              width: double.infinity,
              child: TabBar(
                controller: _tabController,
                padding: EdgeInsets.zero,
                indicatorPadding: EdgeInsets.zero,
                labelPadding: EdgeInsets.zero,
                //isScrollable: true,
                //tabAlignment: TabAlignment.start,
                dividerColor: currentSelectedColorScheme.secondaryContainer,
                dividerHeight: 2,
                indicatorColor: Colors.transparent,

                ///
                /// Tabs
                ///
                tabs: widget.databasesNotifier.keys
                    .map<Widget>((dbPath) => Theme(
                          data: DbManager.getDbCurrentThemeData(dbPath),
                          child: TabWidget(
                              text: DbManager.getPrettyName(dbPath),
                              isSelected: (dbPath == currentSelectedDbPath),
                              onPressed: () {
                                _confirmCloseTabAndDatabase(dbPath);
                              }),
                        ))
                    .toList(),
              ),
            ),
          ),
        ),
        Expanded(

            ///
            /// TabBarViews
            ///
            child: PageStorage(
          bucket: _bucket,
          child: TabBarView(
            controller: _tabController,
            children: widget.databasesNotifier.keys.map<Widget>((dbPath) {
              TablesModel tablesNotifier =
                  widget.databasesNotifier.getByKey(dbPath)!.tablesNotifier;

              return ListenableBuilder(
                  listenable: tablesNotifier,
                  builder: (context, child) {
                    return DatabaseViewerWidget(
                      key: PageStorageKey<String>('${getMd5Hash(dbPath)}_TabBarView'),
                      tablesNotifier: tablesNotifier,
                      dbPath: dbPath,
                    );
                  });
            }).toList(),
          ),
        ))
      ],
    );
  }

  /// Start the close tab/database process by displaying a dialog with a
  /// confirm button, which will close the tab/database when pressed.
  void _confirmCloseTabAndDatabase(String dbPath) {
    AppState.debug('TabManagerWidget::_confirmCloseTabAndDatabase():');
    if (DbManager.getDbDirtyState(dbPath)) {
      AppState.showAppDialog(title: 'Discard unsaved changes?', message: '''
This database has unsaved changes, you'll lose them if you close it.

If you want to keep your changes, close this popup and press the 'Export' button.
        ''', actions: <Widget>[
        TooltipIconButton(
          text: 'Discard and close',
          iconData: Icons.thumb_down_rounded,
          layout: TooltipIconButtonLayout.iconOnRight,
          onPressedCallback: () {
            _closeTabAndDatabase(dbPath);
            AppState.get('mainNavigatorKey').currentState?.pop();
          },
        ),
      ]);
    } else {
      _closeTabAndDatabase(dbPath);
    }
  }

  /// Close the tab and associated database.
  void _closeTabAndDatabase(String dbPath) {
    DbManager.removeDbByPathKey(dbPath);
  }

  /// Select and display a specific database based on the [index] number of a
  /// tab.
  void _setCurrentSelectedDbPathByIndex(index) {
    String currentSelectedDbPath = '';
    if (DbManager.numberOfDatabases > 0) {
      currentSelectedDbPath = widget.databasesNotifier.keys[index];
    }

    AppState.update('currentSelectedDbPath', currentSelectedDbPath);
  }

  /// Update the tab controller to show the currently-opened databases and
  /// display the currently-selected database.
  void _updateTabController() {
    int totalDatabases = DbManager.numberOfDatabases;
    if (totalDatabases != _tabController.length) {
      int lastTabIndex = (totalDatabases > 0) ? totalDatabases - 1 : 0;

      _tabController.dispose();
      _tabController = TabController(
        length: totalDatabases,
        initialIndex: lastTabIndex,
        animationDuration:
            AppState.getPreference('disableUiAnimations') ? Duration.zero : kTabScrollDuration,
        vsync: this,
      );
      _tabController.addListener(() {
        if (!_tabController.indexIsChanging) {
          _setCurrentSelectedDbPathByIndex(_tabController.index);
        }
      });

      _setCurrentSelectedDbPathByIndex(lastTabIndex);
    }
  }

  void _returnToStartPage() {
    String? currentRoute = ModalRoute.of(context)?.settings.name;
    String startPageRoute = BS.routePrefixInner + BS.routeInnerTabsPage;
    AppState.debug('No databases present');
    AppState.debug('currentRoute: ${currentRoute ?? 'null'}');
    if (mounted && currentRoute != startPageRoute) {
      AppState.get('nestedNavigatorKey').currentState?.pushNamed(BS.routeInnerStartPage);
    } else {
      AppState.debug('START PAGE ALREADY OPENED, NO NAVIGATION NEEDED');
    }
  }
}
