import 'package:sqlite3/sqlite3.dart';

import 'app_state.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'sql_helpers.dart';

/// Low level SQL functions to operate on the special search playlist.
/// This playlist behaves almost the same as a local playlist, only it is filled with matches for
/// specific search terms found in the streams table (which includes history)
///
/// Only [DbManager] should call functions directly from this class. Other classes should call
/// wrapper functions in [DbManager].
class SqlSearchPlaylist {
  /// Update the search playlist with new search results.
  static void updateSearchPlaylist(
    String dbPath,
    String searchTerms, {
    bool includeChannels = true,
    bool includeTitles = true,
  }) {
    AppState.debug('SqlSearchPlaylist::updateSearchPlaylist():');

    List<String> includedColumnList = [];
    if (includeChannels) {
      includedColumnList.add('uploader');
    }
    if (includeTitles) {
      includedColumnList.add('title');
    }
    String columns = includedColumnList.join(', ');

    int? searchPlaylistUid = DbManager.getSearchPlaylistUid(dbPath);
    if (searchPlaylistUid != null && columns.isNotEmpty) {
      AppState.debug('\tsearchPlaylistUid: ${searchPlaylistUid}');

      SqlHelpers.tryAndCatchErrors(() {
        final Database db = sqlite3.open(dbPath);

        /// Create a FTS5 table that indexes the streams table (which holds all streams including
        /// history) Only do this once as the streams table should never change while a database/zip
        /// is open Only the title column is needed for the search TABLE IS TEMPORARY.
        db.execute('''
          CREATE VIRTUAL TABLE temp.search_results_uids_fts USING fts5(
            ${columns}, content='${BS.streamsTableId}', content_rowid='uid'
            , tokenize='porter'
          );
        ''');

        /// Populate the FTS index
        db.execute('''
          INSERT INTO search_results_uids_fts (rowid, ${columns})
            SELECT uid, ${columns}
            FROM ${BS.streamsTableId};
        ''');

        /// Search on the fts table and store results.
        ResultSet searchResultStreamUidResultSet = db.select('''
          SELECT s.uid
          FROM ${BS.streamsTableId} AS s
          JOIN search_results_uids_fts AS r ON s.uid = r.rowid
          WHERE r.search_results_uids_fts MATCH '$searchTerms';
        ''');

        //AppState.debug('\tstream uids: ${searchResultStreamUidResultSet.toString()}');

        // Fill in the join table. Join index increments with each added stream.
        /// Don't insert duplicates where playlistid and streamid are both matched
        int joinIndex = 0;
        for (final searchResultRow in searchResultStreamUidResultSet) {
          int searchResultStreamUid = searchResultRow['uid'];
          ResultSet alreadyExistsRecordSet = db.select('''
          SELECT * FROM ${BS.joinTableId}
          WHERE stream_id = ${searchResultStreamUid}
            AND playlist_id = ${searchPlaylistUid}
        ''');
          if (alreadyExistsRecordSet.isEmpty) {
            db.execute('''
            INSERT INTO ${BS.joinTableId} (
              'playlist_id', 'stream_id', 'join_index'
            )
            VALUES ( ${searchPlaylistUid}, ${searchResultStreamUid}, ${joinIndex} );
          ''');
          }
          joinIndex++;
        }

        db.dispose();
      }, SqlHelpers.getTransactionSchemaDetails(toSchemaVersion: DbManager.getSchemaVersion(dbPath)));
    } else {
      AppState.debug('\tSearch playlist not found');
    }
  }
}
