// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'location.dart';
import 'destination_chooser.dart';
import 'enum.dart';
import 'screen.dart';
import 'table_model.dart';
import 'tooltip_icon_button.dart';

/// Buttons used to perform actions based on currently-selected items.
///
/// Depending on the type of widget the selection can be:
/// - Streams within a playlist
/// - Channels within the 'channel subscriptions' widget
class SelectedItemsButtonsWidget extends StatefulWidget {
  final Location location;
  final TableModel tableNotifier;

  const SelectedItemsButtonsWidget({
    super.key,
    required this.location,
    required this.tableNotifier,
  });

  @override
  State<SelectedItemsButtonsWidget> createState() => _SelectedItemsButtonsWidgetState();
}

class _SelectedItemsButtonsWidgetState extends State<SelectedItemsButtonsWidget> {
  Location? _chosenLocation;

  @override
  Widget build(BuildContext context) {
    ColorScheme colorScheme = Theme.of(context).colorScheme;

    bool somethingIsSelected = widget.tableNotifier.selectedRows.isNotEmpty;

    bool screenIsCompact = Screen.isCompact(context);

    late Color indicatorColor = somethingIsSelected
        ? colorScheme.secondary
        : colorScheme.outlineVariant;
    double indicatorThickness = 1;

    /// COPYING SELECTED ITEMS
    /// Subscriptions/remote playlists require another database to copy to
    /// Local playlists need at least 1 **other** local playlist in any opened database
    /// Search playlists need at least 1 local playlist in any opened database
    bool copyIsEnabled =
        ((widget.location.tableType == TableType.subscriptions ||
                widget.location.tableType == TableType.remotePlaylists) &&
            DbManager.numberOfDatabases > 1) ||
        (widget.location.tableType == TableType.localPlaylist &&
            DbManager.getTotalNumberOfLocalPlaylistsInAllDbs() > 1) ||
        (widget.location.tableType == TableType.search &&
            DbManager.getTotalNumberOfLocalPlaylistsInAllDbs() > 0);

    return Stack(
      children: [
        // Indicator bar (vertical segment]
        PositionedDirectional(
          top: BS.paddingMedium,
          height: indicatorThickness,
          start: 0,
          end: 0,
          child: ColoredBox(color: indicatorColor),
        ),
        // Indicator bar (horizontal segment]
        PositionedDirectional(
          top: 0,
          height: BS.paddingMedium,
          start: 0,
          width: indicatorThickness,
          child: ColoredBox(color: indicatorColor),
        ),
        Container(
          padding: EdgeInsets.only(left: BS.paddingMedium),
          alignment: Alignment.centerLeft,
          child: Wrap(
            spacing: BS.paddingMedium,
            children: [
              TooltipIconButton(
                text: screenIsCompact ? 'Copy' : 'Copy selected to',
                iconData: Icons.copy_rounded,
                layout: TooltipIconButtonLayout.iconOnLeft,
                onPressedCallback: copyIsEnabled && somethingIsSelected
                    ? () {
                        _confirmCopySelectedItems(context: context);
                      }
                    : null,
              ),
              if (widget.location.tableType != TableType.search)
                TooltipIconButton(
                  text: screenIsCompact ? 'Delete' : 'Delete selected',
                  iconData: Icons.delete_rounded,
                  layout: TooltipIconButtonLayout.iconOnLeft,
                  onPressedCallback: somethingIsSelected ? _confirmThenDeleteSelectedItems : null,
                ),
            ],
          ),
        ),
      ],
    );
  }

  /// Start the copy/move process by displaying a dialog with a confirm button,
  /// which will copy or move the items when pressed.
  Future<void> _confirmCopySelectedItems({
    required BuildContext context,
    bool isMoveMode = false,
  }) async {
    return showDialog<void>(
      context: context,
      builder: (BuildContext context) {
        return StatefulBuilder(
          builder: (context, setState) {
            return Center(
              child: SimpleDialog(
                children: [
                  DestinationChooser(
                    locationFrom: widget.location,
                    itemCount: widget.tableNotifier.selectedRows.length,
                    copyOperationType: CopyOperationType.itemsFromTable,
                    //
                    isMoveMode: isMoveMode,
                    moveModeSetter: (bool? value) {
                      setState(() {
                        isMoveMode = value ?? false;
                      });
                    },
                    //
                    chosenLocation: _chosenLocation,
                    chosenLocationSetter: (Location location) {
                      //AppState.debug(
                      //    'SelectedItemsButtonsWidget::chosenLocationSetter:$destination');
                      setState(() {
                        _chosenLocation = location;
                      });
                    },
                    chosenLocationGetter: () {
                      return _chosenLocation;
                    },
                    //
                    copyFunction: _copySelectedItems,
                  ),
                ],
              ),
            );
          },
        );
      },
    );
  }

  /// Update the database to copy the selected items to the new location.
  void _copySelectedItems({
    required Location locationFrom,
    required Location locationTo,
    bool isMoveMode = false,
  }) {
    //AppState.debug(
    //    'Copy ${locationFrom.dbPath}, ${locationFrom.tableId} to ${locationTo.dbPath}, ${locationTo.tableId}');

    switch (widget.location.tableType) {
      case TableType.remotePlaylists:
        DbManager.copyItemsFromRemotePlaylists(
          dbPathFrom: locationFrom.dbPath,
          dbPathTo: locationTo.dbPath,
          remotePlaylistUids: widget.tableNotifier.selectedRows,
          isMoveMode: isMoveMode,
        );
      case TableType.subscriptions:
        DbManager.copyItemsFromChannelSubscriptions(
          dbPathFrom: locationFrom.dbPath,
          dbPathTo: locationTo.dbPath,
          subscriptionUids: widget.tableNotifier.selectedRows,
          isMoveMode: isMoveMode,
        );
      case TableType.localPlaylist:
      case TableType.search:
      default:
        DbManager.copyItemsFromLocalPlaylist(
          dbPathFrom: locationFrom.dbPath,
          dbPathTo: locationTo.dbPath,
          playlistUidFrom: locationFrom.tableId,
          playlistUidTo: locationTo.tableId,
          rowIndices: widget.tableNotifier.selectedRows,
          isMoveMode: isMoveMode,
        );
    }

    if (isMoveMode) {
      widget.tableNotifier.clearRowSelection();
    }
  }

  /// Update the database to delete the selected items.
  void _deleteSelectedItems() {
    AppState.debug('Deleting: ${widget.tableNotifier.selectedRows}');
    switch (widget.location.tableType) {
      case TableType.remotePlaylists:
        DbManager.deleteItemsFromRemotePlaylists(
          dbPath: widget.location.dbPath,
          remotePlaylistUids: widget.tableNotifier.selectedRows,
        );
      case TableType.subscriptions:
        DbManager.deleteItemsFromChannelSubscriptions(
          dbPath: widget.location.dbPath,
          subscriptionUids: widget.tableNotifier.selectedRows,
        );
      case TableType.localPlaylist:
      case TableType.search:
      default:
        DbManager.deleteItemsFromLocalPlaylist(
          dbPath: widget.location.dbPath,
          playlistUid: widget.location.tableId,
          rowIndices: widget.tableNotifier.selectedRows,
        );
    }
    widget.tableNotifier.clearRowSelection();
  }

  /// Start the deletion process by displaying a dialog with a confirm button,
  /// which will delete the playlist when pressed.
  void _confirmThenDeleteSelectedItems() {
    int itemsToDelete = widget.tableNotifier.selectedRows.length;
    AppState.showAppDialog(
      title: 'Delete ${itemsToDelete.toString()} ${itemsToDelete == 1 ? 'item' : 'items'}?',
      actions: <Widget>[
        TooltipIconButton(
          text: 'Delete',
          iconData: Icons.playlist_remove_rounded,
          layout: TooltipIconButtonLayout.iconOnRight,
          onPressedCallback: () {
            _deleteSelectedItems();
            AppState.get('mainNavigatorKey').currentState?.pop();
          },
        ),
      ],
    );
  }
}
