// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'constants.dart';
import 'enum.dart';
import 'preference_item_boolean.dart';
import 'preference_item_radio_group.dart';
import 'screen.dart';

/// Page/panel for displaying and changing app preferences.
class Preferences extends StatefulWidget {
  const Preferences({super.key});

  @override
  State<Preferences> createState() => _PreferencesState();
}

class _PreferencesState extends AppState<Preferences> {
  /// We want to rebuild when 'preferences' changes, so the widgets eg
  /// checkboxes show their correct current state.
  @override
  List<String>? listenForChanges = ['preferences'];

  @override
  Widget build(BuildContext context) {
    bool screenIsCompact = Screen.isCompact(context);
    var layout = screenIsCompact ? BS.preferenceItemLayoutCompact : BS.preferenceItemLayoutStandard;

    ColorScheme colorScheme = Theme.of(context).colorScheme;
    return Material(
      child: SafeArea(
        child: Stack(
          children: [
            SingleChildScrollView(
              child: Column(
                mainAxisAlignment: MainAxisAlignment.start,
                children: [
                  const Text('Preferences', style: TextStyle(fontSize: BS.fontSizePageHeading)),
                  SizedBox(height: layout['between']),

                  ///
                  PreferenceSectionHeading('Theme'),
                  PreferenceItemRadioGroup(
                    id: 'brightnessOverride',
                    screenIsCompact: screenIsCompact,
                    title: 'Theme brightness',
                    options: const {
                      'Dark': 'dark',
                      'Light': 'light',
                      'Follow device settings': 'system',
                    },
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'darkIsTrueBlack',
                    screenIsCompact: screenIsCompact,
                    title: 'True black in dark mode',
                    description: 'Eg for OLED displays',
                  ),

                  ///
                  PreferenceSectionHeading('Extra buttons'),
                  PreferenceItemBoolean(
                    id: 'showJsonImportButton',
                    screenIsCompact: screenIsCompact,
                    title: 'Show JSON import button',
                    description: 'Enables playlists to be imported via JSON',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'showInstantYoutubePlaylistButton',
                    screenIsCompact: screenIsCompact,
                    title: 'Show YouTube playlist button',
                    description: 'Opens playlist in a browser as a YouTube playlist',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'showPlaylistShuffleButton',
                    screenIsCompact: screenIsCompact,
                    title: 'Show playlist shuffle button',
                    description: 'Shuffles the order of a playlist',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'showPlaylistDeduplicateButton',
                    screenIsCompact: screenIsCompact,
                    title: 'Show playlist deduplicate button',
                    description: 'Removes duplicate streams from a playlist',
                  ),
                  PreferenceItemBoolean(
                    id: 'showPlaylistReverseButton',
                    screenIsCompact: screenIsCompact,
                    title: 'Show playlist reverse button',
                    description: 'Reverses the order of a playlist',
                  ),

                  ///
                  PreferenceSectionHeading('User Interface'),
                  PreferenceItemBoolean(
                    id: 'allowRowReordering',
                    screenIsCompact: screenIsCompact,
                    title: 'Allow rows to be dragged',
                    description:
                        'If checked, rows/tracks can be dragged to change the order of the playlist',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemRadioGroup(
                    id: 'dbInfoToShow',
                    screenIsCompact: screenIsCompact,
                    title: 'Database info to show',
                    description:
                        "Info to identify a file and which database version it contains (NewPipe's database has changed over time)",
                    options: {
                      'None': DbInfoToShow.none.name,
                      'File path only': DbInfoToShow.filepathOnly.name,
                      'Extended info': DbInfoToShow.extended.name,
                    },
                  ),
                  PreferenceItemBoolean(
                    id: 'disableUiAnimations',
                    screenIsCompact: screenIsCompact,
                    title: 'Disable animations',
                    description: "Turn off animations such as the glowing 'Export Zip' button",
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'showSmallIconButtonLabels',
                    screenIsCompact: screenIsCompact,
                    title: 'Show labels for small buttons',
                    description: 'Show/hide text labels for small buttons on playlists',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'showFloatingIconButtonLabels',
                    screenIsCompact: screenIsCompact,
                    title: 'Show labels for floating buttons',
                    description: 'Show/hide text labels for large buttons at bottom of screen',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'shortenUrlsForDisplay',
                    screenIsCompact: screenIsCompact,
                    title: 'Shorten URLs for display',
                    description:
                        "Eg changes 'https://youtube.com/watch/abcdefg' to '/watch/abcdefg'",
                  ),
                  PreferenceSectionHeading('History'),
                  PreferenceItemBoolean(
                    id: 'rememberLastOpenedDirectory',
                    screenIsCompact: screenIsCompact,
                    title: 'Remember last opened directory',
                    description: 'When choosing a zip, start in the same directory as last time',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'saveSearchTableToZip',
                    screenIsCompact: screenIsCompact,
                    title: 'Export Search Result Playlist with zips',
                    description:
                        "Search results are stored in a special playlist which can be stored in exported zips (the playlist will be named something like '1f9e039a2b-search'.",
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceSectionHeading('Debugging'),
                  PreferenceItemBoolean(
                    id: 'primaryKeyColumnIsHidden',
                    screenIsCompact: screenIsCompact,
                    title: 'Hide primary key column in tables',
                    description: 'Internal SQLite IDs',
                  ),
                  SizedBox(height: layout['between']),
                  PreferenceItemBoolean(
                    id: 'tableUidIsHidden',
                    screenIsCompact: screenIsCompact,
                    title: 'Hide extra table info',
                    description: 'Internal info about tables',
                  ),
                  SizedBox(
                    height: screenIsCompact
                        ? BS.scrollBottomBufferCompact
                        : BS.scrollBottomBufferStandard,
                  ),
                ],
              ),
            ),
            Positioned(
              top: 0,
              left: 0,
              child: Material(
                type: MaterialType.transparency,
                child: Tooltip(
                  message: 'Back',
                  child: FloatingActionButton(
                    shape: const RoundedRectangleBorder(
                      borderRadius: BorderRadius.only(
                        bottomRight: Radius.circular(BS.cornerButtonRadius),
                      ),
                    ),
                    backgroundColor: colorScheme.surface,
                    foregroundColor: colorScheme.onSurface,
                    heroTag: null,
                    elevation: BS.highButtonElevation,
                    onPressed: () {
                      Navigator.pop(context);
                    },
                    child: const Icon(Icons.arrow_back_rounded),
                  ),
                ),
              ),
            ),
          ],
        ),
      ),
    );
  }
}

class PreferenceSectionHeading extends StatelessWidget {
  const PreferenceSectionHeading(this.text, {super.key});

  final String text;

  @override
  Widget build(BuildContext context) {
    bool screenIsCompact = Screen.isCompact(context);
    var layout = screenIsCompact ? BS.preferenceItemLayoutCompact : BS.preferenceItemLayoutStandard;

    return Column(
      children: [
        SizedBox(height: layout['between']),
        SizedBox(height: layout['between']),
        Text(text, style: TextStyle(fontSize: BS.fontSizeSectionHeading)),
        SizedBox(height: layout['between']),
      ],
    );
  }
}
