// SPDX-License-Identifier: GPL-3.0-only

import 'package:BendyStraw/app_state.dart';
import 'package:BendyStraw/tappable_editable_textfield.dart';
import 'package:flutter/material.dart';

import 'constants.dart';
import 'db_manager.dart';
import 'enum.dart';
import 'location.dart';
import 'playlist_buttons_widget.dart';
import 'playlist_icon_and_item_count.dart';
import 'screen.dart';
import 'search_buttons.dart';

/// Header for a playlist.
///
/// Containing:
/// - The playlist name, in an editable [TextField]
/// - Playlist-specific buttons
class PlaylistHeaderWidget extends StatefulWidget {
  final String name;
  final String dbPath;
  final dynamic tableId;
  final int itemCount;
  final TableType tableType;

  const PlaylistHeaderWidget({
    super.key,
    required this.dbPath,
    required this.tableId,
    required this.name,
    required this.itemCount,
    required this.tableType,
  });

  @override
  State<PlaylistHeaderWidget> createState() => _PlaylistHeaderWidgetState();
}

class _PlaylistHeaderWidgetState extends AppState<PlaylistHeaderWidget> {
  late Color _renamableTextColor;
  late Color _renamableTextBackgroundColor;
  late Color _tapToEditTextColor;
  late Color _tapToEditTextBackgroundColor;

  late bool _isSearchPlaylist;
  late bool _playlistIsEmpty;

  bool _searchIsInProgress = false;

  @override
  late List<String>? listenForChanges = ['preferences'];

  @override
  void didChangeDependencies() {
    super.didChangeDependencies();
    ColorScheme colorScheme = Theme.of(context).colorScheme;
    _renamableTextColor = colorScheme.primary;
    _renamableTextBackgroundColor = colorScheme.onPrimary;
    _tapToEditTextColor = colorScheme.primary;
    _tapToEditTextBackgroundColor = colorScheme.surface;

    _isSearchPlaylist = (widget.tableType == TableType.search);

    /// Liston for search-related changes specific to this database
    if (_isSearchPlaylist) {
      listenForChanges!.addAll([
        '${widget.name}-terms',
        '${widget.name}-includeChannels',
        '${widget.name}-includeTitles'
      ]);
    }
  }

  @override
  Widget build(BuildContext context) {
    _playlistIsEmpty = widget.itemCount == 0;

    return Padding(
      padding: const EdgeInsets.only(right: BS.paddingMedium),
      child: Wrap(
        spacing: BS.paddingMedium,
        crossAxisAlignment: WrapCrossAlignment.start,
        runAlignment: WrapAlignment.spaceBetween,
        alignment: WrapAlignment.start,
        children: [
          Row(
            mainAxisSize: MainAxisSize.min,
            spacing: BS.paddingMedium,
            crossAxisAlignment: CrossAxisAlignment.start,
            children: [
              ///
              ///
              /// Icon and item count.
              SizedBox(
                height: BS.playlistHeaderTopRowHeight,
                child: PlaylistIconAndItemCount(
                  tableType: widget.tableType,
                  label: widget.itemCount.toString(),
                  searchIsInProgress: _searchIsInProgress,
                ),
              ),
              Flexible(
                child: Wrap(
                  spacing: BS.paddingMedium,
                  crossAxisAlignment: WrapCrossAlignment.start,
                  runAlignment: WrapAlignment.spaceBetween,
                  alignment: WrapAlignment.start,
                  children: [
                    ///
                    ///
                    /// Editable playlist name (or search field)
                    ///
                    SizedBox(
                      height: BS.playlistHeaderTopRowHeight,
                      width: BS.renamableTextFieldWidth,
                      child: Align(
                        alignment: Alignment.centerLeft,
                        child: TappableEditableTextField(
                          foregroundColor: _renamableTextColor,
                          backgroundColor: _renamableTextBackgroundColor,
                          tapToEditForegroundColor: _tapToEditTextColor,
                          tapToEditBackgroundColor: _tapToEditTextBackgroundColor,
                          text: _isSearchPlaylist
                              ? AppState.get('${widget.name}-terms')
                              : widget.name,
                          disableTapping: _isSearchPlaylist,
                          taptPromptText:
                              _isSearchPlaylist ? '' : 'Tap again to edit playlist name',
                          onUpdate:
                              _isSearchPlaylist ? _updateSearchPlaylist : _renameLocalPlaylist,
                          moveCaretToEndOfSelection: !_isSearchPlaylist,
                          hintText: _isSearchPlaylist ? 'search here...' : null,
                          searchPlaylistName: _isSearchPlaylist ? widget.name : null,
                        ),
                      ),
                    ),

                    /// Search button and checkboxes if needed
                    if (_isSearchPlaylist)
                      ConstrainedBox(
                        constraints: BoxConstraints(minHeight: BS.playlistHeaderTopRowHeight),
                        child: SearchButtons(
                            searchFunction: _updateSearchPlaylist, playlistName: widget.name),
                      ),
                  ],
                ),
              ),
            ],
          ),

          ///
          ///
          /// Buttons.
          ///
          Padding(
            padding: EdgeInsets.only(
              left: Screen.isCompact(context)
                  ? BS.paddingMedium + BS.paddingExtraSmall
                  : BS.paddingMedium,
            ),
            child: PlaylistButtonsWidget(
              playlistDisplayName: widget.name,
              location: Location(
                dbPath: widget.dbPath,
                tableId: widget.tableId,
                tableType: widget.tableType,
              ),
              disabled: _playlistIsEmpty,
              isSearchPlaylist: _isSearchPlaylist,
            ),
          )
        ],
      ),
    );
  }

  /// Perform a search of streams and fill the search playlist with the results.
  void _updateSearchPlaylist(String terms) {
    if (terms.isNotEmpty) {
      setState(() {
        _searchIsInProgress = true;
      });
      Future.delayed(Duration(milliseconds: 100), () {
        DbManager.updateSearchPlaylist(
          widget.dbPath,
          terms,
          includeTitles: AppState.get('${widget.name}-includeTitles'),
          includeChannels: AppState.get('${widget.name}-includeChannels'),
        );
        setState(() {
          _searchIsInProgress = false;
        });
      });
    }
  }

  /// The name of the playlist has been edited in the [TextField], so update
  /// the database to reflect the change.
  void _renameLocalPlaylist(String newTableName) {
    AppState.debug('PlaylistHeaderWidget::_renameLocalPlaylist(\'$newTableName\')');
    DbManager.renameLocalPlaylist(widget.dbPath, widget.tableId, newTableName);
    ColorScheme colorScheme = Theme.of(context).colorScheme;
    _renamableTextColor = colorScheme.primary;
    _renamableTextBackgroundColor = colorScheme.onPrimary;
    setState(() {});
  }
}
