// SPDX-License-Identifier: GPL-3.0-only

import 'package:flutter/material.dart';

import 'app_state.dart';
import 'block_fade_decorated_container.dart';
import 'constants.dart';
import 'db_manager.dart';
import 'file_manager.dart';
import 'floating_icon_button.dart';
import 'tween_sequences.dart';

/// Show a filechooser for opening/adding/saving NewPipe zips containing database files.
class FileManagerWidget extends StatefulWidget {
  const FileManagerWidget({super.key});

  @override
  State<FileManagerWidget> createState() => _FileManagerWidgetState();
}

class _FileManagerWidgetState extends AppState<FileManagerWidget>
    with SingleTickerProviderStateMixin {
  @override
  List<String>? listenForChanges = [
    'preferences',
    'aDatabaseWasDirtied',
    'currentSelectedDbPath',
    'fileLoadOrSaveIsInProgress',
  ];

  late AnimationController _exportButtonAttractAnimationController;
  late Animation<double> _exportButtonAttractAnimation;
  final _exportButtonAttractAnimationDuration = Duration(milliseconds: 2000);

  bool _wereUnsavedChanges = false;

  @override
  void initState() {
    super.initState();
    _exportButtonAttractAnimationController =
        AnimationController(
          vsync: this, // the SingleTickerProviderStateMixin
          duration: _exportButtonAttractAnimationDuration,
        )..addListener(() {
          setState(() {});
        });

    _exportButtonAttractAnimation = buttonAttractTweenSequence.animate(
      _exportButtonAttractAnimationController,
    );
  }

  @override
  void dispose() {
    _exportButtonAttractAnimationController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    String currentSelectedDbPath = AppState.get('currentSelectedDbPath');
    bool areUnsavedChanges =
        currentSelectedDbPath.isNotEmpty && DbManager.getDbDirtyState(currentSelectedDbPath);
    bool fileLoadOrSaveIsInProgress = AppState.get('fileLoadOrSaveIsInProgress');

    if (areUnsavedChanges != _wereUnsavedChanges) {
      _wereUnsavedChanges = areUnsavedChanges;
      if (areUnsavedChanges) {
        if (AppState.getPreference('disableUiAnimations')) {
          _exportButtonAttractAnimationController.stop();
          _exportButtonAttractAnimationController.value =
              _exportButtonAttractAnimationController.upperBound;
        } else {
          AppState.debug('START ATTRACT ANIMATION');
          _exportButtonAttractAnimationController.reset();
          _exportButtonAttractAnimationController.forward();
        }
      }
    }

    bool showFloatingIconButtonLabels = AppState.getPreference('showFloatingIconButtonLabels');

    ColorScheme currentSelectedDbColorScheme = currentSelectedDbPath.isEmpty
        ? Theme.of(context).colorScheme
        : DbManager.getDbCurrentThemeData(currentSelectedDbPath).colorScheme;
    Color dbBackgroundColor = currentSelectedDbColorScheme.surface;
    Color dbForegroundColor = currentSelectedDbColorScheme.onSurface;
    Color dbBackgroundAttentionColor = currentSelectedDbColorScheme.secondaryContainer;
    Color dbForegroundAttentionColor = currentSelectedDbColorScheme.primary;

    return BlockFadeDecoratedContainer(
      color: dbBackgroundColor,
      child: Row(
        mainAxisAlignment: MainAxisAlignment.spaceBetween,
        children: [
          FloatingIconButton(
            iconData: Icons.add_to_photos_rounded,
            hoverText: 'Add Zip',
            label: showFloatingIconButtonLabels ? 'Add Zip' : null,
            onPressed: fileLoadOrSaveIsInProgress ? null : FileManager.pickZipDbFile,
            backgroundColor: dbBackgroundColor,
            foregroundColor: dbForegroundColor,
            borderRadius: const BorderRadius.only(topRight: Radius.circular(BS.cornerButtonRadius)),
          ),
          if (AppState.getPreference('showJsonImportButton') && currentSelectedDbPath.isNotEmpty)
            FloatingIconButton(
              iconData: Icons.file_open_rounded,
              hoverText: 'Import JSON',
              label: showFloatingIconButtonLabels ? 'Import JSON' : null,
              onPressed: fileLoadOrSaveIsInProgress ? null : FileManager.pickJsonPlaylistFile,
              backgroundColor: dbBackgroundColor,
              foregroundColor: dbForegroundColor,
            ),
          areUnsavedChanges
              ? FloatingIconButton(
                  iconData: Icons.file_download_rounded,
                  hoverText: 'Export Zip',
                  label: showFloatingIconButtonLabels ? 'Export Zip' : null,
                  onPressed: fileLoadOrSaveIsInProgress ? null : FileManager.saveFile,
                  backgroundColor: dbBackgroundAttentionColor.withValues(
                    alpha: _exportButtonAttractAnimation.value,
                  ),
                  foregroundColor: dbForegroundAttentionColor,
                  borderRadius: const BorderRadius.only(
                    topLeft: Radius.circular(BS.cornerButtonRadius),
                  ),
                )
              : FloatingIconButton(
                  iconData: Icons.file_download_rounded,
                  hoverText: '(No unsaved changes)',
                  label: showFloatingIconButtonLabels ? '' : null,
                  onPressed: null,
                  backgroundColor: dbBackgroundColor,
                  foregroundColor: dbForegroundColor,
                  borderRadius: const BorderRadius.only(
                    topLeft: Radius.circular(BS.cornerButtonRadius),
                  ),
                ),
        ],
      ),
    );
  }
}
